package ro.sync.util.editorvars.expander;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ro.sync.util.editorvars.parser.EditorVariablesParser;
import ro.sync.util.editorvars.parser.ParseException;
import ro.sync.util.editorvars.parser.model.AskEditorVariable;
import ro.sync.util.editorvars.parser.model.CompoundLexicalItem;
import ro.sync.util.editorvars.parser.model.EditorVariable;
import ro.sync.util.editorvars.parser.model.LexicalItem;
import ro.sync.util.editorvars.parser.model.PlainText;

/**
 * Parses an expression that contains editor variables and reconstructs it by replacing 
 * the editor variables with their values. 
 */
public class EditorVariableExpander {
  
  /**
   * Logger for logging.
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(EditorVariableExpander.class.getName());
  /**
   * Use this to register an resolver for any editor variable, no matter its name.
   */
  public static final String ANY_VARIABLE_NAME = "";
  
  /**
   * Registered resolvers.
   */
  private final Map<String, EditorVariableResolver> expanders = new HashMap<>();
  /**
   * Error listeners for the parser.
   */
  private final List<ErrorListener> errorListeners = new ArrayList<>();
  
  /**
   * Add an error listener to the list of listeners.
   * 
   * @param errorListener The error listener to add.
   */
  public void addErrorListener(ErrorListener errorListener) {
    errorListeners.add(errorListener);
  }
  
  /**
   * Remove an error listener from the list of listeners.
   * 
   * @param errorListener The error listener to remove.
   */
  public void removeErrorListener(ErrorListener errorListener) {
    errorListeners.remove(errorListener);
  }
  
  /**
   * Register a resolver for a given variable name.
   * 
   * @param editorVariableName Variable name. Use {@link ANY_VARIABLE_NAME} to register a fall back resolver
   *                           that is called when there is no resolver registered for a specific variable name.
   * @param resolver Variable value resolver.
   */
  public void register(String editorVariableName, EditorVariableResolver resolver) {
    expanders.put(editorVariableName, resolver);
  }
  
  /**
   * Serialize lexical items.
   * 
   * @param lexicalItems The list of lexical items to serialize.
   * 
   * @return the serialized string.
   * 
   * @throws ParseException Editor variables with bad syntax.
   * @throws OperationCancelledException The user canceled the operation. 
   */
  protected String expand(List<LexicalItem> lexicalItems) throws ParseException, OperationCancelledException {
    StringBuilder stringBuilder = new StringBuilder();
    for (LexicalItem item : lexicalItems) {
      expandItem(stringBuilder, item);
    }
    return stringBuilder.toString();
  }
  
  /**
   * Expands all the editor variables contained inside the given expression.
   * 
   * @param expression Expression that contains editor variables.
   * 
   * @return The initial expression with editor variables expanded.
   * 
   * @throws ParseException Bad syntax inside the detected editor variables.
   * @throws OperationCancelledException The user canceled the operation.
   */
  public String expand(String expression) throws ParseException, OperationCancelledException {
    EditorVariablesParser editorVarsParser = new EditorVariablesParser(expression);
    editorVarsParser.addErrorListeners(errorListeners);

    List<LexicalItem> items = editorVarsParser.parse();

    return expand(items);
  }
  
  /**
   * Iterates over all editor variables from within the expression.
   *  
   * @param expression A string that potentially contains editor variables.
   * @param visitor    Editor variables visitor.
   *
   * @throws ParseException The expression can't be parsed.
   */
  public void iterateEditorVariables(String expression, Consumer<EditorVariable> visitor) throws ParseException {
    EditorVariablesParser editorVarsParser = new EditorVariablesParser(expression);
    editorVarsParser.addErrorListeners(errorListeners);
    
    List<LexicalItem> items = editorVarsParser.parse();
    visit(items, visitor);
  }

  /**
   * Iterates over all editor variables from within the list.
   * 
   * @param items Lexical items, potential editor variables.
   * @param visitor Editor variables visitor.
   */
  private static void visit(List<LexicalItem> items, Consumer<EditorVariable> visitor) {
    items.stream().forEach((LexicalItem ev) -> {
      if (ev instanceof EditorVariable) {
        EditorVariable editorVariable = (EditorVariable) ev;
        
        visitor.accept(editorVariable);
        visit(editorVariable.getParams(), visitor);
      }
    });
  }

  /**
   * Expand and editor variable.
   * 
   * @param edVar    The editor variable to expand.
   * 
   * @return the expanded editor variable.
   * 
   * @throws ParseException Unable to parse editor variables.
   * @throws OperationCancelledException The user canceled the operation. 
   */
  private String expandEditorVariable(EditorVariable edVar) throws ParseException, OperationCancelledException {
    LOGGER.debug("edVar {}", edVar.getName());

    // Expand parameters if needed.
    List<LexicalItem> params = edVar.getParams();
    
    // Step 1. If we have parameters, expand them.
    if (params != null && !params.isEmpty()) {
      params = expandParameters(params);
      edVar = new EditorVariable(edVar.getName(), params);
    }
    
    LOGGER.debug("Expanded params {}", edVar.getParams());
    
    // Step 2. Expand the editor variable.
    String expanded = null;
    EditorVariableResolver resolver = getExpander(edVar.getName());
    
    LOGGER.debug("{}, resolver: {}", edVar.getName(), resolver);
    
    if (resolver == null) {
      // Unknown variable. Keep it as it is.
      expanded = EditorVariableUtil.serialize(edVar);
    } else {
      expanded = resolver.resolve(edVar);
      
      final String expandedTemp = expanded;
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Expanded {}", URLUtil.filterPasswords(String.valueOf(expandedTemp)));
      }

      if (expanded != null) {
        if (
            // Avoid parsing the returned value if it doesn't, at least, contains a variable start token.
            expanded.contains("${")) {
          // Step3. Maybe the returned value is itself an editor variable that needs expanding.
          EditorVariablesParser parser = new EditorVariablesParser(expanded);
          // Install error listeners on the new parser as well.
          parser.addErrorListeners(errorListeners);
          List<LexicalItem> newItems = parser.parse();
          expanded = expand(newItems);
        }
      } else {
        // NULL value. Do not expand.
        expanded = EditorVariableUtil.serialize(edVar);
      }
    }
    
    return expanded;
  }

  /**
   * Gets an expander registered for the given editor variable.
   * 
   * @param varName Editor variable name.
   * 
   * @return An expander or <code>null</code> if there isn't one mapped 
   *         neither for this variable name, nor for {@link #ANY_VARIABLE_NAME}.
   */
  private EditorVariableResolver getExpander(String varName) {
    EditorVariableResolver editorVariableResolver = expanders.get(varName);
    if (editorVariableResolver == null) {
      // A global expander.
      editorVariableResolver = expanders.get(ANY_VARIABLE_NAME);
    }
    return editorVariableResolver;
  }

  /**
   * Expand parameters of an editor variable.
   * 
   * @param params    The list of parameters.
   * 
   * @return A list of expanded parameters.
   * 
   * @throws ParseException Editor variables with bad syntax.
   * @throws OperationCancelledException The user canceled the operation.
   */
  private List<LexicalItem> expandParameters(List<LexicalItem> params) throws ParseException, OperationCancelledException {
    List<LexicalItem> newParams = new ArrayList<>(params.size());
    for (LexicalItem parameter : params) {
      PlainText plainText = expandLexicalItem(parameter);
      newParams.add(plainText);
    }
    
    return newParams;
  }

  /**
   * Expands the lexical items and creates a plain text over its value.
   * 
   * @param item Item that possibly contains editor variables.
   * 
   * @return A plain text with the expanded value.
   * 
   * @throws ParseException Problems while trying to expand.
   * @throws OperationCancelledException The user canceled the operation. 
   */
  private PlainText expandLexicalItem(LexicalItem item) throws ParseException, OperationCancelledException {
    StringBuilder expandedValue = new StringBuilder();
    expandItem(expandedValue, item);

    // Construct a new parameter from all the expanded parts.
    return new PlainText(expandedValue.toString());
  }

  /**
   * Expands the lexical item to a string value.
   * 
   * @param collector The expanded string value will be put in this collector. 
   * @param lexicalItem Lexical item to expand.
   * 
   * @throws ParseException Problems while expanding.
   * @throws OperationCancelledException The user canceled the operation. 
   */
  private void expandItem(StringBuilder collector, LexicalItem lexicalItem) throws ParseException, OperationCancelledException {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Expand item: {}", URLUtil.filterPasswords(String.valueOf(lexicalItem)));
    }
    if (lexicalItem.getType() == LexicalItem.Type.EDITOR_VARIABLE) {
      EditorVariable edVar = (EditorVariable) lexicalItem;
      // The parameter is another editor variable. EXPAND IT!
      String expanded = expandEditorVariable(edVar);
      collector.append(expanded);
    } else if (lexicalItem.getType() == LexicalItem.Type.ASK_EDITOR_VARIABLE) {
      AskEditorVariable edVar = (AskEditorVariable) lexicalItem;
      // The parameter is another editor variable. EXPAND IT!
      String expanded = expandAskEditorVariable(edVar);
      collector.append(expanded);
    } else if (lexicalItem.getType() == LexicalItem.Type.PLAIN_TEXT) {
      // Keep it as it is.
      collector.append(((PlainText) lexicalItem).getText());
    } else if (lexicalItem.getType() == LexicalItem.Type.COMPOUND_ITEM) {
      List<LexicalItem> parts = ((CompoundLexicalItem) lexicalItem).getLexicalItems();
      for (LexicalItem part : parts) {
        expandItem(collector, part);
      }
    } else {
      LOGGER.error("Unexpected part in parameter {}", lexicalItem.getType());
    }
  }

  /**
   * Checks the lexical item and expands it to {@link PlainText}.
   * 
   * @param lexicalItem The lexical item to check and expand.
   *
   * @return The expanded lexical item.
   *
   * @throws OperationCancelledException The user canceled the operation. 
   * @throws ParseException              Problems while trying to expand.
   */
  private LexicalItem checkAndExpandLexicalItem(LexicalItem lexicalItem) throws ParseException, OperationCancelledException {
    if (lexicalItem != null) {
      lexicalItem = expandLexicalItem(lexicalItem);
    }
    return lexicalItem;
  }

  /**
   * Expand an 'ask' editor variable. 
   * <pre>
   * ${ask('message', combobox, ('real_value1':'rendered_value1';...;'real_valueN':'rendered_valueN'), 'default')}
   * </pre>
   * 
   * @param askEdVar The "ask" editor variable to expand.
   * 
   * @return the expanded editor variable.
   * 
   * @throws ParseException Editor variables with bad syntax.
   * @throws OperationCancelledException The user canceled the operation.
   */
  private String expandAskEditorVariable(AskEditorVariable askEdVar) throws ParseException, OperationCancelledException {
    final AskEditorVariable askEdVarTemp = askEdVar;
    LOGGER.debug("Expand ask: {}", () -> URLUtil.filterPasswords(String.valueOf(askEdVarTemp)));
    
    // Expand parameters if needed.
    LexicalItem message = checkAndExpandLexicalItem(askEdVar.getMessage());
    LexicalItem inputType = checkAndExpandLexicalItem(askEdVar.getInputType());
    
    Iterable<LexicalItem> valueEnumeration = askEdVar.getValueEnumeration();
    
    if (askEdVar.hasPendingValue()) {
      try {
        valueEnumeration = checkAndExpandPendingAskValueEnum(valueEnumeration.iterator().next());
      } catch (ParseException ex) {
        // Unable to expand pending value:label part. Abort and serialize the $ask variable as it is.

        return EditorVariableUtil.serialize(askEdVar);
      }
    }
    
      List<LexicalItem>  expandedVE = new ArrayList<>();
      for (LexicalItem li : valueEnumeration) {
        LexicalItem lexicalItem = checkAndExpandLexicalItem(li);
        expandedVE.add(lexicalItem);
      }
      valueEnumeration = expandedVE;
      
    LexicalItem defaultValue = checkAndExpandLexicalItem(askEdVar.getDefaultValue());
    LexicalItem answerId = checkAndExpandLexicalItem(askEdVar.getAnswerId());


    askEdVar = new AskEditorVariable(message, inputType, valueEnumeration, defaultValue, answerId);

    // Step 2. Expand the editor variable.
    String expanded = null;
    EditorVariableResolver expander = getExpander(askEdVar.getName());
    if (expander == null) {
      expanded = EditorVariableUtil.serialize(askEdVar);
    } else {
      expanded = expander.resolve(askEdVar);
      
      if (expanded != null) {
        // Step3. Maybe the returned value is itself an editor variable that needs expanding.
        EditorVariablesParser parser = new EditorVariablesParser(expanded);
        parser.addErrorListeners(errorListeners);
        List<LexicalItem> newItems = parser.parse();
        expanded = expand(newItems);
      } else {
        // NULL value, do not expand.
        expanded = EditorVariableUtil.serialize(askEdVar);
      }
    }
    
    final String expandedToLog = expanded;
    LOGGER.debug("ASK result: {}", () -> URLUtil.filterPasswords(String.valueOf(expandedToLog)));

    return expanded;
  }

  /**
   * The value enumeration part was specified as a lexical item, most likely as another editor variable.
   * When this happens, we need to:
   * <ol>
   *   <li>Expand/resolve the lexical unit and obtain the string value</li>
   *   <li>Parse the obtained string as a value:label enumeration</li>
   * </ol>
   * <pre>
   *   ${ask('combobox with quotes', combobox, ( ${xpath_eval('real_value1':'rendered_value1')} ))}
   * </pre>
   *
   * @param pending Lexical item to expand.
   * 
   * @return The value enumeration part was specified as a lexical item, most likely as another editor variable.
   * 
   * @throws ParseException              If cannot parse the enumeration part.
   * @throws OperationCancelledException The user canceled the operation. 
   */
  private List<LexicalItem> checkAndExpandPendingAskValueEnum(LexicalItem pending) throws ParseException, OperationCancelledException {
    StringBuilder expandedValue = new StringBuilder();
    expandItem(expandedValue, pending);
    String toParse = expandedValue.toString();
    
    EditorVariablesParser parser = new EditorVariablesParser(toParse);
    
    // Install error listeners on the new parser as well.
    parser.addErrorListeners(errorListeners);
    return parser.parseAskValueEnumerationPart();
  }
}