goog.provide('sync.editors.MarkdownEditingSupport');


sync.editors.MarkdownEditingSupport = class MarkdownEditingSupport extends sync.internal_api.TextEditingSupportImpl {

/**
 * Markdown editing support.
 *
 * @param {sync.Editor} editor The editor.
 * @param {sync.api.Workspace.LoadingOptions} options the editor loading options.
 *
 * @constructor
 */
constructor(editor, options) {
  super(editor, editor.getActionsManager(), options);
  this.editor = editor;
  
  this.errorsModel_ = new sync.model.ValidationErrorsModel();
  this.registerDisposable(this.errorsModel_);
}

/**
 * Register the markdown actions.
 *
 * @param {sync.api.Editor} editor the editor.
 */
registerActions(editor) {
  var actionsManager = editor.getActionsManager();
  actionsManager.registerAction('ReportProblem', sync.actions.ReportProblemAction.getInstance());
  actionsManager.registerAction('Help', new sync.actions.ShowHelp());

  var ih1 = new md.InsertHeadingAction(tr(msgs.MD_HEADING_1), "Heading1_", '#', this.editor);
  actionsManager.registerAction('H1', ih1);

  var ih2 = new md.InsertHeadingAction(tr(msgs.MD_HEADING_2), "Heading2_", '##', this.editor);
  actionsManager.registerAction('H2', ih2);

  var ih3 = new md.InsertHeadingAction(tr(msgs.MD_HEADING_3), "Heading3_", '###', this.editor);
  actionsManager.registerAction('H3', ih3);

  var ihr = new md.InsertHorizontalRuleAction(tr(msgs.MD_HORIZONTAL_RULE), "Line", this.editor);
  actionsManager.registerAction('HR', ihr);

  var mba = new md.MarkdownMarkTextAction(tr(msgs.MD_BOLD), "Bold", '**', this.editor, 'M1 B');
  actionsManager.registerAction('Bold', mba);

  var mia = new md.MarkdownMarkTextAction(tr(msgs.MD_ITALIC), "Italic", '_',  this.editor, 'M1 I');
  actionsManager.registerAction('Italic', mia);

  var msa = new md.MarkdownMarkTextAction(tr(msgs.MD_STRIKETHROUGH), "StrikeThrough", '~~', this.editor);
  actionsManager.registerAction('StrikeThrough', msa);

  var cba = new md.MarkCodeBlockAction(tr(msgs.MD_CODE_BLOCK), "CodeBlock", this.editor);
  actionsManager.registerAction('CodeBlock', cba);

  var qba = new md.MarkQuoteBlockAction(tr(msgs.MD_QUOTE_BLOCK), "Quote", this.editor);
  actionsManager.registerAction('QuoteBlock', qba);

  var iia = new md.InsertUrlAction(tr(msgs.MD_INSERT_IMAGE), "Image", editor, {
	  textLabel: tr(msgs.MD_ALTERNATE_TEXT_LABEL),
	  dialogTitle: tr(msgs.MD_INSERT_IMAGE),
	  prefix: '!'});
  actionsManager.registerAction('InsertImage', iia);

  var ila = new md.InsertUrlAction(tr(msgs.MD_INSERT_LINK), "Link", editor, {
	  textLabel: tr(msgs.MD_TEXT_LABEL),
	  replaceSelection: true,
	  dialogTitle: tr(msgs.MD_INSERT_LINK)
  });
  actionsManager.registerAction('InsertLink', ila);

  var ita = new md.InsertTableAction(tr(msgs.MD_INSERT_TABLE), "Table", this.editor);
  actionsManager.registerAction('InsertTable', ita);

  var iola = new md.InsertListAction(tr(msgs.MD_INSERT_ORDERED_LIST), "OrderedList", "1.", this.editor, 'M2 M1 7');
  actionsManager.registerAction('InsertOrderedList', iola);

  var iula = new md.InsertListAction(tr(msgs.MD_INSERT_UNORDERED_LIST), "UnorderedList", "-", this.editor, 'M2 M1 8');
  actionsManager.registerAction('InsertUnorderedList', iula);

  var itla = new md.InsertListAction(tr(msgs.MD_INSERT_TASK_LIST), "TaskList", "- []", this.editor);
  actionsManager.registerAction('InsertTaskList', itla);

  let actions = [ih1, ih2, ih3, ihr, mba, mia, msa, cba, qba, iia, ila, ita, iola, iula, itla];
  return actions;
}

initActions_ (cmEditor, actions) {
  for(var i=0; i < actions.length; i++) {
    let action = actions[i];
    action.setMirrorEditor(cmEditor);

    // add the shortcut for the command if any
    let cmdShortcut = action.getShortcut();

    if(cmdShortcut) {
      //transforms shortcut from webauthor to CodeMirror format
      cmdShortcut = cmdShortcut.replace('M1 ', goog.userAgent.MAC ? 'Cmd-' : 'Ctrl-');
      cmdShortcut = cmdShortcut.replace('M2 ', 'Shift-');
      // Make sure to call the right actionPerformed.
      cmEditor.getOption("extraKeys")[cmdShortcut] = action.actionPerformed.bind(action);
    }
  }
}

/** @override */
async load(newDoc, docContainer, problemReporter) {
  this.mdActions_ = this.registerActions(this.editor);

  await this.loadMarkdownCodeMirrorStyle_();
  await this.loadMarkdownCodeMirrorMode_();

  goog.dom.removeChildren(docContainer);

  let classValue = this.options.diffUrl || this.options.previewMode ? "markdownpage nopadding" : "markdownpage";
  let pageDiv = goog.dom.createDom("div", {class:classValue, "data-role": "document"});
  docContainer.appendChild(pageDiv);

  await super.load(newDoc, pageDiv, problemReporter);
  this.initCmEditor_();

  return this;
}

loadMarkdownCodeMirrorStyle_() {
  if (window.markdownCodeMirrorStyleLoaded) {
    return Promise.resolve();
  }

  return sync.util.promise.nativePromise(sync.util.dom.loadCSSFile('../plugin-resources/web-author-markdown-editing-support/gen/markdown.css'))
    .then(() => {
      window.markdownCodeMirrorStyleLoaded = true;
    });
}

loadMarkdownCodeMirrorMode_() {
  if (window.codeMirrorModeLoaded) {
    return Promise.resolve();
  }

  return sync.util.promise.nativePromise(sync.util.dom.loadJsFile('../plugin-resources/web-author-markdown-editing-support/gen/codemirror-markdown-mode.js'))
    .then(() => {
      window.codeMirrorModeLoaded = true;
    });
}

initCmEditor_() {
  let cmEditor = this.getCodeMirrorEditor();

  window.installCodeMirrorMarkdownMode();
  cmEditor.setOption("mode", {
    name: "markdown",
    highlightFormatting: true
  });

  if (this.options.previewMode || this.options.diffUrl) {
    // Make sure is read-only on preview.
    // Make read-only on diff by fear of bugs on various connectors.
  } else {
    workspace.getViewManager().installView('markdown-validation', new md.ErrorsView());
  }

  this.initActions_(cmEditor, this.mdActions_)

  // Initialize validation.
  var schematronUrl = this.getSchematronUrl();
  if (schematronUrl && this.options['validate.as.you.type'] !== 'false') {
    this.validator_ = new md.Validation(cmEditor, this.errorsModel_, this.getUrl(), schematronUrl, this.iframe);
  }
}

/**
 * @return {md.Validation} the validator.
 */
getValidator() {
  return this.validator_;
}

/**
 * @return {string} The URL of the Schematron to use for validation.
 */
getSchematronUrl() {
  return this.options.schematronUrl;
}

/** @override */
getFirstToolbar() {
	var builtinToolbar = new sync.view.ExtensionToolbar(this.editor.getActionsManager());
  this.registerDisposable(builtinToolbar);

  builtinToolbar.config = new sync.actions.ActionsConfig({
    toolbars: [{
      type: "list",
      name: "text",
      children: [
    	  {
       	    type: "action",
            id: 'H1'
           },{
         	 type: "action",
             id: 'H2'
           },{
          	 type: "action",
              id: 'H3'
           },{
              type: "action",
              id: 'HR'
           },{
        	  type: "sep"
           },{
            type: "action",
            id: "Bold"
           },{
        	 type: "action",
        	 id: "Italic"
           },{
        	 type: "action",
        	 id: "StrikeThrough"
           },{
        	 type: "sep"
           },{
        	 type: "action",
        	 id: "CodeBlock"
           },{
        	 type:"action",
        	 id: "QuoteBlock"
           },{
        	 type: "sep"
           },{
        	 type: "action",
        	 id: "InsertLink"
           },{
        	 type: "action",
        	 id: "InsertImage"
           },{
        	 type: "sep"
           },{
        	  type: "action",
        	  id : "InsertOrderedList"
           },{
        	  type: "action",
        	  id : "InsertUnorderedList"
           },{
        	  type: "action",
        	  id : "InsertTaskList"
           },{
        	  type: "action",
        	  id: "InsertTable"
           }
       ]
    }]
  });

  return builtinToolbar;
}


/** @override */
getType() {
  return sync.api.Editor.EditorTypes.MARKDOWN;
}

/** @override */
getErrorsModel() {
  return this.errorsModel_;
}

};//end MarkdownEditingSupport class block.
