/**
 * Marks the input as invalid and registers listener to remove the mark on input change.
 *
 * @param input the input to mark as invalid until the first input event on it.
 */
function markInvalid (input) {
  input.classList.add('invalid-input');

  var handleInput = function () {
    input.classList.remove('invalid-input');
    input.removeEventListener('input', handleInput);
  };
  input.addEventListener('input', handleInput);
}

function trimConsecutiveWhitespaces(credential) {
  return credential.replaceAll(/(\s)\s+/g, "$1");
}

/**
 * @return {{password: string, username: string}=} The entered credentials, if valid.
 */
function getAndValidateCredentials () {
  var userInput = document.querySelector('input[name="username"]');
  var passInput = document.querySelector('input[name="password"]');
  var passConfirmInput = document.querySelector('input[name="confirm-password"]');
  var adminSetupPasswordInput = document.querySelector('input[name="adminSetupPassword"]');

  var username = trimConsecutiveWhitespaces(userInput.value);
  var pass = trimConsecutiveWhitespaces(passInput.value);
  var passConfirm = trimConsecutiveWhitespaces(passConfirmInput.value);
  var adminSetupPassword = trimConsecutiveWhitespaces(adminSetupPasswordInput.value);

  if(!username) {
    this.showInvalidCredentialMessage("The username must not be empty.");
    markInvalid(userInput);
  } else if(pass.length < 12) {
    this.showInvalidCredentialMessage("The password must have at least 12 characters.");
  } else if(pass.length > 128) {
    this.showInvalidCredentialMessage("The password must have at most 128 characters.");
  } else if (!passConfirm) {
    markInvalid(passConfirmInput);
  } else if (pass !== passConfirm) {
    passConfirmInput.value = "";
    markInvalid(passConfirmInput);
  } else if (!adminSetupPassword) {
    markInvalid(adminSetupPasswordInput);
    this.showInvalidCredentialMessage("You must enter the Admin Setup Password from the server's logs.");
  } else {
    return {username: username, password: pass, setupPassword: adminSetupPassword};
  }

  return null;
}

function showInvalidCredentialMessage (message) {
  let invalidCredentialMessage = document.querySelector('form .invalid-credential-message');
  invalidCredentialMessage.classList.add('visible');
  invalidCredentialMessage.textContent = message;
  document.querySelector('.btn-lg').blur();

  let passInput = document.querySelector('input[name="password"]');
  passInput.value = '';
  let passConfirmInput = document.querySelector('input[name="confirm-password"]');
  passConfirmInput.value = '';
  passInput.addEventListener('input', function() {
    invalidCredentialMessage.classList.remove('visible');
  });
}

/**
 * Logs in using the provided credentials.
 * @param username the username to use.
 * @param password the password to use.
 */
function login (username, password, setupPassword) {
  var http = new XMLHttpRequest();
  var url = "../admin-login";
  http.open("POST", url, true);
  //Send the proper header information along with the request
  http.setRequestHeader("Content-type", "application/x-www-form-urlencoded");
  http.setRequestHeader('X-Requested-With', 'WA');
  http.onreadystatechange = function () {
    if (http.readyState == 4 && http.status == 200) {
      window.location.reload();
    } else if (http.readyState == 4) {
      window.location.reload();
    }
  };
  var params = 'username=' + encodeURIComponent(username) +
      '&password=' + encodeURIComponent(password) + 
      "&setupPassword=" + encodeURIComponent(setupPassword);
  http.send(params);
}

// handle button click
document.querySelector('.btn-lg').addEventListener('click', function () {
  var validCredentials = getAndValidateCredentials();
  if (validCredentials) {
    var http = new XMLHttpRequest();
    var url = "../admin-anon/configure-user";
    http.open("POST", url, true);

    http.setRequestHeader("Content-type", "application/x-www-form-urlencoded");
    http.setRequestHeader('X-Requested-With', 'WA');

    http.onreadystatechange = function () {
      if (http.readyState == 4 && http.status == 200) {
        window.location.reload();
      } else if (http.readyState == 4) {
        var msgHolder = document.querySelector('.msg_holder');
        if (msgHolder) {
          while (msgHolder.firstChild) {
            msgHolder.removeChild(ff.firstChild);
          }
          var message = document.createElement('div');
          message.className  = 'message';
          message.textContent = tr(msgs.FAILED_TO_CREATE_ADMIN_USER_,
              {
                '$APP_NAME': tr(msgs.WEB_AUTHOR_NAME_),
                '$SUPPORT_EMAIL': 'support@oxygenxml.com'
              }
          );

          msgHolder.appendChild(message);
        }
      }
    };
    var params = 'username=' + encodeURIComponent(validCredentials.username)
        + '&password=' + encodeURIComponent(validCredentials.password)
        + '&setupPassword=' + encodeURIComponent(validCredentials.setupPassword)
    http.send(params);
  }
});