

// handle setting the server-url input
(() => {
  
  /**
   * The server is bundled with WA pr CF and does not present this option. 
   */
  if(! document.getElementById('publicUrlOption')) {
    return;
  }
  
  
  
  
  (function registerServerUrlInputChangesHandlers() {
    let handleInputValueTimeout = null;

    document.querySelector('.server-url-input-section-input').addEventListener('input', function(e) {
      clearTimeout(handleInputValueTimeout);
      handleInputValueTimeout = setTimeout(function() {
        let value = e.target.value || '';

        setNewServerUrl(value);
      }, 500);
    });
  })();
  
  

  /**
   * Handle Activation Signature changes
   */
  (() => {
    const publicUrlOption = document.getElementById('publicUrlOption');
    const publicUrlInput = document.querySelector('.server-url-input-section-input');
    
    // initial enabled/disable state of the server Url input. 
    publicUrlInput.disabled = ! publicUrlOption.checked;

    const onUseServerUrlSelect = (e) => {
      let checked = e.target.checked;
      
      publicUrlInput.disabled = ! checked;
      
      if (checked) {
        let serverUrlValue = publicUrlInput.value;

        if (serverUrlValue) {
          setNewServerUrl(serverUrlValue);
        } else {
          // If the URL used to access the LS is valid we suggest it for use
          suggestTheCurrentUrlIfValid(publicUrlInput);

          // User need to provide a valid serverURL
          disableRegisterActivateButton();
        }
      } else {
        // the server URL is no longer used in activation.
        hideInvalidUrlErrorMessage();

        enableRegisterActivateButton();

        let serverUrlValue = publicUrlInput.value;
        if(serverUrlValue) {
          setNewServerUrl(null);
        }
      }
    };

    publicUrlOption.addEventListener('change', onUseServerUrlSelect);
  })();

  
  /**
   * Functions library.
   */
  
  
  /**
   * Set the new serverUrlValue.
   * 
   * @param newValue the new server URL value.
   */
  function setNewServerUrl(newValue) {
    // disable form until the URL is correctly set server-side.
    disableRegisterActivateButton();

    if (!isValidServerURL(newValue)) {
      // the user provided an invalid URL.
      showInvalidUrlErrorMessage();

      return;
    }
    
    hideInvalidUrlErrorMessage();

    fetch('server-url', {
      method: 'POST',
      credentials: 'same-origin',
      body: newValue
    })
      .catch(error => {
        console.error('Error:', error);
      })
      .then(response => {
        if (response.ok) {
          hideActivationSignatureUpdateError();

          response.text().then(setNewServerSignatureInUI);

          // enable form on succeess
          if (newValue) {
            enableRegisterActivateButton();
          } else {
            if (isUseServerUrlSelected()) {
              // The Use server URL option is selected but no Server URL was provided
              disableRegisterActivateButton();

              // show Invalid URL error.
              showInvalidUrlErrorMessage();
            } else {
              enableRegisterActivateButton();
            }
          }
        } else {
          // show error on failure
          showActivationSignatureUpdateError();
        }
      });
  }

  

  function isValidServerURL(url) {
    // no server URL is used in activation.
    if(!url) {
      return true;
    }
    
    try {
      const parsedUrl = new URL(url);
      const localDomains = ['localhost', '127.0.0.1'];

      if (localDomains.includes(parsedUrl.hostname)) {
        return false;
      }

      // Ensure the path ends with '/license-servlet'
      if (!parsedUrl.pathname.endsWith('/license-servlet')) {
        return false;
      }

      return true;
    } catch (e) {
      return false;
    }
  }

  
  function isUseServerUrlSelected() {
    return document.getElementById('publicUrlOption').checked;
  }
  
  
  /**
   * Disable the license Register / Activate button
   */
  function disableRegisterActivateButton() {
    document.getElementById('noLicenseFormSubmitButton').disabled = true;
    document.getElementById('noLicenseForm').disabled = true;
  }
  
  /**
   * Enable the license Register / Activate button
   */
  function enableRegisterActivateButton() {
    document.getElementById('noLicenseFormSubmitButton').disabled = false;
    document.getElementById('noLicenseForm').disabled = false;
  }
  
  
  function setNewServerSignatureInUI(newServerSignature) {
    const serverSignatureElement = document.getElementById('server-signature');
    serverSignatureElement.textContent = newServerSignature;
  }

  
  function hideInvalidUrlErrorMessage() {
    document.querySelector('.server-url-input-error').style.display = 'none';
  }

  
  function showInvalidUrlErrorMessage() {
    document.querySelector('.server-url-input-error').style.display = 'block';
  };
  
  /**
   * If the current URL used to access the LS is valid we suggest it for use
   * 
   * @param publicUrlInput the server Url input.
   */
  function suggestTheCurrentUrlIfValid(publicUrlInput) {
    let currentUrlLicenseRequestEndpoint = getLicenseRequestEndpoint();

    if (isValidServerURL(currentUrlLicenseRequestEndpoint)) {
      publicUrlInput.value = currentUrlLicenseRequestEndpoint;
      publicUrlInput.select();
      publicUrlInput.focus();
     
      // set the Url server-side also. 
      setNewServerUrl(currentUrlLicenseRequestEndpoint);
    }
  }

  
  function getLicenseRequestEndpoint() {
    let currentUrl = window.location.href;
    const url = new URL(currentUrl);
    if (url.pathname.endsWith('/index.jsp')) {
      url.pathname = url.pathname.replace('/index.jsp', '/license-servlet');
    } else {
      url.pathname = url.pathname.replace(/\/$/, '') + '/license-servlet';
    }
    return url.toString();
  }
  

  function showActivationSignatureUpdateError() {
    document.querySelector('.activation-signature-error-message').style.display = 'block';
  }


  function hideActivationSignatureUpdateError() {
    document.querySelector('.activation-signature-error-message').style.display = 'none';
  }
})();
