<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    xmlns:map="http://www.w3.org/2005/xpath-functions/map"
    xmlns:f="http://www.oxygenxml.com/xsl/functions" exclude-result-prefixes="xs f map"
    version="2.0">
    
    <xsl:variable name="restMethods" 
        select="('get', 'put', 'post', 'delete', 'options', 'head', 'patch', 'trace')"/>
   
    <!-- The outputclass value set on elements that can have markdown content. -->
    <xsl:param name="markdownContentInBodydivOutputClass" select="'oxy-inBodydiv-markdownConvert'" as="xs:string"/>
    
    <!-- The outputclass value set on elements that can have markdown content. -->
    <xsl:param name="markdownContentOutputClass" select="'oxy-markdownConvert'" as="xs:string"/>
    
    <!-- The outputclass value set on elements that have to be converted to original format. -->
    <xsl:param name="toOriginalFormatOutputClass" select="'oxy-originalFormatConvert'" as="xs:string"/>
    
    <!-- The outputclass value set on elements that can have content that need to be unescaped. -->
    <xsl:param name="toUnescapeOutputClass" select="'oxy-toUnescape'" as="xs:string"/>
    
    <!-- The outputclass value set on elements that can have ID attributes that need to be unescaped. -->
    <xsl:param name="toUnescapeIdOutputClass" select="'oxy-toUnescapeId'" as="xs:string"/>
    
    <!-- The outputclass value set on external documents that have to be processed. -->
    <xsl:param name="externalOpenApiOutputClass" select="'oxy-external-openapi'" as="xs:string"/>

    <xsl:include href="handleIntro.xsl"/>
    <xsl:include href="handleComponents.xsl"/>
    
    <xsl:import href="openAPI_v2.0_to_v3.0.xsl"/>

    <xsl:template match="/">
        <xsl:choose>
            <xsl:when test="/*[swagger = '2.0']">
                <xsl:variable name="v3Content">
                    <xsl:apply-templates mode="convertV2"/>
                </xsl:variable>
                
               <!-- 
                <xsl:result-document href="output-v3Content.xml">
                    <xsl:copy-of select="$v3Content"/>
                </xsl:result-document>
               -->
                
                <xsl:apply-templates select="$v3Content/*" mode="v3"/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:apply-templates select="." mode="v3"/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- The entry point for version 3.0 -->
    <xsl:template match="JSON" mode="v3">
        <dita>
            <xsl:call-template name="intro">
                <xsl:with-param name="info" select="info"/>
                <xsl:with-param name="jsonSchemaDialect" select="jsonSchemaDialect"/>
                <xsl:with-param name="servers" select="servers"/>
                <xsl:with-param name="security" select="security"/>
                <xsl:with-param name="externalDocs" select="externalDocs"/>
            </xsl:call-template>
            
            <xsl:if test="paths">
                <xsl:call-template name="paths">
                    <xsl:with-param name="paths" select="paths"/>
                </xsl:call-template>
            </xsl:if>
            
            <xsl:if test="webhooks">
                <xsl:call-template name="webhooks">
                    <xsl:with-param name="webhooks" select="webhooks"/>
                </xsl:call-template>
            </xsl:if>
            
            <xsl:if test="components">
                <xsl:call-template name="components">
                    <xsl:with-param name="componentsNode" select="components"></xsl:with-param>
                </xsl:call-template>
            </xsl:if>
            
            <xsl:call-template name="extensionsHandlerAsTopic">
                <xsl:with-param name="element" select="."/>
            </xsl:call-template>
        </dita>
    </xsl:template>
    
    <!-- Generate DITA for OpenAPI 3.1 webhooks. -->
    <xsl:template name="webhooks">
        <xsl:param name="webhooks" required="no"/>
        <topic id="webhooks" outputclass="openapi-webhooks">
            <title>Webhooks</title>
            <body>
                <xsl:if test="count($webhooks/*/_X24_ref) > 0">
                    <!-- we have references to path items -->
                    <xsl:call-template name="pathItemReferences">
                        <xsl:with-param name="pathItemsList" select="$webhooks/*"/>
                    </xsl:call-template>
                </xsl:if>
            </body>
            <xsl:for-each select="$webhooks/*">
                <xsl:if test="not(./_X24_ref)">
                    <xsl:for-each select="./*[local-name() = $restMethods]">
                        <xsl:call-template name="operation">
                            <xsl:with-param name="operationElement" select="."/>
                            <xsl:with-param name="parentPathElement" select="./parent::*"/>
                            <xsl:with-param name="isWebhookOperation" select="true()"/>
                            <xsl:with-param name="pathValue" select="local-name(./parent::*)"/>
                        </xsl:call-template>
                    </xsl:for-each> 
                </xsl:if>
            </xsl:for-each> 
        </topic>
    </xsl:template>
    
    <!-- A map with Open API Tag elements -->
    <xsl:variable name="tagsMap" as="map(xs:string, element())">
        <xsl:map>
            <xsl:for-each select="JSON/tags">
                <xsl:variable name="name">
                    <xsl:value-of select="./name"/>
                </xsl:variable>
                <xsl:map-entry key="xs:string($name)" select="."></xsl:map-entry>
            </xsl:for-each>
        </xsl:map>
    </xsl:variable>
    
    <!-- Generate DITA for OpenAPI Path objects. -->
    <xsl:template name="paths">
        <xsl:param name="paths" required="no"/>
        
        <!-- Operations with tags -->
        <xsl:for-each-group select="$paths/*/*[local-name() = $restMethods]" 
            group-by="./tags[not(array) and text()], ./tags[array]/array[text()]" 
            composite="no">
            <topic outputclass="openapi-tag">
                <xsl:attribute name="id"><xsl:value-of select="f:correctId(current-grouping-key())"/></xsl:attribute>
                <xsl:attribute name="outputclass" select="concat('openapi-tag ', $toUnescapeIdOutputClass)"/>
                <title outputclass="openapi-tag-title">
                    <xsl:value-of select="concat('API methods: ', current-grouping-key())"/>
                </title> 
                <body>
                    <xsl:if test="map:contains($tagsMap, current-grouping-key())">
                        <xsl:variable name="tagInfoElement" select="map:get($tagsMap, current-grouping-key())"/>
                        <xsl:if test="$tagInfoElement/description">
                            <bodydiv>
                                <xsl:attribute name="outputclass" select="concat('openapi-tag-description ', $markdownContentInBodydivOutputClass)"/>
                                <xsl:value-of select="$tagInfoElement/description"/>
                            </bodydiv>
                        </xsl:if>
                        <xsl:if test="$tagInfoElement/externalDocs">
                            <xsl:call-template name="externalDocumentation">
                                <xsl:with-param name="externalDocsElement" select="$tagInfoElement/externalDocs"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:call-template name="extensionsHandlerAsList">
                            <xsl:with-param name="element" select="$tagInfoElement"/>
                        </xsl:call-template>
                    </xsl:if>
                </body>
                <xsl:for-each select="current-group()">
                    <!-- An operation can have multiple tags. We append tag in ID for creating unique IDs -->
                    <xsl:variable name="idPrefix">
                        <xsl:choose>
                            <xsl:when test="(count(./tags[not(array) and text()]) + count(./tags[array]/array[text()])) > 1">
                                <xsl:value-of select="current-grouping-key()"/>
                            </xsl:when>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:call-template name="operation">
                        <xsl:with-param name="operationElement" select="."/>
                        <xsl:with-param name="parentPathElement" select="./parent::*"/>
                        <xsl:with-param name="isWebhookOperation" select="false()"/>
                        <xsl:with-param name="pathValue" select="local-name(./parent::*)"/>
                        <xsl:with-param name="idPrefix" select="$idPrefix"/>
                    </xsl:call-template>
                </xsl:for-each>  
            </topic>
        </xsl:for-each-group>
        
        <!-- Operations without a tag -->
        <xsl:if test="count($paths/*/*[local-name() = $restMethods and not(tags)]) > 0">
            <topic outputclass="openapi-tag" id="defaultTag">
                <title outputclass="openapi-tag-title">
                    <xsl:value-of select="'API methods: default'"/>
                </title> 
                <body>
                </body>
                <xsl:for-each select="$paths/*/*[local-name() = $restMethods and not(tags)]">
                    <xsl:call-template name="operation">
                        <xsl:with-param name="operationElement" select="."/>
                        <xsl:with-param name="parentPathElement" select="./parent::*"/>
                        <xsl:with-param name="isWebhookOperation" select="false()"/>
                        <xsl:with-param name="pathValue" select="local-name(./parent::*)"/>
                    </xsl:call-template>
                </xsl:for-each>
            </topic>
        </xsl:if>
        
        <!-- Paths with reference -->
        <xsl:if test="count($paths/*/_X24_ref) > 0">
            <topic outputclass="openapi-allPaths" id="paths_references">
                <title>Paths references</title>
                <body>
                    <!-- we have references to path items -->
                    <xsl:call-template name="pathItemReferences">
                        <xsl:with-param name="pathItemsList" select="$paths/*"/>
                    </xsl:call-template>
                </body>
            </topic>
        </xsl:if>
        
        <xsl:call-template name="extensionsHandlerAsTopic">
            <xsl:with-param name="element" select="$paths"/>
        </xsl:call-template>
    </xsl:template>

    <!-- Generate DITA for PathItem references. -->
   <xsl:template name="pathItemReferences">
       <xsl:param name="pathItemsList"/>
       <dl outputclass="openapi-pathItems-refs">
           <xsl:for-each select="$pathItemsList/_X24_ref">
               <dlentry>
                   <dt>
                       <xsl:attribute name="outputclass" select="concat('openapi-pathItem-ref ', $toUnescapeOutputClass)"/>
                       <xsl:value-of select="local-name(./parent::*)"/>
                   </dt>
                   <dd>
                       <xsl:call-template name="reference">
                           <xsl:with-param name="referenceNode" select="."/>
                       </xsl:call-template>
                   </dd>
               </dlentry>
           </xsl:for-each> 
       </dl>
   </xsl:template>

    <!-- Generate DITA for an Operation Object. -->
    <xsl:template name="operation">
        <xsl:param name="operationElement" as="node()"/>
        <xsl:param name="parentPathElement" as="node()"/>
        <xsl:param name="isWebhookOperation" as="xs:boolean"/>
        <xsl:param name="pathValue" required="no"/>
        <xsl:param name="idPrefix" required="no"/>
        
        <xsl:variable name="method" select="local-name($operationElement)"/>
        <xsl:variable name="topicId" select="f:getOperationId($operationElement, local-name($parentPathElement), $idPrefix)"/>
        
        <topic>
            <xsl:attribute name="id">
                <xsl:value-of select="$topicId"/>
            </xsl:attribute>
            <xsl:attribute name="outputclass" select="concat('openapi-operation ', $toUnescapeIdOutputClass)"/>
            <title outputclass="openapi-operation-title">
                <xsl:attribute name="outputclass" select="concat('openapi-operation-title openapi-method-', $method)"></xsl:attribute>
                <ph>
                    <xsl:attribute name="id" select="concat($topicId, '-operationName')"/>
                    <ph>
                        <xsl:attribute name="outputclass" select="concat('openapi-operation-method-', $method)"></xsl:attribute>
                        <xsl:value-of select="upper-case($method)"/>
                    </ph>
                    
                    <xsl:variable name="pathOutputClassVal">
                        <xsl:choose>
                            <xsl:when test="$isWebhookOperation">
                                <xsl:value-of select="'openapi-operation-webhook '"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:value-of select="'openapi-operation-path '"/>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    <xsl:if test="$pathValue">
                        <xsl:value-of select="' '"/>
                        <ph><xsl:attribute name="outputclass" select="concat($pathOutputClassVal, $toUnescapeOutputClass)"/><xsl:value-of select="$pathValue"/></ph>
                    </xsl:if>
                </ph>
            </title>
            <xsl:if test="f:checkOperationHasSummary($operationElement)">
                <shortdesc>
                    <ph>
                        <xsl:attribute name="id" select="concat($topicId, '-shortdesc-ph')"/>
                        <xsl:choose>
                            <xsl:when test="$operationElement/summary">
                                <xsl:value-of select="$operationElement/summary"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:value-of select="$parentPathElement/summary"/>
                            </xsl:otherwise> 
                        </xsl:choose>
                    </ph>
                </shortdesc>
            </xsl:if>
            <body>
                <xsl:if test="$operationElement/description or $parentPathElement/description">
                <bodydiv>
                    <xsl:attribute name="outputclass" select="concat('openapi-operation-description ', $markdownContentInBodydivOutputClass)"/>
                    <xsl:attribute name="id" select="concat($topicId, '-description')"/>
                    <xsl:choose>
                        <xsl:when test="$operationElement/description">
                            <xsl:value-of select="$operationElement/description"/>
                        </xsl:when>
                        <xsl:when test="$parentPathElement/description">
                            <xsl:value-of select="$parentPathElement/description"/>
                        </xsl:when>
                    </xsl:choose>
                </bodydiv>
                </xsl:if>
                <xsl:if test="$operationElement/externalDocs">
                    <xsl:call-template name="externalDocumentation">
                        <xsl:with-param name="externalDocsElement" select="$operationElement/externalDocs"/>
                    </xsl:call-template>
                </xsl:if>
                <xsl:call-template name="params">
                    <xsl:with-param name="operationParams" select="$operationElement/parameters"/>
                    <xsl:with-param name="pathParams" select="$parentPathElement/parameters"/>
                </xsl:call-template>
                <xsl:call-template name="requestBodySection">
                    <xsl:with-param name="requestBody" select="$operationElement/requestBody"/>
                </xsl:call-template>
                <xsl:call-template name="responses">
                    <xsl:with-param name="responses" select="$operationElement/responses"/>
                </xsl:call-template>
                
                <xsl:if test="$operationElement/servers">
                    <xsl:call-template name="servers">
                        <xsl:with-param name="servers" select="$operationElement/servers"/>
                    </xsl:call-template>
                </xsl:if>
                
                <xsl:if test="$operationElement/security">
                    <xsl:call-template name="securityRequirementObject">
                        <xsl:with-param name="securityReqNodes" select="$operationElement/security/*"/>
                    </xsl:call-template>
                </xsl:if>
                
                <xsl:call-template name="extensionsHandlerAsList">
                    <xsl:with-param name="element" select="$operationElement"/>
                </xsl:call-template>
            </body>
            <xsl:if test="$operationElement/callbacks">
                <topic id="callbacks" outputclass="openapi-callbacks">
                    <title>Callbacks</title>
                    <body>
                        <xsl:call-template name="extensionsHandlerAsList">
                            <xsl:with-param name="element" select="$operationElement/callbacks"/>
                        </xsl:call-template>
                    </body>
                    <xsl:call-template name="callbacksMapItems">
                        <xsl:with-param name="calbackItems" select="$operationElement/callbacks/*"/>
                    </xsl:call-template>
                </topic>
            </xsl:if>
        </topic>
    </xsl:template>
    
    <!-- Check if the given operation has a summary -->
    <xsl:function name="f:checkOperationHasSummary" as="xs:boolean">
        <xsl:param name="operationElement"/>
        <xsl:sequence select="$operationElement/summary or $operationElement/parent::*/summary"/>
    </xsl:function>
    
    <!-- Generate DITA for OpenAPI Callback objects map. -->
    <xsl:template name="callbacksMapItems">
        <xsl:param name="calbackItems"/>
        <xsl:for-each select="$calbackItems">
            <topic>
                <xsl:attribute name="id">
                    <xsl:value-of select="f:correctId(local-name())"/>
                </xsl:attribute>
                <xsl:attribute name="outputclass" select="concat('openapi-callback ', $toUnescapeIdOutputClass)"/>
                <title>
                    <xsl:attribute name="outputclass" select="$toUnescapeOutputClass"/>
                    <xsl:value-of select="local-name()"/>
                </title>
                <body>
                    <xsl:if test="./_X24_ref">
                        <div outputclass="openapi-callback-ref">
                            <!-- It's a reference to a callback -->
                            <xsl:call-template name="reference">
                                <xsl:with-param name="referenceNode" select="./_X24_ref"></xsl:with-param>
                            </xsl:call-template>
                        </div>
                    </xsl:if>
                </body>
                <xsl:if test="not(./_X24_ref)">
                    <xsl:for-each select="./*/*[local-name() = $restMethods]">
                        <xsl:call-template name="operation">
                            <xsl:with-param name="operationElement" select="."/>
                            <xsl:with-param name="parentPathElement" select="./parent::*"/>
                            <xsl:with-param name="isWebhookOperation" select="false()"/>
                            <xsl:with-param name="pathValue" select="local-name(./parent::*)"/>
                        </xsl:call-template>
                    </xsl:for-each> 
                </xsl:if>
            </topic>
        </xsl:for-each>
    </xsl:template>
</xsl:stylesheet>
