package ro.sync.util.editorvars.parser.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * An editor variable.
 */
public class EditorVariable extends LexicalItem {
  /**
   * The name of the editor variable (ask, answer, cfn, pd, etc).
   */
  private final String name;
  
  /**
   * The variable's parameters.
   */
  private List<LexicalItem> params;

  /**
   * Constructor.
   * 
   * @param name    Variable name (ask, answer, cfn, pd, etc).
   * @param params  The variable's parameters.
   */
  public EditorVariable(String name, List<LexicalItem> params) {
    super(Type.EDITOR_VARIABLE);
    this.name = name;
    this.params = params == null ? Collections.emptyList() : params;
  }
  
  /**
   * Constructor.
   * 
   * @param name Variable name.
   * @param type Variable type.
   */
  protected EditorVariable(String name, Type type) {
    super(type);
    this.name = name;
  }
  
  /**
   * @return Variable name.
   */
  public String getName() {
    return name;
  }

  /**
   * @return The parameters or an empty list. Never <code>null</code>.
   */
  public List<LexicalItem> getParams() {
    return new ArrayList<>(params);
  }

  /**
   * @return A string representation of the parameters.
   */
  private String paramsToString() {
    StringBuilder stringBuilder = new StringBuilder();
    stringBuilder.append('(');
    for (LexicalItem param : params) {
      stringBuilder.append("\n    ").append(param);
    }
    stringBuilder.append("\n  )");
    return stringBuilder.toString();
  }
  
  /**
   * @return A string representation of the editor variable.
   */
  @Override
  public String toString() {
    return
        "'${" + name + (params == null || params.isEmpty() ? "" : (paramsToString() + "\n")) + "}'"
        + ", type: " + getType();
  }
  
  @Override
  public String getStringValue() {
    return getName();
  }
}
