package ro.sync.util.editorvars.parser.model;

import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * An $ask editor variable. 
 * <pre>${ask('message', combobox, ('real_value1':'rendered_value1';...;'real_valueN':'rendered_valueN'), 'default')}</pre>
 * 
 * @author alex_jitianu
 * @author sorin_carbunaru
 */
public class AskEditorVariable extends EditorVariable {
  /**
   * Editor variable name.
   */
  private static final String ASK_NAME = "ask";

  /**
   * The message to present to the user.
   */
  private final LexicalItem message;
  /**
   * What type of input we request from the user: combobox, radio, etc.
   */
  private final LexicalItem inputType;
  /**
   * An enumeration of pairs (value, label)
   * 
   * Can be just one lexical item in which case, we need to:
   * <ol>
   * <li>Expand/resolve the lexical unit and obtain the string value</li>
   * <li>Serialize the entire $ask editor variable to its string representation.</li>
   * <li>Parse the $ask again so that we now have proper value:label pairs.</li> 
   * </ol>
   * <pre>
   * ${ask('combobox with quotes', combobox, ( ${xpath_eval('real_value1':'rendered_value1')} ))}
   * </pre>
   */
  private Iterable<LexicalItem> valueEnumeration;
  /**
   * Default value.
   */
  private final LexicalItem defaultValue;
  /**
   * The value obtained from the user will be linked to this ID. 
   * Later on, this ID can be used with the $answer editor variable.
   */
  private final LexicalItem answerId;

  /**
   * Constructor.
   * 
   * @param message The message to present to the user.
   * @param inputType What type of input we request from the user: combobox, radio, etc.
   * @param valueEnumeration value=>label mappings.
   * @param defaultValue Default value.
   * @param answerId The value obtained from the user will be linked to this ID. 
   *                 Later on, this ID can be used with the $answer editor variable.
   */
  public AskEditorVariable(
      LexicalItem message,
      LexicalItem inputType,
      Iterable<LexicalItem> valueEnumeration, 
      LexicalItem defaultValue, 
      LexicalItem answerId) {
    super(ASK_NAME, Type.ASK_EDITOR_VARIABLE);
    
    this.message = message;
    this.inputType = inputType;
    this.valueEnumeration = valueEnumeration;
    if (this.valueEnumeration == null) {
      this.valueEnumeration = Collections.emptyList();
    }
    this.defaultValue = defaultValue;
    this.answerId = answerId;
  }
  
  /**
   * @return The message to present to the user.
   */
  public LexicalItem getMessage() {
    return message;
  }

  /**
   * @return What type of input we request from the user: combobox, radio, etc.
   */
  public LexicalItem getInputType() {
    return inputType;
  }

  /**
   * @return value=>label mappings.
   */
  public Iterable<LexicalItem> getValueEnumeration() {
    return valueEnumeration;
  }
  
  /**
   * @return value=>label mappings.
   */
  public Map<LexicalItem, LexicalItem> getValueLabelMapping() {
    Map<LexicalItem, LexicalItem> mapping = new LinkedHashMap<>();
    
    if (!hasPendingValue()) {
      Iterator<LexicalItem> iterator = valueEnumeration.iterator();
      
      while (iterator.hasNext()) {
        LexicalItem value = iterator.next();
        LexicalItem label = iterator.next();
        
        mapping.put(value, label);
      }
    }
    
    return mapping;
  }
  
  /**
   * @return Default value.
   */
  public LexicalItem getDefaultValue() {
    return defaultValue;
  }

  /**
   * @return The value obtained from the user will be linked to this ID. 
   *         Later on, this ID can be used with the $answer editor variable.
   */
  public LexicalItem getAnswerId() {
    return answerId;
  }
  
  @Override
  public List<LexicalItem> getParams() {
    List<LexicalItem> params = new LinkedList<>();
    
    addMessage(params);
    addInputType(params);
    addValueEnumeration(params);
    addDefaultValue(params);
    addAnswerId(params);
    
    return params;
  }

  /**
   * Adds to the list of parameters the message to be presented to the user.
   * 
   * @param params The list of parameters where to add this parameter.
   */
  private void addMessage(List<LexicalItem> params) {
    if (message != null) {
      params.add(message);
    }
  }

  /**
   * Adds to the list of parameters the type of input requested to the user.
   * 
   * @param params The list of parameters where to add this parameter.
   */
  private void addInputType(List<LexicalItem> params) {
    if (inputType != null) {
      params.add(inputType);
    }
  }

  /**
   * Adds to the list of parameters the value=>label mappings.
   *
   * @param params The list of parameters where to add these parameters.
   * 
   * @return <code>true</code> if added some value=>label mappings, <code>false</code> otherwise.
   */
  private boolean addValueEnumeration(List<LexicalItem> params) {
    boolean added = false;
    if (valueEnumeration != null) {
      added = true;
      Iterator<LexicalItem> iterator = valueEnumeration.iterator();
      while (iterator.hasNext()) {
        LexicalItem lexicalItem = iterator.next();
        params.add(lexicalItem);
      }
    }

    return added;
  }

  /**
   * Adds to the list of parameters the default value of this editor variable.
   * 
   * @param params The list of parameters where to add this parameter.
   */
  private void addDefaultValue(List<LexicalItem> params) {
    if (defaultValue != null) {
      params.add(defaultValue);
    }
  }

  /**
   * Adds to the list of parameters the ID of the input value.
   * 
   * @param params The list of parameters where to add this parameter.
   */
  private void addAnswerId(List<LexicalItem> params) {
    if (answerId != null) {
      params.add(answerId);
    }
  }
  
  /**
   * @return <code>true</code> if the value:label map is not computed yet.
   */
  public boolean hasPendingValue() {
    boolean hasPending = false;
    Iterator<LexicalItem> iterator = valueEnumeration.iterator();
    if (iterator.hasNext()) {
      iterator.next();
      
      // If a second item exists, we have the mapping. 
      hasPending = !iterator.hasNext();
    }
    
    return hasPending;
  }
}