<?xml version="1.0" encoding="UTF-8"?>
<!--
    This stylesheet deals with static content images.
    
-->
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    xmlns:css="http://www.w3.org/1998/CSS"    
    xmlns:fo="http://www.w3.org/1999/XSL/Format" 
    xmlns:oxy="http://www.oxygenxml.com/css2fo"
    xmlns:fox="http://xmlgraphics.apache.org/fop/extensions"
    exclude-result-prefixes="xs oxy fox"
    version="2.0">
    
    <xsl:import href="xmltofo-static-content-html-image-maps.xsl"/>
    
    <!-- 
        This matches an image from the static content of an element from the main content of the page.
    -->
    <xsl:template match="css:uri[ancestor::css:before or ancestor::css:after or ancestor::css:content-on-element or ancestor::css:marker or ancestor::css:string-set]">
        <xsl:variable name="parent-element" select="../.."/>
        
        <fo:external-graphic src="{@uri}">
            
            <!-- Alignment. We need to copy here the vertical-align attribute. It does not work left only on the parent inline. -->
            <xsl:variable name="alignment-baseline" select="oxy:get-attr-from-static-or-parent($parent-element, .. , 'alignment-baseline')"/>
            <xsl:if test="$alignment-baseline">
                <xsl:attribute name="alignment-baseline" select="$alignment-baseline"/>
            </xsl:if>
            <xsl:variable name="vertical-align" select="oxy:get-attr-from-static-or-parent($parent-element, .. , 'vertical-align')"/>
            <xsl:if test="$vertical-align">
                <xsl:attribute name="vertical-align" select="$vertical-align"/>
            </xsl:if>
            
            <!-- We cannot use the parent element alignment for the string set inner images, has no meaning. 
                We prefer a middle position as it is easier to create an image to fill the header/footer. -->
            <xsl:if test="ancestor::css:string-set">
                <xsl:attribute name="vertical-align" select="'middle'"/>
            </xsl:if>
            
            <!-- Accessibility: the alternate text. 
                It can be specified directly on the :before, :after, or on the real element. -->
            <xsl:variable name="alt-text" select="(../@css:oxy-alt-text, $parent-element/@css:oxy-alt-text)[1]"/>
            <xsl:if test="$alt-text">
                <xsl:attribute name="fox:alt-text" select="$alt-text"/>
            </xsl:if>
            
            <xsl:choose>
                <!-- If this image is the single image in all :before or :after content, and there is no other 
                    static content around it, then we can  assume the @css:width/@css:height set on the parent 
                    element applies in fact to this image. -->
                <xsl:when test="count($parent-element/css:before/css:uri|
                    $parent-element/css:content-on-element/css:uri|
                    $parent-element/css:after/css:uri) = 1 
                    and 
                    count(../*) = 1 ">
                    
                    <!-- Collect the width from the static or its parent -->
                    <xsl:variable name="pWidth" select="oxy:get-attr-from-static-or-parent($parent-element, .. , 'width')"/>
                    <xsl:variable name="pHeight" select="oxy:get-attr-from-static-or-parent($parent-element, .. , 'height')"/>
                    
                    <!-- If the parent is an inline-block and the width or height are percentual, 
                        then use 100% because the width and height since the values already migrated to 
                        the generated fo:inline-block.
                    -->
                    <xsl:variable name="fullWidth">
                        <xsl:if test="$parent-element/@css:display='inline-block' and contains($pWidth,'%')">
                            <tmp css:width="100%">t</tmp>
                        </xsl:if>
                    </xsl:variable>
                    <xsl:variable name="fullHeight">
                        <xsl:if test="$parent-element/@css:display='inline-block' and contains($pHeight,'%')">
                            <tmp css:height="100%">t</tmp>
                        </xsl:if>
                    </xsl:variable>
                    
                    <xsl:variable name="pWidth"  select="if ($fullWidth/tmp)  then $fullWidth/tmp/@css:width else $pWidth" as="attribute(css:width)*"/>
                    <xsl:variable name="pHeight" select="if ($fullHeight/tmp) then $fullHeight/tmp/@css:height else $pHeight" as="attribute(css:height)*"/>
                    
                    <!-- Copy size constraints -->
                    <xsl:apply-templates select="oxy:get-attr-from-static-or-parent($parent-element, .. , 'max-width')"/>
                    <xsl:apply-templates select="oxy:get-attr-from-static-or-parent($parent-element, .. , 'min-width')"/>
                    <xsl:apply-templates select="oxy:get-attr-from-static-or-parent($parent-element, .. , 'max-height')"/>
                    <xsl:apply-templates select="oxy:get-attr-from-static-or-parent($parent-element, .. , 'min-height')"/>
                    
                    <xsl:choose>
                        
                        <!-- Both sizes are specified. Let it stretch. -->
                        <xsl:when test="$pWidth and $pHeight">
                            <xsl:attribute name="scaling" select="'non-uniform'"/>
                            
                            <xsl:attribute name="content-width" select="$pWidth"/>
                            <xsl:attribute name="content-height" select="$pHeight"/>
                        </xsl:when>
                        
                        <xsl:otherwise>
                            
                            <xsl:choose>
                                <!-- One size is specified. Use uniform scaling. -->
                                <xsl:when test="$pWidth or $pHeight">
                                    <xsl:attribute name="scaling" select="'uniform'"/>
                                    
                                    <xsl:if test="$pWidth">
                                        <xsl:apply-templates select="$pWidth"/>
                                        <xsl:attribute name="content-width" select="'scale-to-fit'"/>
                                    </xsl:if>
                                    
                                    <xsl:if test="$pHeight">
                                        <xsl:apply-templates select="$pHeight"/>
                                        <xsl:attribute name="content-height" select="'scale-to-fit'"/>
                                    </xsl:if>
                                    
                                </xsl:when>
                                
                                <!-- No size specified, but there is a image-resolution somewhere. -->
                                <xsl:when test="@extracted-image-width and @extracted-image-height">
                                    <xsl:variable name="image-resolution" select="oxy:extract-dpi-value(ancestor-or-self::*/@css:image-resolution[1])"/>
                                    
                                    <xsl:choose>
                                        <xsl:when test="string-length($image-resolution) > 0">
                                            <xsl:attribute name="scaling" select="'uniform'"/>	           	                        
                                            <xsl:attribute name="width" select=" concat(@extracted-image-width, 'in div ', $image-resolution)"/>
                                            <xsl:attribute name="content-width" select="'scale-to-fit'"/>
                                            <xsl:attribute name="height" select=" concat(@extracted-image-height, 'in div ', $image-resolution)"/>
                                            <xsl:attribute name="content-height" select="'scale-to-fit'"/>
                                        </xsl:when>
                                    </xsl:choose>
                                </xsl:when>
                                
                                <xsl:otherwise>
                                    <xsl:if test="$parent-element/@css:max-height | $parent-element/@css:min-height">
                                        <xsl:attribute name="content-height" select="'scale-to-fit'"/>
                                    </xsl:if>
                                    <xsl:if test="$parent-element/@css:max-width | $parent-element/@css:min-width">
                                        <xsl:attribute name="content-width" select="'scale-to-fit'"/>
                                    </xsl:if>
                                </xsl:otherwise>
                                
                            </xsl:choose>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:when>
                <xsl:otherwise>
                    <!-- An image that appears together with other static content, we cannot impose the width constraints. -->
                </xsl:otherwise>
            </xsl:choose>
            
        </fo:external-graphic>
    </xsl:template>
    
    <!-- 
        Gets an attribute from the static element or its parent
        @param parent The parent element.
        @param static The static element, for example a :before inside the parent element.
        @param attribute-name The name of the attribute to extract.
        @return the attribute, or empty.
    -->
    <xsl:function name="oxy:get-attr-from-static-or-parent" as="item()*">
        <xsl:param name="parent"/>
        <xsl:param name="static"/>
        <xsl:param name="attribute-name"/>
        
        <xsl:choose>
            <xsl:when test="$static/@css:*[local-name() = $attribute-name]">
                <xsl:copy-of select="$static/@css:*[local-name() = $attribute-name]"/>
            </xsl:when>
            <xsl:when test="$parent/@css:*[local-name() = $attribute-name]">
                <xsl:copy-of select="$parent/@css:*[local-name() = $attribute-name]"/>
            </xsl:when>
        </xsl:choose>
    </xsl:function>
    
    <!-- 
        This matches an image from a page margin box. It cannot have width and height, only image resolution.
        Maybe this should be in the xmltofo-pages.xsl
        
        This receives tunneled parameters send by the page margin box. With their help it 
        can find the 'image-resolution' property used in one of the @page or page margin 
        boxes definitions.
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').               
        @param margin-box-name The margin box, for instance 'top-left-corner', 'bottom-center', etc...
        @param property The name of the property to extract.
    -->
    <xsl:template match="css:uri">
        <xsl:param name="doc" tunnel="yes"/>
        <xsl:param name="page-name" tunnel="yes"/>
        <xsl:param name="page-side" tunnel="yes"/>
        <xsl:param name="page-position" tunnel="yes"/>
        <xsl:param name="page-blank" tunnel="yes"/>
        <xsl:param name="margin-box-name" tunnel="yes"/>
        
        <fo:external-graphic src="{@uri}">
            
            <!-- Alignment. We need to copy here the vertical-align attribute. It does not work left only on the parent inline. -->
            <xsl:variable name="alignment-baseline" select="oxy:get-property-for-page-margin-box-and-inherited-from-page($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name,'alignment-baseline')"/>
            <xsl:if test="$alignment-baseline">
                <xsl:attribute name="alignment-baseline" select="$alignment-baseline"/>
            </xsl:if>
            <xsl:variable name="vertical-align" select="oxy:get-property-for-page-margin-box-and-inherited-from-page($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name,'vertical-align')"/>
            <xsl:if test="$vertical-align">
                <xsl:attribute name="vertical-align" select="$vertical-align"/>
            </xsl:if>
            
            <xsl:if test="@extracted-image-width and @extracted-image-height">
                <xsl:variable name="image-resolution" 
                    select="oxy:extract-dpi-value(oxy:get-property-for-page-margin-box-and-inherited-from-page($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name,'image-resolution'))"/>
                <xsl:if test="$image-resolution">
                    <xsl:attribute name="scaling" select="'uniform'"/>
                    
                    <xsl:attribute name="width" select=" concat(@extracted-image-width, 'in div ', $image-resolution)"/>
                    <xsl:attribute name="content-width" select="'scale-to-fit'"/>
                    <xsl:attribute name="height" select=" concat(@extracted-image-height, 'in div ', $image-resolution)"/>
                    <xsl:attribute name="content-height" select="'scale-to-fit'"/>
                </xsl:if>
            </xsl:if>
        </fo:external-graphic>
        
    </xsl:template>
    
    <!-- 
        Extracts the DPI value (without the 'dpi' suffix) from an expression.
        
        @param resolution-expression The expression.
        @return the value, or null.
    -->
    <xsl:function name="oxy:extract-dpi-value">
        <xsl:param name="resolution-expression"/>
        <xsl:value-of select="substring-before($resolution-expression, 'dpi')"/>
    </xsl:function>
    
    <!-- 
        Ignore this attribute. The alternate text is set directly on the fo:graphic in the css:uri template.
    -->
    <xsl:template match="@css:oxy-alt-text" priority="2"/>
    
</xsl:stylesheet>