<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    xmlns:css="http://www.w3.org/1998/CSS"
    xmlns:fo="http://www.w3.org/1999/XSL/Format"
    xmlns:fox="http://xmlgraphics.apache.org/fop/extensions"
    xmlns:oxy="http://www.oxygenxml.com/css2fo"
    xmlns:chfox="http://www.oxygenxml.com/chemistry/extensions/fop"
    xmlns:nd="http://www.oxygenxml.com/css2fo/named-destinations" 
    exclude-result-prefixes="xs oxy css nd"
    version="2.0">    
    
    
    
    <!-- 
      The table. 
        
    -->
    <xsl:template match="*[@css:display='table']">
        <xsl:param name="generated-table-container" select="false()"/>
        <xsl:param name="inhibit-margins-and-width" select="false()" tunnel="yes"/>
        <!-- 
            Create a block container that will include the caption and table 
            The width and margin properties will be migrated to this container.
        -->
        <fo:table-and-caption>
            <xsl:call-template name="generate-inheritable-attributes-from-table-to-apply-on-captions"/>
            
            <xsl:if test="@css:width">
                <xsl:attribute name="width" select="@css:width"/>
            </xsl:if>
            
            <xsl:if test="not($inhibit-margins-and-width)">
                <xsl:if test="@css:margin-top">
                    <xsl:attribute name="space-before" select="@css:margin-top"/>
                </xsl:if>
                <xsl:if test="@css:margin-bottom">
                    <xsl:attribute name="space-after" select="@css:margin-bottom"/>
                </xsl:if>
                <xsl:if test="@css:margin-left and @css:margin-left != 'auto'">
                    <xsl:attribute name="margin-left" select="@css:margin-left"/>
                </xsl:if>
                <xsl:if test="@css:margin-right and @css:margin-right != 'auto'">
                    <xsl:attribute name="margin-right" select="@css:margin-right"/>
                </xsl:if>
				<!-- since the margin-left, margin-right do not accept 'auto' in FO we use an extension -->
                <xsl:if test="@css:margin-left='auto'">
                  <xsl:attribute name="chfox:margin-left" select="@css:margin-left"/>
                </xsl:if>
                <xsl:if test="@css:margin-right='auto'">
                  <xsl:attribute name="chfox:margin-right" select="@css:margin-right"/>
                </xsl:if>
            </xsl:if>            
            
            
            <xsl:apply-templates select="@css:page-break-before|@css:page-break-after|@css:page-break-inside"/>
            <xsl:apply-templates select="@css:oxy-column-break-before|@css:oxy-column-break-after|@css:oxy-column-break-inside"/>
            
            <xsl:if test="parent::css:reference-orientation-container[@reference-orientation]">
                <xsl:attribute name="reference-orientation" select="../@reference-orientation"/>
            </xsl:if>

            <!-- The changebars that are bordering the table content 
                 are moved outside for a better rendering. -->        
            <xsl:apply-templates select="*[@css:display='-oxy-changebar-start'][position()=1]">
              <xsl:with-param name="generate" select="'yes'" tunnel="yes"/>
            </xsl:apply-templates>

            
            <!-- Table caption -->
            <xsl:variable name="captions" select="*[@css:display='table-caption']"/>
            <!-- Process captions top -->
            <xsl:if test="exists($captions)">
                <xsl:apply-templates select="$captions[not(@css:caption-side) or @css:caption-side='top']"/>
            </xsl:if>
          
            <fo:table relative-align="baseline" chfox:html-table-width-policy="yes">
                <xsl:variable name="columns" select="*[@css:display='table-column'] | *[@css:display='table-column-group']/*[@css:display='table-column']"/>
                
                <!-- Width and Layout -->
                <xsl:choose>
                	<xsl:when test="parent::css:reference-orientation-container[@reference-orientation]">
		                <!-- No table layout 'auto' for rotated tables, and no width. -->
                		<xsl:attribute name="table-layout" select="'fixed'"/>                		
                	</xsl:when>
                	<xsl:otherwise>
		                <xsl:if test="@css:table-layout">                	
			                <!-- table-layout property -->
		                    <xsl:attribute name="table-layout" select="@css:table-layout"/>		                    
		                </xsl:if>
		                
                        <xsl:choose>
    		                <xsl:when test="@css:width">
    		                    <!-- 
    		                    	The user specified a width for the table.
    		                    	Since this is set on the block container, 
    		                    	the table should fill the entire block container.
    		                    	We need this as a marker for the 'auto' algorithm 
    		                    	
    		                    	to disambiguate the contentAreaIPD (is it set by the layout, and it can shrink, or it comes from the FO and should be respected?)
    		                    -->
    		                    <xsl:attribute name="width" select="'100%'"/>
    		                </xsl:when>
                        
                            <!-- Fixed layout table, all columns have specified width, no proportions on in their width expression, 
                                  the table has no with on its own. 
                                 The table will have as width the sum of the columns width. -->
                            <xsl:when test="@css:table-layout='fixed' and
                                                not (@css:width) and 
                                                count($columns) > 0 and
                                                count($columns) = count($columns/@css:width) and
                                                not(contains(string-join($columns/oxy:column-width(.), ' '), '*')) and 
                                                not(contains(string-join($columns/oxy:column-width(.), ' '), '%')) 
                                                ">
                                  <xsl:attribute name="width" select="string-join($columns/oxy:column-width(.), ' + ')"/>
                            </xsl:when>
                        
                        </xsl:choose>
                	</xsl:otherwise>
                </xsl:choose>

              <!-- 
                Generates the attributes. 
                Skips the inheritable ones, that were already placed on the parent fo:table-and-caption,
                and the ones we already placed on the fo:table.
              -->
              <xsl:apply-templates select="@*[not(oxy:is-inheritable(local-name()))] except 
                    (@css:display, @css:table-layout, @css:width,
                    @css:margin-bottom, @css:margin-left, 
                    @css:margin-right, @css:margin-top,
                    @css:page-break-after, 
                    @css:page-break-before, 
                    @css:page-break-inside
                    )"/>
                
                
                <xsl:if test="@css:margin-top">
                    <xsl:attribute name="space-before" select="0"/>
                </xsl:if>
                <xsl:if test="@css:margin-bottom">
                    <xsl:attribute name="space-after" select="0"/>
                </xsl:if>
                <xsl:if test="@css:margin-left">
                    <xsl:attribute name="start-indent" select="0"/>
                </xsl:if>
                <xsl:if test="@css:margin-right">
                    <xsl:attribute name="end-indent" select="0"/>
                </xsl:if>
                <xsl:if test="not(@css:border-collapse)">
                    <xsl:attribute name="border-collapse" select="'separate'"/>
                </xsl:if>	                    
                
                <xsl:variable name="body" select="*[@css:display='table-row-group']"/>
                
                <!-- Process table columns -->
                <xsl:if test="exists($columns)">
                    <xsl:call-template name="generate-table-columns">
                        <xsl:with-param name="columns" select="$columns"/>
                    </xsl:call-template>
                </xsl:if>
                
                <!-- Process headers -->
                <xsl:apply-templates select="*[@css:display='table-header-group']"/>
                
                <!-- Process footer -->
                <xsl:apply-templates select="*[@css:display='table-footer-group']"/>
                
                <!-- Process body -->
                <xsl:choose>
                    <xsl:when test="exists($body)">
                        <xsl:apply-templates select="css:before"/>
                        
                        <xsl:apply-templates select="$body">
                              <xsl:with-param name="columns" select="$columns" tunnel="yes"/>                       
                        </xsl:apply-templates>
                        
                      	<xsl:apply-templates select="css:after"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <!-- fo:table-body is required -->
                        <fo:table-body>
                            <xsl:apply-templates select="css:before"/>
                          
                            <xsl:apply-templates select="node() except (
                              *[@css:display='table-caption'], 
                              *[@css:display='table-column'], 
                              *[@css:display='table-header-group'], 
                              *[@css:display='table-footer-group'], 
                              css:before, 
                              css:after)">
                              <xsl:with-param name="columns" select="$columns" tunnel="yes"/>
                            </xsl:apply-templates>                                    
                          
                            <xsl:apply-templates select="css:after"/>
                        </fo:table-body>
                    </xsl:otherwise>
                </xsl:choose>
            </fo:table>


            <!-- Process captions bottom -->
            <xsl:if test="exists($captions)">
                <xsl:apply-templates select="$captions[@css:caption-side='bottom']"/>
            </xsl:if>
            
                    
            <!-- The changebars that are bordering the table content 
                 are moved outside for a better rendering. -->        
            <xsl:apply-templates select="*[@css:display='-oxy-changebar-end'][position()=last()]">
              <xsl:with-param name="generate" select="'yes'" tunnel="yes"/>
            </xsl:apply-templates>
            
        </fo:table-and-caption>
      
    </xsl:template>
    
    <!-- 
        The changebars that are bordering the table content 
        are moved outside the table-and-caption for a better rendering. 
        We let them generate change bar markup only when this template 
        is applied from the table template. 
    
        @param generate Set to 'yes' to allow generation.
        
    -->
    <xsl:template match="*[@css:display='table']/*[@css:display='-oxy-changebar-start'][position()=1]|
                         *[@css:display='table']/*[@css:display='-oxy-changebar-end'][position()=last()]" priority="10">
      <xsl:param name="generate" tunnel="yes" select="'no'"/>
      <xsl:if test="$generate = 'yes'"> 
        <xsl:next-match/>
      </xsl:if>
    </xsl:template>
    
    
    <xsl:template match="*[@css:display='table']/@css:border-collapse">
        <xsl:attribute name="border-collapse" select="."/>
    </xsl:template>
    
    
    
    <!-- 
        css:display='table-header-group' -> fo:table-header
    -->
    <xsl:template match="*[@css:display='table-header-group']" priority="3">
      <xsl:if test="*">
        <fo:table-header>
            <xsl:apply-templates select="@* except @css:display"/>
            <xsl:apply-templates select="css:before"/>
            <xsl:apply-templates select="node() except (css:before, css:after)"/>                                    
            <xsl:apply-templates select="css:after"/>
        </fo:table-header>
      </xsl:if>
    </xsl:template>
    
    <!-- 
        css:display='table-footer-group' -> fo:table-footer
    -->
    <xsl:template match="*[@css:display='table-footer-group']" priority="3">
      <xsl:if test="*">
        <fo:table-footer>
            <xsl:apply-templates select="@* except @css:display"/>
            <xsl:apply-templates select="css:before"/>
            <xsl:apply-templates select="node() except (css:before, css:after)"/>                                    
            <xsl:apply-templates select="css:after"/>
        </fo:table-footer>
      </xsl:if>
    </xsl:template>    
    
    <!-- 
        css:display='table-row-group' -> fo:table-body
    -->
    <xsl:template match="*[@css:display='table-row-group']" priority="3">
        <fo:table-body>
            <xsl:apply-templates select="@* except @css:display"/>
            <xsl:apply-templates select="css:before"/>
            <xsl:apply-templates select="node() except (css:before, css:after)"/>   
            <xsl:apply-templates select="css:after"/>
        </fo:table-body>        
    </xsl:template>
    
     <!-- 
        css:display='table-column-group' -> (/) has no correspondence in FO.
   
    <xsl:template match="*[@css:display='table-column-group']" priority="3">          
          <xsl:apply-templates select="css:before"/>
          <xsl:apply-templates select="node() except (css:before, css:after)"/>   
          <xsl:apply-templates select="css:after"/>
    </xsl:template>
     -->
    
    <!-- 
        css:display='table-row' -> fo:table-row (except empty rows)
    -->
    <xsl:template match="*[@css:display='table-row']" priority="3">
        <xsl:choose>
            <xsl:when test="child::node()">
                <fo:table-row>
                    <!-- Do not copy @id and @nd-id to the fo:table-row because internal links to table rows are not working in FOP. 
                     The template matching the cell first cell will use these @id and @nd-id on the generated fo:table-cell. -->
                    <xsl:apply-templates select="@* except (@css:display, @id, @nd:nd-id)"/>
                    <xsl:apply-templates select="css:string-set"/>                                    
                    <xsl:apply-templates select="css:before"/>
                    <xsl:apply-templates select="node() except (css:before, css:after, css:string-set)"/>                                    
                    <xsl:apply-templates select="css:after"/>
                </fo:table-row>
            </xsl:when>
            <xsl:otherwise>
                <xsl:message>WARNING One or more rows of a table have no cells.</xsl:message>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
         
    <!-- 
        Anything in a table group or table, that is not a row -> 
        	fo:table-row, except for the table caption, that is treated specially, from the table template.
    -->
    <xsl:template match="*[not(@css:display='table-caption') and 
                           not(@css:display='-oxy-changebar-start') and
                           not(@css:display='-oxy-changebar-end') and (
                           ../@css:display='table-row-group' or 
                           ../@css:display='table-header-group' or 
                           ../@css:display='table-footer-group' or 
                           ../@css:display='table')
                           ]" priority="2">
        <fo:table-row>
            <xsl:choose>
                <xsl:when test="@css:display = 'table-cell'">
                    <xsl:next-match/>
                </xsl:when>
                <xsl:otherwise>
                    <fo:table-cell
	                	start-indent="0" 
	                	end-indent="0">
                        <xsl:choose>
                            <xsl:when test="@css:display = 'block'">
                                <xsl:next-match/>
                            </xsl:when>
                            <xsl:otherwise>
                                <fo:block>
                                    <xsl:next-match/>
                                </fo:block>
                            </xsl:otherwise>
                        </xsl:choose>
                    </fo:table-cell>
                </xsl:otherwise>
            </xsl:choose>
        </fo:table-row>
    </xsl:template>

    <!-- 
        Anything in a row, other than changebars, that is not a cell -> 
        report it, it may break the fo:table, 
        FOP does not accept extra cells.
    -->
    <xsl:template match="*[../@css:display='table-row'][
                           not(@css:display='-oxy-changebar-start') and
                           not(@css:display='-oxy-changebar-end')
                           ]" priority="2">
        <xsl:choose>
            <xsl:when test="@css:display = 'table-cell'">
                <xsl:next-match/>
            </xsl:when>
            <xsl:otherwise>
           		<!-- Cannot deal with this, emitting extra cells for elements with display not 'table-cell' can break the table layout. -->
            	<xsl:variable name="content" select="normalize-space(.)"/>
            	<xsl:if test="string-length($content) > 0 and not(@css:display='none')">
	           		<xsl:message>WARNING Filtered out element '<xsl:value-of select="name()"/>' with content: '<xsl:value-of select="$content"/>'. 
	           					 Was expecting an element with the display 'table-cell', but was: '<xsl:value-of select="@css:display"/>'.  </xsl:message>
            	</xsl:if>            	
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    
    <!-- 
        @css:display='table-cell' -> fo:table-cell
    -->
    <xsl:template match="*[@css:display='table-cell']">
        <xsl:param name="columns" tunnel="yes"/>
    
        <fo:table-cell  
        	start-indent="0" 
        	end-indent="0">
          
          
            <!-- Apply the styles set on the column. -->
            <xsl:variable name="idx" select="@css:table-column + 1"/>
            <xsl:variable name="column" select="$columns[$idx]"/>
            <xsl:if test="$column">
            
              <xsl:if test="$column/../@css:display='table-column-group'">
                <!-- Use also the properties defined on the table column group, if the column belongs to one. -->
                <xsl:apply-templates select="$column/../@* except ($column/@css:width, $column/@css:min-width, $column/@css:max-width, $column/@css:background-color)"/>               
              </xsl:if>
              <xsl:apply-templates select="$column/@* except ($column/@css:width, $column/@css:min-width, $column/@css:max-width, $column/@css:background-color)"/>
                             
            </xsl:if>
            
            
            <!-- Deal with row and column span. -->
        	<xsl:apply-templates select="@css:table-row-span"/>
        	<xsl:apply-templates select="@css:table-column-span"/>
            <xsl:apply-templates select="@* except (@css:display, @css:table-row-span, @css:table-column-span)"/>
            
            <!-- Set both @id and @nd-id from the parent row because internal links to table rows are not working in FOP. -->
            <xsl:if test="../(@id | @nd:nd-id) and not(preceding-sibling::*[@css:display='table-cell'])">
                <fo:block><xsl:apply-templates select="../(@id | @nd:nd-id)"/></fo:block>
            </xsl:if>
            
            <fo:block>
                <xsl:apply-templates select="css:before"/>
                <xsl:apply-templates select="node() except (css:before, css:after)"/>                                    
                <xsl:apply-templates select="css:after"/>
            </fo:block>
        </fo:table-cell>
    </xsl:template>
    
    <!-- Implement row span and column span. -->    
    <xsl:template match="@css:table-row-span">
    	<xsl:attribute name="number-rows-spanned" select="."/>
    </xsl:template>
    
    <xsl:template match="@css:table-column-span">
    	<xsl:attribute name="number-columns-spanned" select="."/>
    </xsl:template>
    
    
    <!-- 
    
    Captions. 
    
     -->
    
    <xsl:template match="@css:oxy-show-only-when-caption-repeated-on-next-pages"/>
    <xsl:template match="@css:oxy-caption-repeat-on-next-pages"/>
    
    
    <xsl:template match="css:before[@css:oxy-show-only-when-caption-repeated-on-next-pages='yes']|
                          css:after[@css:oxy-show-only-when-caption-repeated-on-next-pages='yes']">
      <xsl:param name="caption-for-next-page" tunnel="yes"/>
      <xsl:choose>
        <xsl:when test="$caption-for-next-page = 'yes'">
          <!-- It is called when generating a fo:caption-for-next-page -->
          <xsl:next-match/>
        </xsl:when>
        <xsl:otherwise>
          <!-- Ignore, it is called in a normal fo:caption -->
        </xsl:otherwise>
      </xsl:choose>
    
    </xsl:template>
    
     <xsl:template match="*[@css:display='table']/*[@css:display='table-caption']//@id">
      <xsl:param name="caption-for-next-page" tunnel="yes"/>
      <!-- Avoid copying IDs when generating fo:caption-for-next-page to avoid conflicts -->
      <xsl:if test="$caption-for-next-page != 'yes'">
          <xsl:next-match/>   
      </xsl:if>
    </xsl:template>
    
    <!-- 
        @css:display='table-caption' -> fo:block
    -->
    <xsl:template match="*[@css:display='table']/*[@css:display='table-caption']" priority="6">
      <fo:table-caption start-indent="0" end-indent="0">
        <xsl:call-template name="generate-caption-keeps"/>
        <xsl:next-match/>
      </fo:table-caption>

      <!-- If the caption repeats on the next pages, add a new special element. -->
      <xsl:if test="@css:oxy-caption-repeat-on-next-pages='yes'">
        <fo:table-caption-for-next-page start-indent="0" end-indent="0">
          <xsl:call-template name="generate-caption-keeps"/>
          <xsl:next-match>
            <xsl:with-param name="caption-for-next-page" select="'yes'" tunnel="yes"/>
          </xsl:next-match>
        </fo:table-caption-for-next-page>
      </xsl:if>
      
       
    </xsl:template>
    
    <!-- Generate attributes to keep the caption with the grid rows, avoiding page breaks. -->
    <xsl:template name="generate-caption-keeps">
      <xsl:choose >
        <xsl:when test="count(preceding-sibling::*) = 0">
          <xsl:attribute name="keep-with-next" select="'always'"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:attribute name="keep-with-previous" select="'always'"/>
        </xsl:otherwise>
      </xsl:choose>    
    </xsl:template>

    <!-- 
      The caption is not kept in the table, but moved as a sibling relative to it so they are 
      both in the parent fo:table-and-caption.      
      So we have to copy the inheritable properties from the table to the table-and-caption, 
      since the normal inheritance does not apply anymore.
     -->
    <xsl:template name="generate-inheritable-attributes-from-table-to-apply-on-captions">
      <xsl:for-each select="@*">
        <xsl:if test="oxy:is-inheritable(local-name())">
          <xsl:apply-templates select="."/>
        </xsl:if>
      </xsl:for-each>
    </xsl:template>

    
    <xsl:template match="*[@css:display='table-caption']/@css:table-caption"/>
    <xsl:template match="*[@css:display='table-caption']/@css:caption-side"/>
    
    <!-- 
    	Gets the column width from the col element. The column width
    	may be specified using a CSS width property, or as the width attribute in the document.
    	
    	@param column-element The element defining the column properties.
    -->
    <xsl:function name="oxy:column-width">
        <xsl:param name="column-element"/>
		<!-- If the width is not coming from CSS, try some well known attribute names for HTML, CALS-->
        <xsl:variable name="w" select="
        	if ($column-element/@css:width) then 
        			$column-element/@css:width else 
        			if ($column-element/@width) then 
        				$column-element/@width else 
        				$column-element/@colwidth"/>
        				
        <xsl:sequence select="$w"/>                
    </xsl:function>
    
    <!-- 
        Generate fo:table-column elements.
    -->
    <xsl:template name="generate-table-columns">
        <xsl:param name="columns"/>
        
        <xsl:for-each select="$columns">
            <fo:table-column>
                
                <xsl:variable name="w" select="oxy:column-width(.)"/>
                <xsl:choose>
                	<xsl:when test="not($w)">
                		<!-- No width specified. -->
                	</xsl:when>                	
                    <xsl:when test="contains($w, '*')">
                        <!-- It is a proportional width. Solve it to % -->
                        
                        <xsl:variable name="total-prop-width" select="sum(
                            ..//*[contains(oxy:column-width(.), '*')]/number(
                                if (string-length(normalize-space(substring-before(oxy:column-width(.), '*'))) = 0) 
                                    then 1 
                                    else normalize-space(substring-before(oxy:column-width(.), '*')) )
                            )"/>
                                                
                        <xsl:variable name="prop-width" 
                            select="number(
                                if (string-length(normalize-space(substring-before($w, '*'))) = 0) 
                                    then 1 
                                    else normalize-space(substring-before($w, '*')))"/>
                        
                        <xsl:if test="string($total-prop-width) != 'NaN'">
                            <xsl:variable name="percent" select="round(($prop-width div $total-prop-width) *1000000) div 10000"/>
                            <xsl:variable name="percent" select="if (round($percent) = $percent) then $percent else format-number($percent,'##0.0000')"/>		
                            <xsl:attribute name="column-width" select="concat($percent, '%')"/>                                
                        </xsl:if>
                        
                    </xsl:when>
                    <xsl:when test="$w = 'fill'">
    					<!-- Since the fill value is not recognized by FOP, we pass it as a foreign attribute -->
                        <xsl:attribute name="chfox:fill" select="'yes'"></xsl:attribute>
                    </xsl:when>                 
                    
                    <xsl:otherwise>
                        <!-- It is a fixed value, or a percent. Use it as it is -->
                        <xsl:attribute name="column-width" select="$w"/>                                
                    </xsl:otherwise>
                </xsl:choose>
                

                <xsl:if test="../@css:display='table-column-group'">
                  <!-- If the column is in a colgroup, it should inherit some of its attributes. -->
                  <xsl:apply-templates select="../@* except (@css:width, @css:display)"/>
                </xsl:if>
                
                <!-- Process the remaining attributes -->
                <xsl:apply-templates select="@* except (@css:width, @css:display)"/>
                
            </fo:table-column>
        </xsl:for-each>
    </xsl:template>
    
    
    <!-- 
        
        Catches invalid table structures and reports them to the user.
        
        'table-row' and 'table-column' without 'table' parent
    -->
    <xsl:template match="
        *[@css:display='table-column'][not(../@css:display='table') and not(../@css:display='table-column-group')]" 
        priority="3">
        
        <xsl:call-template name="report-invalid-table-parent-structure">
            <xsl:with-param name="expected-parent-display" select="'table'"/>
        </xsl:call-template>        
    </xsl:template>

    <!-- 
        
        Catches invalid table structures and reports them to the user.
        
        'table-cell' without 'table-row' parent
    -->
    <xsl:template match="
        *[@css:display='table-cell'][not(../@css:display='table-row')]">
        
        <xsl:call-template name="report-invalid-table-parent-structure">
            <xsl:with-param name="expected-parent-display" select="'table-row'"/>
        </xsl:call-template>        
    </xsl:template>

    

    <xsl:template match="
        *[@css:display='table-row'][
            not(../@css:display='table') and
            not(../@css:display='table-row-group') and
            not(../@css:display='table-header-group') and
            not(../@css:display='table-footer-group')
            ]" priority="4">
        
        <fo:table table-layout="auto" width="100%">
            <fo:table-body>
                <xsl:next-match/>
            </fo:table-body>
        </fo:table>

    </xsl:template>
    
    
    <!-- 
        
        Reports invalid display values set on an element, in the context of a parent element.
        
        @param expected-parent-display The expected value for the parent @display attribute.
        
    -->
    <xsl:template name="report-invalid-table-parent-structure">
        <xsl:param name="expected-parent-display" as="xs:string" required="yes"/>
        
        <xsl:message xml:space="preserve" terminate="yes">Unexpected display value '<xsl:value-of select="@css:display"/>' for element '<xsl:value-of select="local-name(.)"/>'. The parent '<xsl:value-of select="../local-name()"/>' should have display '<xsl:value-of select="$expected-parent-display"/>'. The parent structure: <xsl:copy-of select=".."/> </xsl:message>
    </xsl:template>
    
    <!-- Template for the -oxy-pdf-table-omit-footer-at-break property-->
    <xsl:template match="@css:oxy-pdf-table-omit-footer-break">
        <xsl:attribute name="table-omit-footer-at-break">true</xsl:attribute>
    </xsl:template>
    
    <!-- Template for the -oxy-pdf-table-omit-header-at-break property -->
    <xsl:template match="@css:oxy-pdf-table-omit-header-break">
        <xsl:attribute name="table-omit-header-at-break">true</xsl:attribute>
    </xsl:template>

</xsl:stylesheet>