<?xml version="1.0" encoding="UTF-8"?>
<!-- 
    Generates different FO elements based on the css:display attribute:
    
    css:display="block" -> fo:block
    css:display="inline" -> fo:inline
    etc..
        
-->
<xsl:stylesheet 
    xmlns:xsl="http://www.w3.org/1999/XSL/Transform" 
    xmlns:xs="http://www.w3.org/2001/XMLSchema" 
    xmlns:fo="http://www.w3.org/1999/XSL/Format" 
    xmlns:oxy="http://www.oxygenxml.com/css2fo" 
    xmlns:css="http://www.w3.org/1998/CSS" 
    
    exclude-result-prefixes="xs oxy css" 
    version="2.0">


    <!-- 
    
    	Default template for all blocks.
    
    -->
    <xsl:template match="*[@css:display = 'block'] | *[@css:display = 'table-caption'] | *[@css:display = 'inline-block'] | css:marker[*]" priority="3">
        <xsl:param name="generate-block" select="'normal'" tunnel="yes"/>
        
        <xsl:choose>
            <xsl:when test="$generate-block = 'normal'">
                <!-- Regular blocks.-->
                <fo:block>
                    <xsl:apply-templates select="@* except @css:display"/>
                    <xsl:call-template name="generate-block-content"/>
                </fo:block>
            </xsl:when>
            <xsl:otherwise>
                <!--
                    Generates the block-container or inline-container inner block.
                    
                    This block is needed to avoid "start-indent" and "end-indent" being inherited from 
                    parent blocks or containers to the children. Note that the zero value is relative to the 
                    block-container we just emitted.
                -->
                <fo:block
                    end-indent="0"
                    start-indent="0">
                    
                    <!-- Accessiblity, text decoration, put it on the block, on the block-container or inline-container is not valid. -->
                    <xsl:apply-templates select="@css:oxy-pdf-tag-type|@css:text-decoration"/>
                    
                    <xsl:call-template name="generate-block-content"/>                
                </fo:block>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!--
        Generates the content of a block, takes into accound the bidi override and direction.        
    -->
    <xsl:template name="generate-block-content">
        
        <!-- We need the specified or inherited direction flag -->
        <xsl:variable name="unicode-bidi" select="@css:unicode-bidi"/>
        <xsl:variable name="direction" select="if ($unicode-bidi) then oxy:get-inherited-direction(.) else false()"/>
        
        
        
        <xsl:choose>
            <xsl:when test="$direction and $unicode-bidi='bidi-override'">
                <fo:bidi-override direction="{$direction}" unicode-bidi="{$unicode-bidi}">
                    <xsl:apply-templates select="node()"/>
                </fo:bidi-override>
            </xsl:when>
            <xsl:otherwise>
                <xsl:apply-templates select="node()"/>
            </xsl:otherwise>
        </xsl:choose>
        
    </xsl:template>
    
    <!--  
	
		Generic template for all elements with display inline-block. Can generate a fo:inline-container with a  
		fo:block-container inside, and/or fo:basic-links around the fo:blocks.
	
	 -->
    <xsl:template match="*[@css:display = 'inline-block']" priority="5">
        <fo:inline>
            
            <!-- The margins set on on inlines are ignored, so we convert them 
                to padding of the parent. -->
            <xsl:if test="@css:margin-left">
                <xsl:attribute name="space-start" select="@css:margin-left"/>
            </xsl:if>
            <xsl:if test="@css:margin-right">
                <xsl:attribute name="space-end" select="@css:margin-right"/>
            </xsl:if>
            
            <fo:inline-container>
                <!-- The width should be set on the inline container, otherwise is getting wide as the parent. -->
                <xsl:apply-templates select="@css:width | 
                                             @css:min-width |
                                             @css:max-width |                                              
                                             @css:vertical-align |
                                             @css:alignment-baseline"/>
                
                <!-- The width must not be set if the inline container is positioned. -->
                <xsl:if test="@css:position='absolute' or @css:position='fixed'">
                    <xsl:attribute name="width" select="0"/>
                </xsl:if>
            
                <!-- The width and height should be placed on the block container, in this way the background is getting as big as the entire element. -->
                <fo:block-container margin="0">
                    <xsl:apply-templates select="@* except (
                                                 @css:width,
                                                 @css:min-width,
                                                 @css:max-width,
                                                 @css:vertical-align,
                                                 @css:alignment-baseline,
                                                 @css:margin-left,
                                                 @css:margin-right,
                                                 @css:margin-top,
                                                 @css:margin-bottom,
                                                 @css:oxy-pdf-tag-type)"/>
                    
                    <!-- The width should be set on the block container if the inline container is positioned. -->
                    <xsl:if test="@css:position='absolute' or @css:position='fixed'">
                        <xsl:apply-templates select="@css:width |
                            @css:min-width |
                            @css:max-width"/>
                    </xsl:if>
                    
                    <xsl:next-match>
                        <xsl:with-param name="generate-block" select="'inner'" tunnel="yes"/>
                    </xsl:next-match>
                </fo:block-container>
            </fo:inline-container>
        </fo:inline>
    </xsl:template>
  
    
    <!--  
        
        Generic template for all elements with display block. Can generate a fo:block, 
		fo:block-container with a block inside, and/or fo:basic-links around the fo:blocks.
	
	 -->
    <xsl:template match="*[@css:display = 'block'] | *[@css:display = 'table-caption'] | css:marker[*]" priority="5">
        <xsl:variable name="altering-indents-attributes"
            select="
                @css:width |
                @css:height |
                @css:min-height |
                @css:margin-top |
                @css:margin-left |
                @css:margin-bottom |
                @css:margin-right |
                @css:border-top-width |
                @css:border-left-width |
                @css:border-bottom-width |
                @css:border-right-width |
                @css:padding-top |
                @css:padding-left |
                @css:padding-bottom |
                @css:padding-right"/>
                
        <xsl:variable name="bidi" select="@css:direction | @css:unicode-bidi"/>

        <xsl:variable name="position" select="@css:position"/>

        <xsl:choose>
            <xsl:when
                test="
                    $altering-indents-attributes or
                    $bidi or
                    $position or
                    parent::css:reference-orientation-container">

                <!-- We need a container to:
                    
                    1. set here the width and height (if this is the case).
                    2. set all the properties that may alter the "start-indent" and "end-indent"
                    
                    This is especially important, since the block-containers define a reference area, and 
                    have their "start-indent" reported to the last reference area, while the normal blocks 
                    have their "start-indent" relative to the parent block, or parent block-container, 
                    whichever is closer.
                    
                    So mixing blocks and containers is messy. To overcome this problem, we use block-containers
                    for all the elements from the intermediate format that have a border, padding or 
                    margin that alter the "start-indent" or "end-indent".
                -->
                <fo:block-container>
        
                    <!-- Now copy the attributes on this container. -->
                    
                    <xsl:choose>
                        <xsl:when test="parent::css:reference-orientation-container">
                            <!-- To fix repetedly rotating fo:container-blocks that nest, set the value to zero. -->
                            <xsl:attribute name="reference-orientation" select="'0'"/>
                            <!-- To not bleed padding outside the area, put margin to zero (CSS Compatibility fix) -->
                            <xsl:attribute name="margin" select="'0'"/>
                            
                            <!-- We do not propagate the "margin" attributes. These are taken care by the parent block-container that does the rotation. -->
                            <xsl:apply-templates select="@* except (@css:margin-left, @css:margin-right, @css:margin-top, @css:margin-bottom, @css:oxy-pdf-tag-type, @css:text-decoration)"/>                            
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:apply-templates select="@* except (@css:oxy-pdf-tag-type, @css:text-decoration)" />
                        </xsl:otherwise>
                    </xsl:choose>
                    
                    <!-- Now generate the content. -->
                    <!-- A normal block, whith no effect on the "start-indent" or "end-indent" -->
                    <xsl:next-match>
                        <xsl:with-param name="generate-block" select="'inner'" tunnel="yes"/>
                    </xsl:next-match>
                    
                </fo:block-container>
            </xsl:when>
            
            <xsl:otherwise>
                <!-- A normal block, whith no effect on the "start-indent" or "end-indent" -->
                <xsl:next-match>
                    <xsl:with-param name="generate-block" select="'normal'" tunnel="yes"/>
                </xsl:next-match>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- 
    
      Space conditionality.
      
     -->  
    <xsl:template match="@css:oxy-space-before-conditionality">
        <xsl:if test="../@css:margin-top">
            <xsl:attribute name="space-before.conditionality" select="../@css:oxy-space-before-conditionality"/>
        </xsl:if>
    </xsl:template>
    <xsl:template match="@css:oxy-space-after-conditionality">
        <xsl:if test="../@css:margin-bottom">
            <xsl:attribute name="space-after.conditionality" select="../@css:oxy-space-after-conditionality"/>
        </xsl:if>
    </xsl:template>
    
    
</xsl:stylesheet>
