<!-- 
	Generates the bookmark tree. 
-->
<xsl:stylesheet 
    xmlns:xsl="http://www.w3.org/1999/XSL/Transform" 
    xmlns:xs="http://www.w3.org/2001/XMLSchema" 
    xmlns:fo="http://www.w3.org/1999/XSL/Format"
    xmlns:oxy="http://www.oxygenxml.com/css2fo" 
    xmlns:css="http://www.w3.org/1998/CSS" 
    exclude-result-prefixes="xs fo css oxy" 
    version="2.0" 
    xpath-default-namespace="http://www.w3.org/1998/CSS">

    <!-- 
		Creates the tree of fo:bookmarks taking into account the elements that 
		have a @bookmark-level, and a css:bookmark-label subelement.
		
		@param bookmark-nodes The list of nodes that have a @bookmark-level, 
		and a css:bookmark-label subelement
	-->
    <xsl:template name="build-bookmark-tree">
        <xsl:param name="bookmark-nodes" as="item()*" required="yes"/>

        <!-- If the parameter is empty, do not generate anything. -->
        <xsl:variable name="first" select="$bookmark-nodes[1]"/>
        <xsl:if test="$first">

            <!-- Determine the position of the next element that has the level 
			     less or equal to the first element level. -->
            <xsl:variable name="tmp">
                <xsl:text>C</xsl:text>                                            
                <xsl:for-each select="$bookmark-nodes[position() > 1]">
                    <xsl:choose>
                        <xsl:when test="@css:bookmark-level > $first/@css:bookmark-level">
                            <xsl:text>C</xsl:text>                            
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:text>E</xsl:text>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:for-each>
            </xsl:variable>
            
            <xsl:variable name="split-position" select="if(contains($tmp, 'E')) then string-length(substring-before($tmp, 'E')) else -1 "/>


            <fo:bookmark>
                <!-- Points to the target element from the content. -->
                <xsl:attribute name="internal-destination" select="if($first/@id) then $first/@id else generate-id($first)"/>

                <!-- Bookmark may be open or closed. -->
                <xsl:if test="$first/@css:bookmark-state">
                    <xsl:attribute name="starting-state">
                        <xsl:choose>
                            <xsl:when test="$first/@css:bookmark-state = 'closed'">hide</xsl:when>
                            <xsl:otherwise>show</xsl:otherwise>
                        </xsl:choose>
                    </xsl:attribute>
                </xsl:if>
                
                <!-- Creates the label-->
                <fo:bookmark-title>
                    <xsl:value-of select="oxy:get-all-static-as-text($first/css:bookmark-label)"/>
                </fo:bookmark-title>
                
                <!-- Recursion for the children. -->
              	<xsl:variable name="part1" select="$bookmark-nodes[position() &lt;= $split-position or $split-position = -1][position() > 1]"/>
                <xsl:call-template name="build-bookmark-tree">
                    <xsl:with-param name="bookmark-nodes" select="$part1"/>
                </xsl:call-template>
                
            </fo:bookmark>

			<!-- Recursion for the siblings. -->
            <xsl:if test="$split-position > 0">
                <xsl:variable name="part2" select="$bookmark-nodes[position() &gt; $split-position]"/>
            	<xsl:call-template name="build-bookmark-tree">
                    <xsl:with-param name="bookmark-nodes" select="$part2"/>
                </xsl:call-template>
            </xsl:if>
        </xsl:if>

    </xsl:template>
    
    
    <!--
        Generates the fo:bookmark-tree if there are elements that 
		have a @bookmark-level, and a css:bookmark-label subelement.
        
        Context: The root node.
    -->
    <xsl:template name="generate-bookmark-tree">
		<xsl:variable name="bookmark-nodes" select="//*[@css:bookmark-level][css:bookmark-label]"/>
        <xsl:if test="$bookmark-nodes">
            <fo:bookmark-tree>
                <xsl:call-template name="build-bookmark-tree">
                    <xsl:with-param name="bookmark-nodes" select="$bookmark-nodes"/>
                </xsl:call-template>
            </fo:bookmark-tree>
        </xsl:if>        
    </xsl:template>


	<!-- 
		Generate an @id for the elements that have a @bookmark-level, and 
		a css:bookmark-label subelement, but they miss the id.  
	-->
    <xsl:template match="@css:bookmark-level[../css:bookmark-label][not(../@id)]" priority="3">
        <xsl:attribute name="id" select="generate-id(..)"/>
        <xsl:next-match/>
    </xsl:template>


    <!-- 
        Do not propagate the bookmark elements and attributes to the FO, 
        are useless there. 
    -->
    <xsl:template match="@css:bookmark-level" priority="2"/>        
    <xsl:template match="@css:bookmark-state" priority="2"/>
    <xsl:template match="css:bookmark-label" priority="2"/>
    
    
</xsl:stylesheet>
