/*
 * Copyright (c) 2000 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 *
 * $Id$
 */
package org.w3c.flute.parser.selectors;

import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.Locator;
import org.w3c.css.sac.NegativeSelector;
import org.w3c.flute.css.sac.Condition;
import org.w3c.flute.css.sac.ConditionalSelector;
import org.w3c.flute.css.sac.DescendantSelector;
import org.w3c.flute.css.sac.ElementSelector;
import org.w3c.flute.css.sac.Selector;
import org.w3c.flute.css.sac.SelectorFactory;
import org.w3c.flute.css.sac.SiblingSelector;
import org.w3c.flute.css.sac.SimpleSelector;

/**
 * Implementation for a selector factory.
 * 
 * @version $Revision$
 * @author  Philippe Le Hegaret, dan
 */
public class SelectorFactoryImpl implements SelectorFactory {

  /**
   * Creates a conditional selector.
   * 
   * @param selector a selector.
   * @param condition a condition
   * @return the conditional selector.
   * @exception CSSException If this selector is not supported.
   */
  @Override
  public ConditionalSelector createConditionalSelector(SimpleSelector selector, Condition condition, Locator locator)
      throws CSSException {
    ConditionalSelectorImpl selectorImpl = new ConditionalSelectorImpl(selector, condition, locator);
    return selectorImpl;
  }

  /**
   * Creates an any node selector.
   * 
   * @return the any node selector.
   * @exception CSSException If this selector is not supported.
   */
  @Override
  public SimpleSelector createAnyNodeSelector(Locator locator) throws CSSException {
    throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
  }

  /**
   * Creates an root node selector.
   * 
   * @return the root node selector.
   * @exception CSSException If this selector is not supported.
   */
  @Override
  public SimpleSelector createRootNodeSelector(Locator locator) throws CSSException {
    throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
  }

  /**
   * Creates an negative selector.
   * 
   * @param selector a selector.
   * @return the negative selector.
   * @exception CSSException If this selector is not supported.
   */
  @Override
  public NegativeSelector createNegativeSelector(SimpleSelector selector, Locator locator) throws CSSException {
    throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
  }

  /**
   * Creates an element selector.
   * 
   * @param namespaceURI the <a href="http://www.w3.org/TR/REC-xml-names/#dt-NSName">namespace
   *                     URI</a> of the element selector.
   * @param localName the <a href="http://www.w3.org/TR/REC-xml-names/#NT-LocalPart">local
   *        part</a> of the element name. <code>NULL</code> if this element
   *        selector can match any element.
   * @return the element selector
   * @exception CSSException If this selector is not supported.
   */
  @Override
  public ElementSelector createElementSelector(String namespaceURI, String localName, Locator locator)
      throws CSSException {
    return new ElementSelectorImpl(namespaceURI, localName, locator);
  }

  /**
   * Creates a pseudo element selector.
   * 
   * @param pseudoName the pseudo element name. <code>NULL</code> if this
   *                   element selector can match any pseudo element.
   *     
   * @return the element selector
   * @exception CSSException If this selector is not supported.
   */
  @Override
  public ElementSelector createPseudoElementSelector(String namespaceURI, String pseudoName, Locator locator)
      throws CSSException {
    if (namespaceURI != null) {
      throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
    } else {
      return new PseudoElementSelectorImpl(pseudoName, locator);
    }
  }

  /**
   * Creates a descendant selector.
   *
   * @param parent the parent selector
   * @param descendant the descendant selector
   * @param isSubject true if the simple selector has been marked as the subject. http://dev.w3.org/csswg/selectors4/#subject
   * @return the combinator selector.
   * @exception CSSException If this selector is not supported.
   */
  @Override
  public DescendantSelector createDescendantSelector(Selector parent, SimpleSelector descendant,
      boolean isSubject, Locator locator) throws CSSException {
    return new DescendantSelectorImpl(parent, descendant, isSubject, locator);
  }

  /**
   * Creates a child selector.
   *
   * @param parent the parent selector
   * @param child the child selector
   * @param isSubject true if the simple selector has been marked as the subject. http://dev.w3.org/csswg/selectors4/#subject
   * @return the combinator selector.
   * @exception CSSException If this selector is not supported.
   */
  @Override
  public DescendantSelector createChildSelector(Selector parent, SimpleSelector child,
      boolean isSubject, Locator locator) throws CSSException {
    return new ChildSelectorImpl(parent, child, isSubject, locator);
  }

  /**
   * Creates a direct adjacent selector.
   *
   * @param child the child selector
   * @param directAdjacent the direct adjacent selector
   * @return the combinator selector.
   * @exception CSSException If this selector is not supported.
   */
  @Override
  public SiblingSelector createDirectAdjacentSelector(short nodeType, Selector child,
      SimpleSelector directAdjacent, Locator locator) throws CSSException {
    if (nodeType != 1) {
      throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
    } else {
      return new DirectAdjacentSelectorImpl(child, directAdjacent, locator);
    }
  }

  /**
   * Creates a direct adjacent selector.
   *
   * @param nodeType The type of the node.
   * @param child the child selector
   * @param directAdjacent the direct adjacent selector
   * @return the combinator selector.
   * @exception CSSException If this selector is not supported.
   */
  @Override
  public SiblingSelector createGeneralAdjacentSelector(short nodeType, Selector child,
      SimpleSelector directAdjacent, Locator locator) throws CSSException {
    if (nodeType != 1) {
      throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
    } else {
      return new GeneralAdjacentSelectorImpl(child, directAdjacent, locator);
    }
  }


}
