/*
 * Copyright (c) 2000 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 *
 * $Id$
 */
package org.w3c.flute.parser.selectors;

import java.util.Map;

import org.w3c.css.sac.Locator;
import org.w3c.flute.css.sac.DescendantSelector;
import org.w3c.flute.css.sac.Selector;
import org.w3c.flute.css.sac.SimpleSelector;
import org.w3c.flute.css.sac.SubjectSelector;

/**
 * Implementation of a descendant selector.
 * 
 * @version $Revision$
 * @author  Philippe Le Hegaret, Dan
 */
public class DescendantSelectorImpl extends AbstractLocalizableSelector implements DescendantSelector {

    private Selector       ancestorSelector;
    private SimpleSelector simpleSelector;
    /**
     * <code>true</code> if the simple selector is the subject in a larger descendant selector.
     */
    private boolean isSubject;
    
    /**
     * <code>true</code> if this selector simple selector is the subject,
     * or there is a subject in the ancestor selector.
     */
    private boolean hasSubject;

    /**
     * An integer indicating the type of <code>Selector</code>
     */
    @Override
    public short getSelectorType() {
       return Selector.SAC_DESCENDANT_SELECTOR;
    }

    /**
     * Creates a new descendant selector.
     * 
     * @param ancestorSelector The selector that should matches the ancestor.
     * @param simpleSelector The selector that matches the current element.
     * @param isSubject <code>true</code> if the simple selector is the subject in a larger descendant selector.
     * @param locator The CSS parser locator. It is used for back-mapping. Can be <code>null</code>.
     */
    public DescendantSelectorImpl(Selector ancestorSelector, SimpleSelector simpleSelector, boolean isSubject, Locator locator) {
        super(locator);
        this.ancestorSelector = ancestorSelector;
      	this.simpleSelector = simpleSelector;
        this.isSubject = isSubject;
        
        this.hasSubject = isSubject || 
            ((ancestorSelector instanceof SubjectSelector) && 
              ((SubjectSelector)ancestorSelector).hasSubject());
    }
    
        
    /**
     * Returns the parent selector.
     */    
    @Override
    public Selector getAncestorSelector() {
        return ancestorSelector;
    }

    /*
     * Returns the simple selector.
     */    
    @Override
    public SimpleSelector getSimpleSelector() {
        return simpleSelector;
    }
    
    /**
     * @return <code>true</code> if the simple selector is the subject in a larger descendant selector.
     */
    @Override
    public boolean isSubject() {
      return isSubject;
    }

    /**
     * Checks if the selector has a subject.
     * 
     * @return <code>true</code> if this selector simple selector is the 
     * subject, or there is a subject in the ancestor selector.
     */
    @Override
    public boolean hasSubject() {
      return hasSubject;
    }
    

    /**
     * Gets the selector from the subject part upwards.
     *  
     * @return The ancestor selector for the subject.
     */
    @Override
    public Selector getSubjectAncestorSelector() {
      if (hasSubject()){ 
        SubjectSelector desc = this;
        while (!desc.isSubject()) {
          // Go upward into the selector until the subject is found.
          desc = (SubjectSelector) ((DescendantSelector)desc).getAncestorSelector();
        }
        return desc;
      } else {
        return null;
      }
    }

    /**
     * Gets the depth of the selector.
     * 
     * @return The depth of the selector, in node levels. The level, for the direct 
     * children 1, for grandsons 2, etc..
     */
    @Override
    public int getDepth() {
      return getDepth(this);
    }
    
    /**
     * Recursively compute the depth of a selector.
     * 
     * @param ds The descendant selector.
     * @return The depth of the selector, in node levels. The level, for the direct 
     * children 1, for grandsons 2, etc..
     */
    private int getDepth(DescendantSelector ds){
      int d = 1;

      if (ds != null){
        Selector ancestorSelector = ds.getAncestorSelector();
        if (ancestorSelector instanceof DescendantSelector){
          d = getDepth((DescendantSelector) ancestorSelector) + 1;  
        }
      }
      
      return d;
    }

    /**
     * @see org.w3c.flute.css.sac.SubjectSelector#setIsSubject(boolean)
     */
    @Override
    public void setIsSubject(boolean isSubject) {
      this.isSubject = isSubject;
    }

    /**
     * @see org.w3c.flute.css.sac.Selector#getSerialization(Map)
     */
    @Override
    public String getSerialization(Map<String, String> proxyNamespaceMapping) {
      return getSerialization(" ", proxyNamespaceMapping);
    }
    
   /**
    * Gets a serialization of this selector.
    * 
    * @param descSep The separator between the ancestor and the simple selector.
    * @see org.w3c.flute.css.sac.Selector#getSerialization(Map)
    */
    protected String getSerialization(String descSep, Map<String, String> proxyNamespaceMapping) {
      return ancestorSelector.getSerialization(proxyNamespaceMapping) + descSep + simpleSelector.getSerialization(proxyNamespaceMapping)  + (isSubject?"!":"");
    }

}
