/*
 * Copyright (c) 2000 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 *
 * $Id$
 */
package org.w3c.flute.parser.selectors;

import java.util.Map;
import java.util.Set;

import org.w3c.css.sac.Locator;
import org.w3c.flute.css.sac.ElementSelector;
import org.w3c.flute.css.sac.Selector;

/**
 * @version $Revision$
 * @author  Philippe Le Hegaret
 */
public class ElementSelectorImpl extends AbstractLocalizableSelector implements ElementSelector {

  /**
   * The element local name.
   */
  private String localName;

  /**
   * The namespace of the URI.
   */
  private String nsURI;

  /**
   * A flag showing if this selector is the subject of a CSS level 4 subject selector.
   */
  private boolean isSubject;

  /**
   * Creates a new ElementSelectorImpl
   * @param nsURI  The namespace of the URI.
   * @param localName The element local name.
   * @param locator The CSS parser locator. It is used for back-mapping. Can be <code>null</code>.
   */
  public ElementSelectorImpl(String nsURI, String localName, Locator locator) {
    super(locator);
    this.nsURI = nsURI;
    this.localName = localName;
  }

  /**
   * An integer indicating the type of <code>Selector</code>
   */
  @Override
  public short getSelectorType() {
    return Selector.SAC_ELEMENT_NODE_SELECTOR;
  }

  /**
   * Returns the
   * <a href="http://www.w3.org/TR/REC-xml-names/#dt-NSName">namespace
   * URI</a> of this element selector.
   * <p><code>NULL</code> if this element selector can match any namespace.</p>
   */
  @Override
  public String getNamespaceURI() {
    return nsURI;
  }

  /**
   * Returns the
   * <a href="http://www.w3.org/TR/REC-xml-names/#NT-LocalPart">local part</a>
   * of the
   * <a href="http://www.w3.org/TR/REC-xml-names/#ns-qualnames">qualified
   * name</a> of this element.
   * <p><code>NULL</code> if this element selector can match any element.</p>
   */
  @Override
  public String getLocalName() {
    return localName;
  }

  /**
   * Sets the subject flag.
   * 
   * @param isSubject <code>true</code> if this selector is the subject of a larger selector.
   */
  @Override
  public void setIsSubject(boolean isSubject) {
    this.isSubject = isSubject;
  }

  /**
   * Gets the subject flag.
   * 
   * @return Returns <code>true</code> if this selector is the subject of a larger selector.
   */
  @Override
  public boolean isSubject() {
    return isSubject;
  }

  /**
   * @see org.w3c.flute.css.sac.SubjectSelector#hasSubject()
   */
  @Override
  public boolean hasSubject() {
    return isSubject;
  }

  /**
   * @see org.w3c.flute.css.sac.SubjectSelector#getSubjectAncestorSelector()
   */
  @Override
  public Selector getSubjectAncestorSelector() {
    return this;
  }
  /**
   * @see org.w3c.flute.css.sac.Selector#getSerialization(Map)
   */
  @Override
  public String getSerialization(Map<String, String> proxyNamespaceMapping) { 
    StringBuilder sb = new StringBuilder();
    
    if (localName != null){
      if (nsURI != null && !org.w3c.flute.parser.ParserUtil.ANY_NAMESPACE.equals(nsURI)) {
        
        String prefix = getPrefix(proxyNamespaceMapping);
        if (prefix != null){
          sb.append(prefix);
        } else {
          sb.append(nsURI);
        }
        sb.append("|");
      }
      sb.append(localName);
    } else {
      sb.append('*');
    }
    
    if (isSubject) {
      sb.append('!');
    }
    
    return sb.toString();
  }

  /**
   * Gets the prefix for the current namespace URI.
   * 
   * @param proxyNamespaceMapping The prefix to namespace mapping. 
   * @return The prefix, or null if not found.
   */
  private String getPrefix(Map<String, String> proxyNamespaceMapping) {
    String prefix = null;
    if(proxyNamespaceMapping != null){
        Set<Map.Entry<String, String>> entrySet = proxyNamespaceMapping.entrySet();
        for (Map.Entry<String, String> entry : entrySet) {
          if (nsURI.equals(entry.getValue())){
            prefix = entry.getKey();
          }          
        }
      }

    return prefix;
  }
  
  
}
