/*
 * Copyright (c) 1999 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 *
 * $Id: ConditionFactory.java,v 1.2 2000/02/15 02:07:34 plehegar Exp $
 */
package org.w3c.flute.css.sac;

import org.w3c.css.sac.CSSException;

/**
 * @version $Revision: 1.2 $
 * @author  Philippe Le Hegaret
 */
public interface ConditionFactory {

  /**
   * Creates an and condition
   *
   * @param first the first condition
   * @param second the second condition
   * @return A combinator condition
   * @exception CSSException if this exception is not supported.
   */
  CombinatorCondition createAndCondition(Condition first, Condition second) throws CSSException;

  /**
   * Creates an or condition
   *
   * @param first the first condition
   * @param second the second condition
   * @return A combinator condition
   * @exception CSSException if this exception is not supported.
   */
  CombinatorCondition createOrCondition(Condition first, Condition second) throws CSSException;

  /**
   * Creates a negative condition
   *
   * @param condition the condition
   * @return A negative condition
   * @exception CSSException if this exception is not supported.
   */
  NegativeCondition createNegativeCondition(Condition condition) throws CSSException;

  
  /**
   * Creates an attribute condition
   *
   * @param localName the localName of the attribute
   * @param namespaceURI the namespace URI of the attribute
   * @param specified <code>true</code> if the attribute must be specified
   *                  in the document.
   * @param value the value of this attribute.
   * @return An attribute condition
   * @exception CSSException if this exception is not supported.
   */
  AttributeCondition createAttributeCondition(String localName, String namespaceURI,
      boolean specified, String value) throws CSSException;

  /**
   * Creates an id condition
   *
   * @param value the value of the id.
   * @return An Id condition
   * @exception CSSException if this exception is not supported.
   */
  AttributeCondition createIdCondition(String value) throws CSSException;

  /**
   * Creates a language condition
   *
   * @param lang the code of the value of the language.
   * @return A language condition
   * @exception CSSException if this exception is not supported.
   */
  LangCondition createLangCondition(String lang) throws CSSException;

  /**
   * Creates a "one of" attribute condition
   *
   * @param localName the localName of the attribute
   * @param namespaceURI the namespace URI of the attribute
   * @param specified <code>true</code> if the attribute must be specified
   *                  in the document.
   * @param value the value of this attribute.
   * @return A "one of" attribute condition
   * @exception CSSException if this exception is not supported.
   */
  AttributeCondition createOneOfAttributeCondition(String localName, String namespaceURI,
      boolean specified, String value) throws CSSException;

  /**
   * Creates a "begin hyphen" attribute condition
   *
   * @param localName the localName of the attribute
   * @param namespaceURI the namespace URI of the attribute
   * @param specified <code>true</code> if the attribute must be specified
   *                  in the document.
   * @param value the value of this attribute.
   * @return A "begin hyphen" attribute condition
   * @exception CSSException if this exception is not supported.
   */
  AttributeCondition createBeginHyphenAttributeCondition(String localName, String namespaceURI,
      boolean specified, String value) throws CSSException;

  /**
   * Creates a class condition
   *
   * @param namespaceURI the namespace URI of the attribute
   * @param value the name of the class.
   * @return A class condition
   * @exception CSSException if this exception is not supported.
   */
  AttributeCondition createClassCondition(String namespaceURI, String value) throws CSSException;

  /**
   * Creates a pseudo class condition
   *
   * @param namespaceURI the namespace URI of the attribute
   * @param value the name of the pseudo class
   * @param pseudoLevel The level of the pseudo element. This is <code>1</code> for the 
   * simple :before and :after and the <code>N</code> value from :before(N), :after(N).
   *  
   * @return A pseudo class condition
   * @exception CSSException if this exception is not supported.
   */
  AttributeCondition createPseudoClassCondition(String namespaceURI, String value, int pseudoLevel)
      throws CSSException;

  /**
   * Creates a "only one" child condition
   *
   * @return A "only one" child condition
   * @exception CSSException if this exception is not supported.
   */
  Condition createOnlyChildCondition() throws CSSException;

  /**
   * Creates a "only one" type condition
   *
   * @return A "only one" type condition
   * @exception CSSException if this exception is not supported.
   */
  Condition createOnlyTypeCondition() throws CSSException;

  /**
   * Creates a content condition
   *
   * @param data the data in the content
   * @return A content condition
   * @exception CSSException if this exception is not supported.
   */
  ContentCondition createContentCondition(String data) throws CSSException;

  
  //
  //
  // =========================== Oxygen extensions for CSS L3, L4 and Custom =============================
  //
  //
  
  
  /**
   * 
   * Create condition for:
   * 
   * E[foo*="bar"]  an E element whose "foo" attribute value contains the substring "bar" 
   * @param localName the localName of the attribute
   * @param namespaceURI the namespace URI of the attribute
   * @param specified <code>true</code> if the attribute must be specified
   *                  in the document.
   * @param value the value of this attribute.
   * @return A "begin hyphen" attribute condition
   * @exception CSSException if this exception is not supported.
   */
  AttributeCondition createContainsAttributeCondition(String localName, String namespaceURI,
      boolean specified, String value) throws CSSException;

  /**
   * Create condition for:
   * 
   * E[foo$="bar"]  an E element whose "foo" attribute value ends exactly with the string "bar"   Attribute selectors   3
   * 
   * @param localName the localName of the attribute
   * @param namespaceURI the namespace URI of the attribute
   * @param specified <code>true</code> if the attribute must be specified
   *                  in the document.
   * @param value the value of this attribute.
   * @return A "begin hyphen" attribute condition
   * @exception CSSException if this exception is not supported.
   */
  AttributeCondition createEndWithAttributeCondition(String localName, String namespaceURI,
      boolean specified, String value) throws CSSException;

  /**
   * Creates a "begin with" attribute condition:
   * E[foo^="bar"]  an E element whose "foo" attribute value begins exactly with the string "bar"   Attribute selectors   3
   *
   * @param localName the localName of the attribute
   * @param namespaceURI the namespace URI of the attribute
   * @param specified <code>true</code> if the attribute must be specified
   *                  in the document.
   * @param value the value of this attribute.
   * @return A "begin hyphen" attribute condition
   * @exception CSSException if this exception is not supported.
   */
  AttributeCondition createBeginWithAttributeCondition(String localName, String namespaceURI,
      boolean specified, String value) throws CSSException;

  /**
   * Creates a nth-child condition. The parameters may be one of the constants: "odd", "even", 
   * or an expression like "1", "+3", or "5n+2". For the last one, the first number represents 
   * the cycle, the second is the delta.
   * 
   * For more information, please read here: http://www.w3.org/TR/css3-selectors/#nth-child-pseudo
   * 
   * @param pseudoFunction The name of the pseudo-function, like "nth-child".
   * @param cycle The cycle after which the condition is triggered again. Can be zero for no cycle.
   * @param delta The displacement in the cycle after which the condition is triggered.
   * @return A nth-child type condition.
   * @throws CSSException When the structural condition could not be built. 
   */
  Condition createStructuralCondition(String pseudoFunction, int cycle, int delta) throws CSSException;

  /**
   * Creates a negative condition, over a selector. 
   * Dan: In CSS3 there is support to negate a simple selector, not just a condition.
   *
   * @param selector the condition
   * @return A negative condition
   * @exception CSSException if this exception is not supported.
   */
  NegativeCondition createNegativeCondition(Selector selector) throws CSSException;

  /**
   * Creates a :has condition, over a selector. 
   *
   * @param selector the relative descendant selector
   * @return A :has condition
   */  
  HasCondition createHasCondition(Selector selector);
}
