/*
 * Copyright (c) 2000 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 *
 * $Id$
 */
package org.w3c.flute.parser.selectors;

import java.util.Map;

import org.w3c.flute.css.sac.AttributeCondition;
import org.w3c.flute.css.sac.Condition;

/**
 * Implementation of a pseudoclass condition.
 * 
 * @version $Revision$
 * @author  Philippe Le Hegaret. Dan
 */
public class PseudoClassConditionImpl implements AttributeCondition {

  /**
   * The pseudo class name.
   */
  private String pseudoClassName;
  
  /**
   * The pseudo element level. This is the N value from the :before(N) constructs. 
   * A simple :before has the implicit N value equal to 1.
   */
  private int pseudoLevel;

  /**
   * Creates a new pseudo class condition.
   * 
   * @param pseudoClassName The pseudo class name.
   * 
   * @param pseudoLevel The level of the pseudo element. 
   *  <p>Example:</p> 
   *  <pre>
   *    elem:before{
   *      content: "AA";
   *    }
   *    elem:before(2){
   *      content: "BB";
   *    }
   *    elem:before(5){
   *      content: "CC";
   *    }
   *  </pre>
   *  This is rendered on the screen as the string "CCBBAA", The ':before' selector is exactly 
   *  equivalent to ':before(1)'.
   *  <p>
   *  The 2 and 5 are the <code>pseudoLevel</code> parameters.
   *     
   */
  public PseudoClassConditionImpl(String pseudoClassName, int pseudoLevel) {
    this.pseudoClassName = pseudoClassName;
    this.pseudoLevel = pseudoLevel;
  }
  
  /**
   * Creates a new pseudo class condition with an nth of 1.
   * 
   * @param pseudoClassName The pseudo class name.
   */
  public PseudoClassConditionImpl(String pseudoClassName) {
    this(pseudoClassName, 1);
  }


  /**
   * Get the pseudo level. This is the N value from the :before(N) constructs. 
   * A simple :before has the implicit N value equal to 1.
   * 
   * @return Returns the pseudo level value.
   */
  public int getPseudoLevel() {
    return pseudoLevel;
  }
  
  /**
   * An integer indicating the type of <code>Condition</code>.
   */
  @Override
  public short getConditionType() {
    return Condition.SAC_PSEUDO_CLASS_CONDITION;
  }

  /**
   * Returns the
   * <a href="http://www.w3.org/TR/REC-xml-names/#dt-NSName">namespace
   * URI</a> of this attribute condition.
   * <p><code>NULL</code> if :
   * <ul>
   * <li>this attribute condition can match any namespace.
   * <li>this attribute is an id attribute.
   * </ul>
   */
  @Override
  public String getNamespaceURI() {
    return null;
  }

  /**
   * Returns the
   * <a href="http://www.w3.org/TR/REC-xml-names/#NT-LocalPart">local part</a>
   * of the
   * <a href="http://www.w3.org/TR/REC-xml-names/#ns-qualnames">qualified
   * name</a> of this attribute.
   * <p><code>NULL</code> if :
   * <ul>
   * <li><p>this attribute condition can match any attribute.
   * <li><p>this attribute is a class attribute.
   * <li><p>this attribute is an id attribute.
   * <li><p>this attribute is a pseudo-class attribute.
   * </ul>
   */
  @Override
  public String getLocalName() {
    return null;
  }

  /**
   * Returns <code>true</code> if the attribute must have an explicit value
   * in the original document, <code>false</code> otherwise.
   */
  @Override
  public boolean getSpecified() {
    return true;
  }

  /**
   * Returns the value of the attribute.
   * If this attribute is a class or a pseudo class attribute, you'll get
   * the class name (or psedo class name) without the '.' or ':'.
   */
  @Override
  public String getValue() {
    return pseudoClassName;
  }
  
  /**
   * @see org.w3c.flute.css.sac.Condition#getSerialization(Map)
   */
  @Override
  public String getSerialization(Map<String, String> proxyNamespaceMapping) {
    return ":" + pseudoClassName + (pseudoLevel != 1? ("(" + pseudoLevel +")"):"");
  }
}
