<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" 
      xmlns:xs="http://www.w3.org/2001/XMLSchema"
      xmlns:local="urn:local-functions"
      xmlns:rsiwp="http://reallysi.com/namespaces/generic-wordprocessing-xml"
      xmlns:stylemap="urn:public:dita4publishers.org:namespaces:word2dita:style2tagmap"
      xmlns:relpath="http://dita2indesign/functions/relpath"
      xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
      xmlns:m="http://www.w3.org/1998/Math/MathML"
      
      exclude-result-prefixes="xs rsiwp stylemap local relpath xsi"
      version="2.0">

  <!--==========================================
    Simple Word Processing Markup to DITA generic transformation
    
    Copyright (c) 2009, 2016 DITA For Publishers, Inc.

    Transforms a simple word processing document into DITA maps and
    topics using a style-to-tag mapping.
    
    This transform is intended to be the base for more specialized
    transforms that provide style-specific overrides.
    
    The input to this transform is a simple ML document created
    by transforming some proprietary word processing or DTP
    format, such as DOCX or IDML.
    
    This transform is intended to be used from a format-specific shell
    that generates the simple ML instance and then applies
    this transform. The shell should supply the root template.
       
    This transform operates in two phases:

    Phase 1: Generate map and topic markup as a single physical document 
    
      Phase 1 constructs all the map and topic elements as a single physical
    document because XSLT 2 does not allow generation of a new result document
    while you are already producing a result document. Each eventual result
    document is indicated by a rsiwp:result-document element, which
    provides the details needed to construct the final result document.
    
      This document reflects any map and topic hierarchy implied by the
    paragraphs.
    
    Phase 2: Generate result documents
    
    Phase 2 operates on the output of the phase 2 to generate each 
    result document, using the rsiwp:result-document elements.
    
    Each result document is processed in the mode "final-fixup" which
    allows extension code to adjust the DITA markup produced by
    the style-to-tag map, for example, to add missing containers,
    reorder things, or otherwise correct it as needed.
    
    =========================================== -->

<!-- 
  The root importer of this module must also import the following module:
  
  
  <xsl:import href="../../org.dita4publishers.common.xslt/xsl/lib/relpath_util.xsl"/>
 -->  
  
  <!-- Paragraphs indexed by the first child bookmark that doesn't have a name starting with "_" 
       Names starting with "_" appear to be autogenerated for TOCs and whatnot.
       Not sure this logic is foolproof. May just have to look up paragraphs directly without
       using a key table.
       
       Adding a prefix to the @name value to make it clearer that we're talking about bookmark
       labels, which in Word may be just simple numbers. The Wordml2simple process does the same
       thing.
    -->
  
  <xsl:key name="bookmarksByName" match="rsiwp:bookmarkStart" use="@name"/>
  
  <xsl:template match="rsiwp:document">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:message> + [INFO] ++++++++++++++++++++++++++++++ 
 + [INFO] 
 + [INFO] simple2dita: Processing intermediate simpleML XML to generate DITA XML...
 + [INFO]    
 + [INFO] ++++++++++++++++++++++++++++++</xsl:message>
    <!-- Capture the input document so we can pass it around into contexts where the original
         document context gets lost.
      -->
    <xsl:variable name="simpleWpDoc" as="document-node()" select="root(.)"/>
    
    <xsl:message> + [INFO] simple2dita: Phase 1: Generate DITA maps and topics as a single XML document....</xsl:message>
    <xsl:variable name="resultDocs" as="document-node()">
      <xsl:document>
        <xsl:apply-templates>
          <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
          <xsl:with-param name="simpleWpDoc" as="document-node()" tunnel="yes" select="$simpleWpDoc"/>
        </xsl:apply-templates>
      </xsl:document>
    </xsl:variable>
    
    <!-- Write out the base resultDocs data for debugging purposes -->
    <xsl:if test="$saveIntermediateDocs">
      <xsl:variable
        name="tempDoc"
        select="relpath:newFile($outputDir, 'resultDocs.xml')"
        as="xs:string"/>
      <xsl:message> + [DEBUG] saving $resultDocs to 
<xsl:sequence select="$tempDoc"/> 
</xsl:message>
      <xsl:result-document href="{$tempDoc}" format="indented">
        <xsl:sequence select="$resultDocs"/>
      </xsl:result-document>
    </xsl:if>
    <!-- Fixup xrefs in the result documents: -->
    <xsl:message> + [INFO] simple2dita: Fixing up xrefs in the result documents...</xsl:message>
    <xsl:variable name="resultDocsFixedUp" as="node()*">
      <xsl:apply-templates select="$resultDocs" mode="resultdocs-xref-fixup">
        <xsl:with-param name="simpleWpDoc" as="document-node()" tunnel="yes"
          select="$simpleWpDoc"
        />                  
      </xsl:apply-templates>
    </xsl:variable>
    <!-- Write out the fixed-up resultDocs data for debugging purposes -->
    <xsl:if test="$saveIntermediateDocs">
      <xsl:variable
        name="tempDoc"
        select="relpath:newFile($outputDir, 'resultDocsFixedUp.xml')"
        as="xs:string"/>
      <xsl:message> + [DEBUG] saving $resultDocsFixedUp to 
        <xsl:sequence select="$tempDoc"/>        
      </xsl:message>
      <xsl:result-document href="{$tempDoc}">
        <xsl:sequence select="$resultDocsFixedUp"/>
      </xsl:result-document>
    </xsl:if>
    <!-- Generate the result documents: -->
    <xsl:message> + [INFO] simple2dita: Phase 2: Generate the result map and topic documents...</xsl:message>
    <xsl:apply-templates select="$resultDocsFixedUp" mode="generate-result-docs">
      <xsl:with-param name="simpleWpDoc" as="document-node()" tunnel="yes"
        select="$simpleWpDoc"
      />                  
    </xsl:apply-templates>
    <xsl:message> + [INFO] simple2dita: Done processing simpleML document.</xsl:message>
    
  </xsl:template>
  
  <xsl:template match="rsiwp:body">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] rwiwp:body</xsl:message>
    </xsl:if>    
    
    
    <xsl:variable name="newMapUrl" as="xs:string"
      select="relpath:newFile($outputDir, $rootMapUrl)"
    />
    <xsl:variable name="newTopicUrl" as="xs:string"
      select="relpath:newFile($outputDir, $rootTopicUrl)"
    />

    <xsl:apply-templates>
      <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      <xsl:with-param name="parentMapUrl" as="xs:string" tunnel="yes" 
        select="''"/>
      <xsl:with-param name="newMapUrl" as="xs:string"  select="$newMapUrl"/>
      <xsl:with-param name="newTopicUrl" as="xs:string"  select="$newTopicUrl"/>
    </xsl:apply-templates>
  </xsl:template>
    
  <xsl:template match="rsiwp:body/rsiwp:topic">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="parentMapUrl" as="xs:string" tunnel="yes" select="''"/>
    <xsl:param name="newTopicUrl" as="xs:string" />
    
<!--    <xsl:variable name="doDebug" as="xs:boolean" select="true()"/>-->
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] rwiwp:body/rsiwp:topic</xsl:message>
    </xsl:if>
    <xsl:call-template name="makeTopic">
      <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      <xsl:with-param name="parentMapUrl" as="xs:string?" tunnel="yes" select="$parentMapUrl"/>
      <xsl:with-param name="topicUrl" as="xs:string" select="$newTopicUrl"/>    
      <xsl:with-param name="topicName" as="xs:string" select="$finalRootTopicName" tunnel="yes"/>
      <xsl:with-param name="simpleWpDoc" as="document-node()" tunnel="yes"
        select="root(.)"
       />
    </xsl:call-template>
  </xsl:template>
  
  <xsl:template match="rsiwp:map">
    <!-- generate a map document -->
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="parentMapUrl" as="xs:string" tunnel="yes" select="''"/>
    <xsl:param name="newMapUrl" as="xs:string"/><!-- Absolute path for the new map -->
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] rsiwp:map: parentMapUrl="<xsl:value-of select="$parentMapUrl"/>"</xsl:message>
      <xsl:message> + [DEBUG] rsiwp:map:    newMapUrl="<xsl:value-of select="$newMapUrl"/>"</xsl:message>
    </xsl:if>
    
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] rsiwp:map: newMapUrl="<xsl:value-of select="$newMapUrl"/>"</xsl:message>
    </xsl:if>
    
    <xsl:variable name="resultUrl" as="xs:string"
      select="$newMapUrl"
    />
    
    <xsl:message> + [INFO] makeMap: Creating new map document "<xsl:sequence select="$resultUrl"/>"...</xsl:message>
    
    <xsl:variable name="formatName" select="@format" as="xs:string?"/>
    
    <xsl:variable name="format" select="key('formats', $formatName, $styleMapDoc)[1]" as="element()?"/>
    <xsl:if test="not($format)">
      <xsl:message terminate="yes"> + [ERROR] map: Failed to find &lt;output&gt; element for @mapFormat value "<xsl:value-of select="$formatName"/>" specified for style "<xsl:value-of select="@styleName"/>" <xsl:value-of select="concat(' [', @styleId, ']')"/>. Check your style-to-tag mapping.</xsl:message>
    </xsl:if>
    
    <xsl:variable name="mapElement" as="node()*">
      <xsl:call-template name="constructMap">
        <xsl:with-param name="format" as="element()" select="$format"/>
        <xsl:with-param name="mapUrl" as="xs:string" select="$newMapUrl"/>
      </xsl:call-template>
    </xsl:variable>
    
    
    <rsiwp:result-document href="{$resultUrl}"
      doctype-public="{$format/@doctype-public}"
      doctype-system="{$format/@doctype-system}"
      indent="yes"
      >
      <xsl:message> + [INFO] Applying final-fixup mode to <xsl:sequence select="$resultUrl"/>...</xsl:message>
      <xsl:apply-templates select="$mapElement" mode="final-fixup">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates>      
    </rsiwp:result-document>
    
    <xsl:if test="$doDebug">        
      <xsl:message> + [DEBUG] rsiwp:map: Done.</xsl:message>
    </xsl:if>
  </xsl:template>
  
  <xsl:template name="constructMap">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="format" as="element()"/>
    <xsl:param name="mapUrl" as="xs:string"/>
    
    <xsl:variable name="schemaAtts" as="attribute()*">
      <xsl:if test="$format/@noNamespaceSchemaLocation">
        <xsl:attribute name="xsi:noNamespaceSchemaLocation"
          select="string($format/@noNamespaceSchemaLocation)"
        />
      </xsl:if>
      <xsl:if test="$format/@schemaLocation != ''">
        <xsl:attribute name="xsi:schemaLocation"
          select="string($format/@schemaLocation)"
        />
      </xsl:if>      
    </xsl:variable>
    
    <xsl:variable name="prologType" as="xs:string"
      select="
      if (@prologType and @prologType != '')
      then @prologType
      else 'topicmeta'
      "
    />
    
    <xsl:element name="{@mapType}">
      <xsl:sequence select="$schemaAtts"/>
      <xsl:call-template name="generateXtrcAtt"/>  
      <xsl:attribute name="xml:lang" 
        select="if (@langAttValue) 
        then string(@langAttValue) 
        else $language"
      />
      <xsl:attribute name="isMap" select="'true'"/>
      
      <!-- The first paragraph can simply trigger a (possibly) untitled map, or
          it can also be the map title. If it's the map title, generate it.
          First paragraph can also generate a root topicref and/or a topicref
          to a topic in addition to the map.
          
          Note: For paragraphs that generate both maps and topics, it doesn't matter
          if the structureType or secondStructureType is topicTitle, map, or mapTitle: the
          result will be the same, a new map and a new topic.
        -->
      <xsl:apply-templates select="rsiwp:maptitle">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates>
      
      <!-- Gather any paragraphs that contribute to the root map's map-level metadata: -->
      <xsl:if test="*[string(@topicZone) = 'topicmeta' and string(@containingTopic) = 'rootmap']">
        <xsl:variable name="prologParas" select="*[string(@topicZone) = 'topicmeta' and string(@containingTopic) = 'rootmap']" as="node()*"/>
        <!-- Now process any map-level topic metadata paragraphs. -->
        <xsl:element name="{$prologType}">
          <xsl:call-template name="generateXtrcAtt"/>
          <xsl:call-template name="handleTopicProlog">
            <xsl:with-param name="content" select="$prologParas"/>
          </xsl:call-template>
        </xsl:element>
      </xsl:if>
      <!--<xsl:variable name="doDebug" as="xs:boolean" select="true()"/>-->
      <xsl:if test="$doDebug">
        <xsl:message> + [DEBUG] Applying templates to map element children...</xsl:message>
      </xsl:if>
      <xsl:apply-templates 
        select="
        rsiwp:mapref | 
        rsiwp:topicref | 
        rsiwp:topicGroup | 
        rsiwp:topicHead | 
        rsiwp:map | 
        rsiwp:topic
        ">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
        <xsl:with-param name="parentMapUrl" as="xs:string" select="$mapUrl" tunnel="yes"/>
      </xsl:apply-templates>
    </xsl:element>
    
  </xsl:template>
  
  <xsl:template match="rsiwp:maptitle">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <!-- Generate the map title -->
    <xsl:choose>
      <xsl:when test="@containerType">
        <xsl:element name="{@containerType}">
          <xsl:apply-templates>
            <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
          </xsl:apply-templates>
        </xsl:element>
      </xsl:when>
      <xsl:otherwise>
        <xsl:apply-templates>
          <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
        </xsl:apply-templates>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  
  <xsl:template match="rsiwp:navtitle">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:variable name="topicmetaType" as="xs:string"
      select="if (@topicrefTopicmetaType) then @topicrefTopicmetaType else 'topicmeta'"
    />
    <xsl:if test="$generateNavtitlesBoolean">
      <xsl:element name="{$topicmetaType}">
        <xsl:call-template name="generateXtrcAtt">
          <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
        </xsl:call-template>
        <xsl:apply-templates mode="navtitle">
          <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
        </xsl:apply-templates>
        
        <xsl:call-template name="handleTopicrefMetadata">
          <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
        </xsl:call-template>
      </xsl:element>
    </xsl:if>
  </xsl:template>
  
  <xsl:template match="rsiwp:topicHead/rsiwp:p | 
                       rsiwp:topicref/rsiwp:p |
                       rsiwp:topicGroup/rsiwp:p
                       " priority="10">
    <!-- Suppressed in default mode -->
  </xsl:template>
  
  <xsl:template name="handleTopicrefMetadata">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <!-- Context is an rsiwp:topicHead, topicref, or topicGroup element -->
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] handleTopicrefMetadata: Handling metadata...</xsl:message>
    </xsl:if>
    <xsl:variable name="topicrefMetadata" as="element()*"
      select="following-sibling::rsiwp:p[@containingTopic = 'currenttopicref' and 
                                         @topicZone = 'topicmeta'] |
              ../rsiwp:topic/rsiwp:p[@containingTopic = 'currenttopicref' and 
                                         @topicZone = 'topicmeta']"
    />
    <xsl:if test="$doDebug">

      <xsl:message> + [DEBUG] handleTopicrefMetadata: topicrefMetadata=<xsl:sequence select="local:reportParas($topicrefMetadata)"/></xsl:message>
    </xsl:if>
    <metadata><!-- FIXME: Should parameterize this but not bothering to for now. -->
      <xsl:for-each select="$topicrefMetadata">
        <xsl:call-template name="transformPara"/>
      </xsl:for-each>   
    </metadata>
  </xsl:template>
  
  <xsl:template mode="navtitle" match="rsiwp:*">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:variable name="navtitleType" as="xs:string" 
      select="if (@navtitleType) then @navtitleType else 'navtitle'"/>
    <xsl:element name="{$navtitleType}">
      <xsl:call-template name="generateXtrcAtt">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:call-template>
      <xsl:call-template name="transformParaContent">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:call-template>    
    </xsl:element>
  </xsl:template>

  <xsl:template match="rsiwp:mapref">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="parentMapUrl" as="xs:string" tunnel="yes"/>

    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] rsiwp:mapref: parentMapUrl="<xsl:value-of select="$parentMapUrl"/></xsl:message>
    </xsl:if>
    <xsl:variable name="newMapUrl" as="xs:string"
      select="local:getResultUrlForMap(rsiwp:map, $parentMapUrl)"
      />
    <xsl:variable name="newMapRelativeUrl" as="xs:string"
      select="relpath:getRelativePath(relpath:getParent($parentMapUrl), $newMapUrl)"
    />
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] rsiwp:mapref:         newMapUrl="<xsl:value-of select="$newMapUrl"/></xsl:message>
      <xsl:message> + [DEBUG] rsiwp:mapref: newMapRelativeUrl="<xsl:value-of select="$newMapRelativeUrl"/></xsl:message>
    </xsl:if>
    <xsl:variable name="tagname" as="xs:string"
      select="if (@maprefType) then @maprefType else 'mapref'"
    />
    <xsl:element name="{$tagname}">
      <xsl:attribute name="href" select="$newMapRelativeUrl"/>
      <xsl:call-template name="generateXtrcAtt"/>
      <xsl:if test="@maprefFormat">
        <xsl:attribute name="format" select="@maprefFormat"/>
      </xsl:if>
      <xsl:apply-templates><!-- The only child of rsiwp:mapref should be rsiwp:map -->
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
        <xsl:with-param name="newMapUrl" as="xs:string"  select="$newMapUrl"/>
      </xsl:apply-templates>
    </xsl:element>
  </xsl:template>

  <xsl:template match="rsiwp:topicref[rsiwp:topic]" priority="10">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="parentMapUrl" as="xs:string" tunnel="yes"/>
    
<!--    <xsl:variable name="doDebug" as="xs:boolean" select="true()"/>-->

    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] rsiwp:topicref: parentMapUrl="<xsl:value-of select="$parentMapUrl"/>"</xsl:message>
    </xsl:if>
    
    <xsl:variable name="topicrefType" as="xs:string"
      select="if (@topicrefType) then @topicrefType else 'topicref'"
    />
    <xsl:variable name="topicName" as="xs:string?">
      <xsl:apply-templates mode="topic-name" select="rsiwp:topic">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates>
    </xsl:variable>

    <xsl:variable name="topicUrl" as="xs:string"
      select="local:getResultUrlForTopic(rsiwp:topic, $parentMapUrl, $topicrefType, $topicName)"
    />
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] rsiwp:topicref: topicUrl="<xsl:value-of select="$topicUrl"/>"</xsl:message>
    </xsl:if>
    <xsl:variable name="topicRelativeUrl" as="xs:string"
      select="relpath:getRelativePath(relpath:getParent($parentMapUrl), $topicUrl)"
    />
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] rsiwp:topicref: topicRelativeUrl="<xsl:value-of select="$topicRelativeUrl"/>"</xsl:message>
    </xsl:if>
    <xsl:element name="{$topicrefType}">
      <xsl:attribute name="href" select="$topicRelativeUrl"/>
      <xsl:call-template name="generateXtrcAtt"/>
      <xsl:sequence select="@chunk, @collection-type, @processing-role, @toc"/>
      <xsl:apply-templates>
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
        <xsl:with-param name="topicUrl" as="xs:string" select="$topicUrl" tunnel="yes"/>
        <xsl:with-param name="topicName" as="xs:string" tunnel="yes" select="$topicName"/>
      </xsl:apply-templates>
    </xsl:element>
  </xsl:template>

  <xsl:template match="rsiwp:topicHead | rsiwp:topicGroup | rsiwp:topicref">
    <!-- Includes topicrefs with no child topic elements. -->
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="parentMapUrl" as="xs:string" tunnel="yes"/>
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] rsiwp:topichead</xsl:message>
    </xsl:if>
    
    <xsl:variable name="topicrefType" as="xs:string"
      select="if (@topicrefType) then @topicrefType else 'topichead'"
    />
    
    <xsl:element name="{$topicrefType}">
      <xsl:call-template name="generateXtrcAtt"/>
      <xsl:sequence select="
        @chunk, 
        @collection-type, 
        @outputclass, 
        @print,
        @processing-role, 
        @toc 
        "/>
      <xsl:apply-templates>
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates>
    </xsl:element>
  </xsl:template>

  <xsl:template match="rsiwp:topic">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="parentMapUrl" as="xs:string" tunnel="yes"/>
    <xsl:param name="topicUrl" as="xs:string" tunnel="yes"/><!-- Result URL for the topic document or its parent topic -->
    <xsl:param name="topicName" as="xs:string" tunnel="yes"/><!-- File/ID for the topic -->
    
    <!-- FIXME: This really needs to go through a mode for generating IDs
                that will result in the correct ID but the current
                ID-generation framework doesn't provide for just generating
                ID values, only ID attributes.
                
                Or else need to rethink how makeTopic works with regard to
                topic ID generation.
      -->
    <xsl:variable name="topicName" as="xs:string"
      select="if ($topicName = '') then generate-id(.) else $topicName"
    />
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] rsiwp:topic:  topicUrl="<xsl:value-of select="$topicUrl"/></xsl:message>
      <xsl:message> + [DEBUG] rsiwp:topic: topicName="<xsl:value-of select="$topicName"/></xsl:message>
    </xsl:if>
    
    <xsl:call-template name="makeTopic">
      <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      <xsl:with-param name="topicUrl" as="xs:string?" select="$topicUrl"/>
      <xsl:with-param name="topicName" as="xs:string" select="$topicName"/>
      <xsl:with-param name="simpleWpDoc" as="document-node()" tunnel="yes"
        select="root(.)"
       />
    </xsl:call-template>

  </xsl:template>
  
  <xsl:template mode="generate-result-docs" match="rsiwp:result-document" priority="10">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:message> + [INFO] Generating result document "<xsl:sequence select="string(@href)"/>..."</xsl:message>
    <xsl:result-document href="{@href}" 
      doctype-public="{@doctype-public}"
      doctype-system="{@doctype-system}">
      <xsl:apply-templates select="./*" mode="generate-result-docs"/>
    </xsl:result-document>
  </xsl:template>
  
  <xsl:template mode="generate-result-docs" match="*" priority="5">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:copy copy-namespaces="no">
      <xsl:apply-templates select="@*,node()" mode="#current"/>
    </xsl:copy>
  </xsl:template>
  
  <xsl:template mode="generate-result-docs" match="@xtrc | @xtrf" priority="10">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:choose>
      <xsl:when test="$includeWordBackPointersBoolean">
        <xsl:sequence select="."/>
      </xsl:when>
      <xsl:otherwise>
        <!-- Don't copy the attribute -->
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  
  <xsl:template mode="generate-result-docs" match="@*|node()">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:copy/>
  </xsl:template>
  
  <xsl:template match="rsiwp:p[string(@structureType) = 'skip']" priority="10">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
  </xsl:template>
  
  <xsl:template match="rsiwp:p" name="transformPara">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:variable name="doDebug" as="xs:boolean" select="false()"/>
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] rsiwp:p (transformPara): text=<xsl:sequence select="substring(., 1, 40)"/></xsl:message>
    </xsl:if>
    <xsl:variable name="tagName" as="xs:string"
      select="
      if (@tagName) 
      then string(@tagName)
      else 'p'
      "
    />
    <xsl:if test="not(./@tagName)">
      <xsl:message> + [WARNING] No @tagName value for paragraph with style "<xsl:sequence select="string(@style)"/>". This should not happen.</xsl:message>
    </xsl:if>
    <xsl:choose>
      <xsl:when test="count(./* except (rsiwp:break)) = 0 and normalize-space(.) = ''">
        <xsl:if test="$doDebug">
          <xsl:message> + [DEBUG] Skipping apparently-empty paragraph: <xsl:sequence select="local:reportPara(.)"/></xsl:message>
        </xsl:if>
      </xsl:when>
      <xsl:otherwise>
        <xsl:variable name="idGenerator" as="xs:string">
          <xsl:choose>
            <xsl:when test="string(@idGenerator) = ''">
              <xsl:sequence select="''"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:sequence select="string(@idGenerator)"/>
            </xsl:otherwise>
          </xsl:choose>
          
        </xsl:variable>
        <xsl:element name="{$tagName}">  
          <xsl:call-template name="generateXtrcAtt"/>
          <xsl:attribute name="xtrf" select="ancestor::rsiwp:document[1]/@sourceDoc"/>
          <xsl:apply-templates select="rsiwp:bookmarkStart" mode="generate-para-ids">
            <xsl:with-param name="tagName" select="$tagName"/>
          </xsl:apply-templates>
          <xsl:sequence select="./@outputclass"/>
          <xsl:apply-templates select="@dataName, @typeAttValue"/>
          <xsl:apply-templates select="." mode="generate-id">
            <xsl:with-param name="idGenerator" select="$idGenerator" as="xs:string"/>
            <xsl:with-param name="tagName" select="$tagName" as="xs:string"/>
          </xsl:apply-templates>
          <xsl:if test="@langAttValue != ''">
            <xsl:attribute name="xml:lang" select="string(@langAttValue)"/>
          </xsl:if>                    
          <xsl:apply-templates select="stylemap:additionalAttributes" mode="additional-attributes">
            <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
          </xsl:apply-templates>
          <xsl:call-template name="transformParaContent"/>    
        </xsl:element>
      </xsl:otherwise>
    </xsl:choose>    
  </xsl:template>
  
  <xsl:template mode="#default" match="stylemap:additionalAttributes"/>
  
  
  <xsl:template mode="additional-attributes" match="stylemap:additionalAttributes">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] additional-attributes: stylemap:additionalAttributes, applying templates to stylemap:attribute elements...</xsl:message>
    </xsl:if>

    <xsl:apply-templates mode="#current" select="stylemap:attribute">
      <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
    </xsl:apply-templates>
  </xsl:template>
  
  <xsl:template mode="additional-attributes" match="stylemap:attribute">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>

    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] additional-attributes: stylemap:attribute: <xsl:sequence select="."/></xsl:message>
    </xsl:if>
    <xsl:attribute name="{@name}" select="@value"/>
  </xsl:template>
  
  <xsl:template match="@dataName">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:attribute name="name" select="."/>
  </xsl:template>
  
  <xsl:template match="@typeAttValue">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:attribute name="type" select="."/>
  </xsl:template>
  
  <xsl:template mode="generate-id" match="*">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="idGenerator" select="''"/>
    <xsl:choose>
      <xsl:when test="$idGenerator = ''">
        <!-- Don't generate an ID -->
      </xsl:when>
      <xsl:otherwise>
        <xsl:attribute name="id" select="generate-id(.)"/>
        <!-- This will be removed during ID fixup pass -->
        <xsl:attribute name="idGenerator" select="$idGenerator"/>
      </xsl:otherwise>
    </xsl:choose>
    
  </xsl:template>
  
  <xsl:template name="transformParaContent">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:variable name="isTitlePara" as="xs:boolean"
      select="local:isTopicTitle(.)"
    />
    <!-- Transforms the content of a paragraph, where the containing
         element is generated by the caller. -->
    <xsl:choose>
      <xsl:when test="string(@useContent) = 'elementsOnly'">
        <xsl:apply-templates mode="p-content" select="*">
          <xsl:with-param name="inTitleContext" as="xs:boolean" tunnel="yes"
            select="$isTitlePara"
          />
        </xsl:apply-templates>
      </xsl:when>
      <xsl:when test="string(@putValueIn) = 'valueAtt'">
        <xsl:attribute name="value" select="string(.)"/>
        <xsl:if test="@dataName">
          <xsl:attribute name="name" select="string(@dataName)"/>
        </xsl:if>
      </xsl:when>
      <xsl:otherwise>
        <xsl:apply-templates mode="p-content">
          <xsl:with-param name="inTitleContext" as="xs:boolean" tunnel="yes"
            select="$isTitlePara"
          />          
        </xsl:apply-templates>
      </xsl:otherwise>
    </xsl:choose>
    
  </xsl:template>
  
  <!-- Much of the code used for table transformation comes from the DITA Open Toolkit's h2d plugin -->
  <xsl:template match="rsiwp:table">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:variable name="width">
      <xsl:if test="@width">
        <xsl:value-of select="substring-before(@width,'%')"/>
      </xsl:if>
    </xsl:variable>
    <!--    <xsl:message> + [DEBUG] rsiwp:table: Starting...</xsl:message>-->
    <xsl:variable name="tagName" as="xs:string"
      select="
      if (@tagName) 
      then string(@tagName)
      else 'table'
      "
    />
    <xsl:element name="{$tagName}">  
      <xsl:call-template name="generateXtrcAtt"/>
      <xsl:sequence select="@align"/>
      <xsl:sequence select="@frame"/>
      <xsl:sequence select="@rowsep"/>
      <xsl:sequence select="@colsep"/>
      <xsl:sequence select="@outputclass"/>
      <xsl:choose>
        <xsl:when test="number($width) &lt; 100">
          <xsl:attribute name="pgwide" select="'0'"/>
        </xsl:when>
        <xsl:when test="string-length($width)">
          <xsl:attribute name="pgwide" select="'1'"/>
        </xsl:when>
      </xsl:choose>
      <xsl:if test="@langAttValue != ''">
        <xsl:attribute name="xml:lang" select="string(@langAttValue)"/>
      </xsl:if>
      
      <tgroup>
        <!-- add colspan data here -->
        <xsl:attribute name="cols" select="count(rsiwp:cols/rsiwp:col)"/>
        <xsl:call-template name="create-colspec"/>      
        <xsl:choose>
          <xsl:when test="rsiwp:thead">
            <thead>
              <xsl:apply-templates select="rsiwp:thead/rsiwp:tr"/>
            </thead>
          </xsl:when>
          <xsl:when test="rsiwp:tr[rsiwp:th and not(rsiwp:td)]"> <!-- FIXME: Will this ever select anything? I don't think we make rsiwp:th in wordml2simple -->
            <thead>
              <xsl:apply-templates select="rsiwp:tr[rsiwp:th and not(rsiwp:td)]"/>
            </thead>
          </xsl:when>
        </xsl:choose>
        <tbody>
          <xsl:choose>
            <xsl:when test="rsiwp:tbody/rsiwp:tr[rsiwp:td]|rsiwp:tr[rsiwp:td]">
              <xsl:apply-templates select="rsiwp:tbody/rsiwp:tr[rsiwp:td]|rsiwp:tr[rsiwp:td]"/>         
            </xsl:when>
            <xsl:otherwise>
              <row>
                <entry>Generated row for table with only header rows. DITA requires a body which requires a row.</entry>
              </row>
            </xsl:otherwise>
          </xsl:choose>
        </tbody>
      </tgroup>
    </xsl:element>
  </xsl:template>
  
  <xsl:template match="rsiwp:td|rsiwp:th">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
        <entry>

            <xsl:if test="@rowspan">
                <xsl:attribute name="morerows">
                    <xsl:value-of select="number(@rowspan)-1"/>
                </xsl:attribute>
            </xsl:if>
            <xsl:if test="@colspan">
                <!-- Allow entries to span columns -->
              <xsl:variable name="startColNum" as="xs:integer"
                      select="sum(for $colspan in preceding-sibling::rsiwp:td/@colspan return xs:integer($colspan)) + 
                              count(preceding-sibling::rsiwp:td[not(@colspan)]) + 1"
              />
              <xsl:variable name="endColNum" as="xs:integer"
                      select="sum(for $colspan in preceding-sibling::rsiwp:td/@colspan return xs:integer($colspan)) + 
                              count(preceding-sibling::rsiwp:td[not(@colspan)]) + 
                              xs:integer(@colspan)"
              />
              <xsl:variable name="startColName" as="xs:string"
                select="local:constructColumnName(ancestor::rsiwp:table[1]/rsiwp:cols/rsiwp:col[$startColNum])"
              />
              <xsl:variable name="endColName" as="xs:string"
                select="local:constructColumnName(ancestor::rsiwp:table[1]/rsiwp:cols/rsiwp:col[$endColNum])"
              />
                <xsl:attribute name="namest" select="$startColName"/>
                <xsl:attribute name="nameend" select="$endColName"/>
            </xsl:if>
            <xsl:choose>
                <xsl:when test="@align">
                    <xsl:attribute name="align">
                        <xsl:value-of select="@align"/>
                    </xsl:attribute>
                </xsl:when>
                <xsl:when test="../@align">
                    <xsl:attribute name="align">
                        <xsl:value-of select="../@align"/>
                    </xsl:attribute>
                </xsl:when>
            </xsl:choose>
            <xsl:choose>
                <xsl:when test="@valign">
                    <xsl:attribute name="valign">
                        <xsl:value-of select="@valign"/>
                    </xsl:attribute>
                </xsl:when>
                <xsl:when test="../@valign">
                    <xsl:attribute name="valign">
                        <xsl:value-of select="../@valign"/>
                    </xsl:attribute>
                </xsl:when>
            </xsl:choose>
            <xsl:call-template name="handleBodyParas">
                <xsl:with-param name="bodyParas" select="*"/>
            </xsl:call-template>
            <!-- Original for reference: 
      <xsl:choose>
        <xsl:when test="table">
          <p>
            <xsl:apply-templates select="*|text()|comment()"/>
          </p>
        </xsl:when>
        <xsl:otherwise>
          <xsl:apply-templates select="*|text()|comment()"/>
        </xsl:otherwise>
      </xsl:choose> -->
        </entry>
  </xsl:template>
  
  
  <xsl:template match="rsiwp:tr|rsiwp:thead">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
        <row>
            <xsl:if test="@valign">
                <xsl:attribute name="valign">
                    <xsl:value-of select="@valign"/>
                </xsl:attribute>
            </xsl:if>
            <xsl:apply-templates/>
        </row>
  </xsl:template>
  
  <xsl:template match="rsiwp:td|rsiwp:th" mode="count-cols">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
      <xsl:param name="current-count">1</xsl:param>
      <xsl:variable name="current-span">
          <xsl:choose>
              <xsl:when test="@colspan">
                  <xsl:value-of select="@colspan"/>
              </xsl:when>
              <xsl:otherwise>1</xsl:otherwise>
          </xsl:choose>
      </xsl:variable>
      <xsl:choose>
          <xsl:when test="following-sibling::rsiwp:th or following-sibling::rsiwp:td">
              <xsl:apply-templates select="(following-sibling::rsiwp:th|following-sibling::rsiwp:td)[1]"
                  mode="count-cols">
                  <xsl:with-param name="current-count">
                      <xsl:value-of select="number($current-span) + number($current-count)"/>
                  </xsl:with-param>
              </xsl:apply-templates>
          </xsl:when>
          <xsl:when test="@colspan">
              <xsl:value-of select="number($current-span) + number($current-count) - 1"/>
          </xsl:when>
          <xsl:otherwise>
              <xsl:value-of select="$current-count"/>
          </xsl:otherwise>
      </xsl:choose>
  </xsl:template>
  
  <xsl:template name="create-colspec">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:variable name="rawColwidths" as="xs:double*"
      select="for $a in rsiwp:cols/rsiwp:col/@colwidth 
                  return xs:double(substring-before($a, 'pt'))"
    />
    <xsl:variable name="smallest" as="xs:double"
      select="min($rawColwidths)" 
    
    />
    
    <xsl:for-each select="rsiwp:cols/rsiwp:col">
      <!-- The @colwidth value is always points -->
      <xsl:variable name="pos" as="xs:integer" select="position()"/>
      <xsl:variable name="colwidth" as="xs:string"
        select="if ($tableWidthsProportionalBoolean) 
                   then concat(format-number(($rawColwidths[$pos] div $smallest), '#####.#'), '*')
                   else concat($rawColwidths[$pos], 'pt')"
      />
      <colspec>
        <xsl:attribute name="colname" select="local:constructColumnName(.)"/>
        <xsl:sequence select="@align"/>
        <xsl:attribute name="colwidth" select="$colwidth"/>
      </colspec>
    </xsl:for-each>
  </xsl:template>
    
    
    <!-- Determine which column the current entry sits in. Count the current entry,
     plus every entry before it; take spanned rows and columns into account.
     If any entries in this table span rows, we must examine the entire table to
     be sure of the current column. Use mode="find-matrix-column".
     Otherwise, we just need to examine the current row. Use mode="count-cells". -->
    <xsl:template name="current-cell-position">
        <xsl:choose>
            <xsl:when test="parent::rsiwp:tr/parent::rsiwp:thead">
                <xsl:apply-templates select="(ancestor::rsiwp:table[1]/rsiwp:thead/rsiwp:tr/*[1])[1]"
                    mode="find-matrix-column">
                    <xsl:with-param name="stop-id">
                        <xsl:value-of select="generate-id(.)"/>
                    </xsl:with-param>
                </xsl:apply-templates>
            </xsl:when>
            <xsl:when test="ancestor::rsiwp:table[1]//*[@rowspan][1]">
                <xsl:apply-templates
                    select="(ancestor::rsiwp:table[1]/rsiwp:tbody/rsiwp:tr/*[1]|ancestor::rsiwp:table[1]/rsiwp:tr/*[1])[1]"
                    mode="find-matrix-column">
                    <xsl:with-param name="stop-id">
                        <xsl:value-of select="generate-id(.)"/>
                    </xsl:with-param>
                </xsl:apply-templates>
            </xsl:when>
            <xsl:when test="not(preceding-sibling::rsiwp:td|preceding-sibling::rsiwp:th)">1</xsl:when>
            <xsl:otherwise>
                <xsl:apply-templates select="(preceding-sibling::rsiwp:th|preceding-sibling::rsiwp:td)[last()]"
                    mode="count-cells"/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- Count the number of cells in the current row. Move backwards from the test cell. Add one
     for each entry, plus the number of spanned columns. -->
  <xsl:template match="*" mode="count-cells">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
        <xsl:param name="current-count">1</xsl:param>
        <xsl:variable name="new-count">
            <xsl:choose>
                <xsl:when test="@colspan">
                    <xsl:value-of select="$current-count + number(@colspan)"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="$current-count + 1"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:choose>
            <xsl:when test="not(preceding-sibling::rsiwp:td|preceding-sibling::rsiwp:th)">
                <xsl:value-of select="$new-count"/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:apply-templates select="(preceding-sibling::rsiwp:th|preceding-sibling::rsiwp:td)[last()]"
                    mode="count-cells">
                    <xsl:with-param name="current-count">
                        <xsl:value-of select="$new-count"/>
                    </xsl:with-param>
                </xsl:apply-templates>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- Set up a pseudo-matrix to find the column of the current entry. Start with the first entry
     in the first row. Progress to the end of the row, then start the next row; go until we find
     the test cell (with id=$stop-id).
     If an entry spans rows, add the cells that will be covered to $matrix.
     If we get to an entry and its position is already filled in $matrix, then the entry is pushed
     to the side. Add one to the column count and re-try the entry. -->
  <xsl:template match="*" mode="find-matrix-column">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
        <xsl:param name="stop-id"/>
        <xsl:param name="matrix"/>
        <xsl:param name="row-count">1</xsl:param>
        <xsl:param name="col-count">1</xsl:param>
        <!-- $current-position has the format [1:3] for row 1, col 3. Use to test if this cell is covered. -->
        <xsl:variable name="current-position">[<xsl:value-of select="$row-count"/>:<xsl:value-of
            select="$col-count"/>]</xsl:variable>
        
        <xsl:choose>
            <!-- If the current value is already covered, increment the column number and try again. -->
            <xsl:when test="contains($matrix,$current-position)">
                <xsl:apply-templates select="." mode="find-matrix-column">
                    <xsl:with-param name="stop-id">
                        <xsl:value-of select="$stop-id"/>
                    </xsl:with-param>
                    <xsl:with-param name="matrix">
                        <xsl:value-of select="$matrix"/>
                    </xsl:with-param>
                    <xsl:with-param name="row-count">
                        <xsl:value-of select="$row-count"/>
                    </xsl:with-param>
                    <xsl:with-param name="col-count">
                        <xsl:value-of select="$col-count + 1"/>
                    </xsl:with-param>
                </xsl:apply-templates>
            </xsl:when>
            <!-- If this is the cell we are testing, return the current column number. -->
            <xsl:when test="generate-id(.)=$stop-id">
                <xsl:value-of select="$col-count"/>
            </xsl:when>
            <xsl:otherwise>
                <!-- Figure out what the next column value will be. -->
                <xsl:variable name="next-col-count">
                    <xsl:choose>
                        <xsl:when test="not(following-sibling::*)">1</xsl:when>
                        <xsl:when test="@colspan">
                            <xsl:value-of select="$col-count + number(@colspan) - 1"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="$col-count + 1"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:variable>
                <!-- Determine any values that need to be added to the matrix, if this entry spans rows. -->
                <xsl:variable name="new-matrix-values">
                    <xsl:if test="@rowspan">
                        <xsl:call-template name="add-to-matrix">
                            <xsl:with-param name="start-row">
                                <xsl:value-of select="number($row-count)"/>
                            </xsl:with-param>
                            <xsl:with-param name="end-row">
                                <xsl:value-of select="number($row-count) + number(@rowspan) - 1"/>
                            </xsl:with-param>
                            <xsl:with-param name="start-col">
                                <xsl:value-of select="number($col-count)"/>
                            </xsl:with-param>
                            <xsl:with-param name="end-col">
                                <xsl:choose>
                                    <xsl:when test="@colspan">
                                        <xsl:value-of select="number($col-count) + number(@colspan) - 1"/>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:value-of select="number($col-count)"/>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </xsl:with-param>
                        </xsl:call-template>
                    </xsl:if>
                </xsl:variable>
                <xsl:choose>
                    <!-- If there are more entries in this row, move to the next one. -->
                    <xsl:when test="following-sibling::*">
                        <xsl:apply-templates select="following-sibling::*[1]" mode="find-matrix-column">
                            <xsl:with-param name="stop-id">
                                <xsl:value-of select="$stop-id"/>
                            </xsl:with-param>
                            <xsl:with-param name="matrix">
                                <xsl:value-of select="$matrix"/>
                                <xsl:value-of select="$new-matrix-values"/>
                            </xsl:with-param>
                            <xsl:with-param name="row-count">
                                <xsl:value-of select="$row-count"/>
                            </xsl:with-param>
                            <xsl:with-param name="col-count">
                                <xsl:value-of select="$next-col-count"/>
                            </xsl:with-param>
                        </xsl:apply-templates>
                    </xsl:when>
                    <!-- Otherwise, move to the first entry in the next row. -->
                    <xsl:otherwise>
                        <xsl:apply-templates select="../following-sibling::rsiwp:tr[1]/*[1]" mode="find-matrix-column">
                            <xsl:with-param name="stop-id">
                                <xsl:value-of select="$stop-id"/>
                            </xsl:with-param>
                            <xsl:with-param name="matrix">
                                <xsl:value-of select="$matrix"/>
                                <xsl:value-of select="$new-matrix-values"/>
                            </xsl:with-param>
                            <xsl:with-param name="row-count">
                                <xsl:value-of select="$row-count + 1"/>
                            </xsl:with-param>
                            <xsl:with-param name="col-count">
                                <xsl:value-of select="1"/>
                            </xsl:with-param>
                        </xsl:apply-templates>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- This template returns values that must be added to the table matrix. Every cell in the box determined
     by start-row, end-row, start-col, and end-col will be added. First add every value from the first
     column. When past $end-row, move to the next column. When past $end-col, every value is added. -->
    <xsl:template name="add-to-matrix">
        <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
        <xsl:param name="start-row"/>
        <xsl:param name="end-row"/>
        <xsl:param name="current-row">
            <xsl:value-of select="$start-row"/>
        </xsl:param>
        <xsl:param name="start-col"/>
        <xsl:param name="end-col"/>
        <xsl:param name="current-col">
            <xsl:value-of select="$start-col"/>
        </xsl:param>
        <xsl:choose>
            <xsl:when test="$current-col > $end-col"/>
            <!-- Out of the box; every value has been added -->
            <xsl:when test="$current-row > $end-row">
                <!-- Finished with this column; move to next -->
                <xsl:call-template name="add-to-matrix">
                    <xsl:with-param name="start-row">
                        <xsl:value-of select="$start-row"/>
                    </xsl:with-param>
                    <xsl:with-param name="end-row">
                        <xsl:value-of select="$end-row"/>
                    </xsl:with-param>
                    <xsl:with-param name="current-row">
                        <xsl:value-of select="$start-row"/>
                    </xsl:with-param>
                    <xsl:with-param name="start-col">
                        <xsl:value-of select="$start-col"/>
                    </xsl:with-param>
                    <xsl:with-param name="end-col">
                        <xsl:value-of select="$end-col"/>
                    </xsl:with-param>
                    <xsl:with-param name="current-col">
                        <xsl:value-of select="$current-col + 1"/>
                    </xsl:with-param>
                </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
                <!-- Output the value for the current entry -->
                <xsl:text>[</xsl:text>
                <xsl:value-of select="$current-row"/>:<xsl:value-of select="$current-col"/>
                <xsl:text>]</xsl:text>
                <!-- Move to the next row, in the same column. -->
                <xsl:call-template name="add-to-matrix">
                    <xsl:with-param name="start-row"><xsl:value-of select="$start-row"/></xsl:with-param>
                    <xsl:with-param name="end-row"><xsl:value-of select="$end-row"/></xsl:with-param>
                    <xsl:with-param name="current-row"><xsl:value-of select="$current-row + 1"
                    /></xsl:with-param>
                    <xsl:with-param name="start-col"><xsl:value-of select="$start-col"/></xsl:with-param>
                    <xsl:with-param name="end-col"><xsl:value-of select="$end-col"/></xsl:with-param>
                    <xsl:with-param name="current-col"><xsl:value-of select="$current-col"/></xsl:with-param>
                </xsl:call-template>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
  
  <xsl:template match="rsiwp:run" mode="p-content">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:variable name="tagName" as="xs:string"
      select="
      if (@tagName) 
      then string(@tagName)
      else 'ph'
      "
    />
    <xsl:if test="not(./@tagName)">
      <xsl:message> + [WARNING] No style to tag mapping for character style "<xsl:sequence select="string(@style)"/>"</xsl:message>
    </xsl:if>
    <xsl:variable name="containerType" as="xs:string?"
      select="@containerType"
    />
    <xsl:variable name="doDebug" as="xs:boolean" select="$tagName = 'heChar'"/>
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] Handling run with tag <xsl:value-of select="$tagName"/></xsl:message>
    </xsl:if>
    <xsl:choose>
      <xsl:when test="$containerType != ''">
        <xsl:element name="{$containerType}">
          <xsl:if test="@containerTypeOutputclass">
            <xsl:attribute name="outputclass" select="@containerTypeOutputclass"/>
          </xsl:if>
          <xsl:element name="{$tagName}">
            <xsl:call-template name="generateXtrcAtt"/>
            <xsl:if test="@outputclass">
              <xsl:attribute name="outputclass" select="string(@outputclass)"/>
            </xsl:if>
            <xsl:if test="@langAttValue != ''">
              <xsl:attribute name="xml:lang" select="string(@langAttValue)"/>
            </xsl:if>
            <xsl:variable name="atts" as="attribute()*" select="@*"/>
            <xsl:apply-templates select="stylemap:additionalAttributes" mode="additional-attributes">
              <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
            </xsl:apply-templates>
            <xsl:apply-templates mode="#current"/>
          </xsl:element>
        </xsl:element>
      </xsl:when>
      <xsl:otherwise>
        <xsl:element name="{$tagName}">
          <xsl:call-template name="generateXtrcAtt"/>
          <xsl:if test="@outputclass">
            <xsl:attribute name="outputclass" select="string(@outputclass)"/>
          </xsl:if>
          <xsl:if test="@langAttValue != ''">
            <xsl:attribute name="xml:lang" select="string(@langAttValue)"/>
          </xsl:if>
          <xsl:apply-templates select="stylemap:additionalAttributes" mode="additional-attributes">
            <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
          </xsl:apply-templates>
          <xsl:apply-templates mode="#current"/>
        </xsl:element>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  
  <xsl:template match="text()" mode="p-content">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:copy/>
  </xsl:template>
  
  <xsl:template match="rsiwp:symbol" mode="p-content">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <ph outputclass="symbol"
      ><data name="font" value="{@font}"
      /><xsl:apply-templates mode="#current"
      /></ph>
  </xsl:template>
  
  <xsl:template match="rsiwp:sdt" mode="p-content #default">
    <!-- Could try to generate elements that reflect the SDT tagname or
         alias, but knowing what's allowed is tricky because it depends
         both on the context we're putting things out into and
         the content of the SDT itself.
      -->
    <xsl:apply-templates select="rsiwp:sdtContent/node()" mode="#current"/>
  </xsl:template>
  
  <xsl:template name="handleTopicProlog">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="content" as="node()*"/>
    
    <xsl:call-template name="handleBodyParas">
      <xsl:with-param name="bodyParas" select="$content"/>
    </xsl:call-template>
    
  </xsl:template>
  
  
  <xsl:template name="generateTopicrefAtts">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="topicUrl"/>
    <xsl:param name="mapUrl" tunnel="yes" as="xs:string"/>
    
    <xsl:variable name="topicRelUrl"
      select="relpath:getRelativePath(relpath:getParent($mapUrl), $topicUrl)"
    />
    
    <xsl:attribute name="href" select="$topicRelUrl"/>
    <xsl:if test="@chunk">
      <xsl:copy-of select="@chunk"/>
    </xsl:if>
    <xsl:if test="@collection-type">
      <xsl:copy-of select="@collection-type"/>
    </xsl:if>
    <xsl:if test="@processing-role">
      <xsl:copy-of select="@processing-role"/>
    </xsl:if>
    <xsl:if test="@toc">
      <xsl:copy-of select="@toc"/>
    </xsl:if>
  
    <xsl:call-template name="generateXtrcAtt"/>
      
  </xsl:template>
  
  <xsl:template name="makeTopic">
    <!-- Make a topic. The context element is an rsiwp:topic element -->
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="parentMapUrl" as="xs:string?" tunnel="yes"/>
    <xsl:param name="topicUrl" as="xs:string"/><!-- Result URL for the topic document or its parent topic -->    
    <xsl:param name="topicName" as="xs:string" select="generate-id(.)"/>   
        
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] makeTopic: topicName="<xsl:value-of select="$topicName"/>, topicUrl=<xsl:sequence select="$topicUrl"/></xsl:message>
    </xsl:if>
    
    <!-- If this topic makes a new document, for whatever reason, then
         it will have a topicref parent or rsiwp:body as its parent (root topic
         with no map). If it doesn't have a topicref parent
         then it can't make a new document because there wouldn't be a reference
         to it.
      -->
         
    <xsl:variable name="makeDoc" 
      select="
         count(parent::rsiwp:topicref | 
         parent::rsiwp:body) > 0" 
      as="xs:boolean"
    />

    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] makeTopic: makeDoc=<xsl:value-of select="$makeDoc"/></xsl:message>
    </xsl:if>
    
    <xsl:variable name="formatName" select="./@format" as="xs:string?"/>
    <xsl:if test="$makeDoc and not($formatName)">
      <xsl:message terminate="yes"> + [ERROR] No format= attribute for paragraph style <xsl:sequence select="string(./@styleName)"/>, when topicDoc="yes".</xsl:message>
    </xsl:if>
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] makeTopic: formatName="<xsl:sequence select="$formatName"/>"</xsl:message>
    </xsl:if>
    
    <xsl:variable name="format" select="key('formats', $formatName, $styleMapDoc)[1]" as="element()?"/>
    
    <xsl:if test="$makeDoc and not($format)">
      <xsl:message terminate="yes"> + [ERROR] makeTopic: Failed to find &lt;output&gt; element for @format value "<xsl:value-of select="$formatName"/>" specified for style "<xsl:value-of select="@styleName"/>" <xsl:value-of select="concat(' [', @styleId, ']')"/>. Check your style-to-tag mapping.</xsl:message>
    </xsl:if>
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] makeTopic: format="<xsl:sequence select="$format"/>"</xsl:message>
    </xsl:if>

    <xsl:variable name="schemaAtts" as="attribute()*">
      <xsl:choose>
        <xsl:when test="$makeDoc">
          <xsl:if test="$format/@noNamespaceSchemalocation">
            <xsl:attribute name="xsi:noNamespaceSchemaLocation"
              select="string($format/@noNamespaceSchemaLocation)"
            />
          </xsl:if>
          <xsl:if test="$format/@schemaLocation != ''">
            <xsl:attribute name="xsi:schemaLocation"
              select="string($format/@schemaLocation)"
            />
          </xsl:if>
        </xsl:when>
        <xsl:otherwise>
          <xsl:sequence select="()"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] makeTopic: schemaAtts=<xsl:sequence select="$schemaAtts"/></xsl:message>
    </xsl:if>        
    
    <xsl:if test="$makeDoc">
      <xsl:message> + [INFO] Creating new topic document "<xsl:sequence select="$topicUrl"/>", <xsl:value-of select="*[1]"/>...</xsl:message>
    </xsl:if>
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] makeTopic: Calling constructTopic, topicName="<xsl:value-of select="$topicName"/>"</xsl:message>
      <xsl:message> + [DEBUG] makeTopic:    topicUrl="<xsl:value-of select="$topicUrl"/>"</xsl:message>
    </xsl:if>
    
    <xsl:variable name="topicElement" as="node()*">
      <xsl:call-template name="constructTopic">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
        <xsl:with-param name="topicName" as="xs:string" select="$topicName"/>
        <xsl:with-param name="schemaAtts" as="attribute()*" select="$schemaAtts"/>
        <xsl:with-param name="topicUrl" as="xs:string" tunnel="yes" select="$topicUrl"/>
      </xsl:call-template>
    </xsl:variable>
    
    <xsl:choose>
      <xsl:when test="$makeDoc">
        
        <xsl:variable name="topicrefType" as="xs:string" 
          select="if (@topicrefType) then @topicrefType else 'topicref'"
        />
        <!-- Now do ID fixup on the result document: -->
        <xsl:message> + [INFO] Applying final-fixup mode to <xsl:sequence select="$topicUrl"/>...</xsl:message>
        <rsiwp:result-document href="{$topicUrl}"
            doctype-public="{$format/@doctype-public}"
            doctype-system="{$format/@doctype-system}"
            >
          <xsl:apply-templates select="$topicElement" mode="final-fixup">
            <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
          </xsl:apply-templates>
        </rsiwp:result-document>
        <xsl:message> + [INFO] final-fixup mode applied.</xsl:message>
      </xsl:when>
      <xsl:otherwise>
        <xsl:variable name="indent" as="xs:string"
          select="string-join(for $i in 1 to count(ancestor::rsiwp:topic) return '   ','')"
        />
         <xsl:message> + [INFO] Creating nested topic<xsl:value-of select="$indent"/> [<xsl:value-of select="$topicName"/>] <xsl:value-of select="*[1]"/></xsl:message>

        <!-- Output the topic as a subtopic of the parent topic or as the root topic
             where there is no map.
          -->
        <xsl:sequence select="$topicElement"/>
      </xsl:otherwise>
    </xsl:choose>

    <!-- Process any subordinate topicrefs or maps.
         This should only produce a result if this topic
         produced a topic document.
      -->
    <xsl:apply-templates 
        select="
        rsiwp:topicref | 
        rsiwp:map | 
        rsiwp:mapref">
      <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
    </xsl:apply-templates>

  </xsl:template>

  <xsl:template mode="final-fixup" match="*">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] final-fixup: handling <xsl:sequence select="name(.)"/></xsl:message>
    </xsl:if>
    <xsl:copy>
      <xsl:apply-templates select="@*,node()" mode="#current">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates>
    </xsl:copy>
  </xsl:template>
  
  <xsl:template mode="final-fixup" match="@id" priority="2">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <!-- Override this template to implement specific ID generators -->
    <xsl:variable name="idGenerator" select="string(../@idGenerator)" as="xs:string"/>
    <xsl:choose>
      <xsl:when test="$idGenerator = '' or $idGenerator = 'default'">
        <xsl:if test="$doDebug">
          <xsl:message> + [DEBUG] final-fixup/@ID: Using default ID generator, returning "<xsl:sequence select="string(.)"/>"</xsl:message>
        </xsl:if>
        <xsl:copy/><!-- Use the base generated ID value. -->
      </xsl:when>
      <xsl:otherwise>
        <xsl:message> - [WARNING] Unrecognized ID generator name "<xsl:sequence select="$idGenerator"/>"</xsl:message>
        <xsl:copy/><!-- Use the base generated ID value. -->
      </xsl:otherwise>
    </xsl:choose>    
  </xsl:template>
  
  <xsl:template mode="final-fixup" match="@idGenerator | @class">
    <!-- Suppress -->
  </xsl:template>
  
  <xsl:template mode="final-fixup" match="@*">
    <xsl:copy/>
  </xsl:template>
  
  <xsl:template mode="final-fixup" match="text() | processing-instruction()">
    <xsl:copy/>
  </xsl:template>
  
  <xsl:template name="constructTopic">
  <!-- Constructs the topic itself. Result is a topic element containing the topic and any nested topics. 
       
       Context is rwiwp:topic element
  -->
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="topicName" as="xs:string" select="generate-id(.)"/>
    <xsl:param name="topicUrl" as="xs:string" tunnel="yes"/>
    <xsl:param name="schemaAtts" as="attribute()*" select="()"/>
    
    
    <xsl:variable name="initialSectionType" as="xs:string" select="string(@initialSectionType)"/>
    <xsl:variable name="level" as="xs:integer" select="1"/><!-- Topics establish a new level context -->
    <xsl:variable name="nextLevel" as="xs:integer" select="$level + 1"/>
 
    <!-- topicName will be used for the @id on the generated topic, so we must ensure it complies with
      the relevant lexical rules -->
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] Original topicName=<xsl:sequence select="$topicName"/></xsl:message>
    </xsl:if>
    
    <!-- Rule #1: @id must start with ':', '_', a-z, or A-Z; if it does not then add on an underscore;
      desire is to continue to provide uniqueness of @id -->
    <xsl:variable name="firstCharacterOfTopicName" select="substring($topicName, 1, 1)" as="xs:string" />
    <xsl:variable name="newTopicName" as="xs:string">
      <xsl:analyze-string regex="[a-zA-Z:_]" select="$firstCharacterOfTopicName">
        <xsl:matching-substring><xsl:sequence select="$topicName"/></xsl:matching-substring>
        <xsl:non-matching-substring><xsl:sequence select="concat(':', $topicName)"/></xsl:non-matching-substring>
      </xsl:analyze-string>
    </xsl:variable>
    <!-- Rule #2: @id must not contain spaces -->
    <xsl:variable name="newTopicName" select="replace($newTopicName, '[ &#9;&#10;&#13;]', '_')" as="xs:string" />
    <!-- Rule #3: @id must only contain numbers, letters ':', '_', '-', or '.' -->
    <xsl:variable name="newTopicName" as="xs:string+">
      <xsl:analyze-string regex="[a-zA-Z:_\.\-0-9]" select="$newTopicName">
        <xsl:matching-substring><xsl:sequence select="."/></xsl:matching-substring>
        <xsl:non-matching-substring><xsl:sequence select="':'"/></xsl:non-matching-substring>
      </xsl:analyze-string>
    </xsl:variable>
    <!-- Assign the new value to $topicName -->
    <xsl:variable name="topicName" select="string-join($newTopicName, '')" as="xs:string" />
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] After fixup to comply with lexical rules for @id, new topicName=<xsl:sequence select="$topicName"/></xsl:message>
    </xsl:if>
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] constructTopic: topicName=<xsl:sequence select="$topicName"/></xsl:message>
      <xsl:message> + [DEBUG] constructTopic: topic=<xsl:sequence select="local:reportPara(.)"/></xsl:message>
    </xsl:if>
    
    <xsl:variable name="topicType" as="xs:string"
      select="@topicType"
    />
    
    <xsl:variable name="bodyType" as="xs:string"
      select="
      if (@bodyType)
      then @bodyType
      else 'body'
      "
    />
    
    <xsl:variable name="prologType" as="xs:string"
      select="
      if (@prologType)
      then @prologType
      else 'prolog'
      "
    />
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] constructTopic: topicType="<xsl:value-of select="$topicType"/>"</xsl:message>
      <xsl:message> + [DEBUG] constructTopic: bodyType="<xsl:value-of select="$bodyType"/>"</xsl:message>
      <xsl:message> + [DEBUG] constructTopic: prologType="<xsl:value-of select="$prologType"/>"</xsl:message>
      <xsl:message> + [DEBUG] constructTopic: initialSectionType="<xsl:value-of select="$initialSectionType"/>"</xsl:message>
    </xsl:if>
    
    <xsl:variable name="titleIndexEntries" as="element()*">
      <xsl:if test="local:isTopicTitle(.)">
        <xsl:sequence select=".//rsiwp:indexterm"/>
      </xsl:if>
    </xsl:variable>
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] constructTopic: Creating topic element <xsl:value-of select="$topicType"/></xsl:message>
      <xsl:message> + [DEBUG] constructTopic: topicName="<xsl:sequence select="$topicName"/>"</xsl:message>
      <xsl:message> + [DEBUG] constructTopic: topicUrl="<xsl:sequence select="$topicUrl"/>"</xsl:message>
    </xsl:if>
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] constructTopic: schemaAtts=<xsl:sequence select="$schemaAtts"/></xsl:message>
    </xsl:if>
    <xsl:element name="{$topicType}">
      <xsl:attribute name="id" select="$topicName"/>
      <xsl:call-template name="generateXtrcAtt">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:call-template>
      <xsl:attribute name="isTopic" select="'true'"/>      
      <xsl:attribute name="xml:lang" 
        select="if (@langAttValue) 
        then string(@langAttValue) 
        else $language"
      />
      <!-- Indicate that this element is in fact a topic as there's no other way
           to know this during the result document processing phase.
        -->
      <xsl:sequence select="$schemaAtts"/>
      <xsl:if test="@outputclass">
        <xsl:attribute name="outputclass" select="@outputclass"/>
      </xsl:if>
      <xsl:variable name="titleTagName" as="xs:string"
        select="if (@tagName)
        then @tagName
        else 'title'
        "
      />
      <xsl:if test="$doDebug">
        <xsl:message> + [DEBUG] constructTopic: Applying templates to firstP...</xsl:message>
      </xsl:if>      
      <xsl:variable name="firstP" as="element()" select="*[1]"/>
      <xsl:apply-templates select="$firstP"><!-- Make the topic title. -->
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates>
      <xsl:if test="$doDebug">
        <xsl:message> + [DEBUG] constructTopic: For-each-group on content...</xsl:message>
      </xsl:if>      
      <xsl:for-each-group 
        select="*[position() > 1] except (rsiwp:topic, rsiwp:topicref, rsiwp:map, rsiwp:mapref)" 
        group-starting-with="*[string(@structureType) = 'topicTitle' and string(@level) = string($nextLevel)]">
        <xsl:if test="$doDebug">
          <xsl:message> + [DEBUG] constructTopic: currentGroup[<xsl:sequence select="position()"/>]: <xsl:sequence select="current-group()"/></xsl:message>
        </xsl:if>      
        <xsl:choose>
          <xsl:when test="current-group()[position() = 1] and current-group()[1][not(string(@structureType) = ('topicTitle'))]">
            
            <xsl:if test="$doDebug">
              <xsl:message> + [DEBUG] constructTopic: First group and it does not start with topicTitle structure type.</xsl:message>
              <xsl:message> + [DEBUG] constructTopic:   currentGroup=<xsl:sequence select="local:reportParas(current-group())"/></xsl:message>
            </xsl:if>
            
            <!-- Prolog and body elements for the topic -->
            <xsl:apply-templates select="current-group()[string(@topicZone) = 'titleAlts']">
              <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
            </xsl:apply-templates>        
            <xsl:apply-templates select="current-group()[string(@topicZone) = 'shortdesc']">
              <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
            </xsl:apply-templates>             
            <xsl:if test="current-group()[string(@topicZone) = 'prolog' or $level = 0] or count($titleIndexEntries) > 0">
                  <xsl:if test="$doDebug">
                    <xsl:message> + [DEBUG] constructTopic: Handling prolog paragraphs in the current group.</xsl:message>
                  </xsl:if>
              <xsl:choose>
                <xsl:when test="$level = 0">
                  <!-- Root topic -->
                  <xsl:if test="$doDebug">
                    <xsl:message> + [DEBUG] constructTopic:  Prolog for root topic...</xsl:message>
                  </xsl:if>
                  <xsl:element name="{$prologType}">
                    <xsl:call-template name="generateXtrcAtt">
                      <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
                    </xsl:call-template>
                    <!-- For root topic, can pull metadata from anywhere in the incoming document. -->
                    <xsl:apply-templates select="root($firstP)//*[string(@containingTopic) = ('root', 'roottopic') and 
                      string(@topicZone) = 'prolog' and 
                      contains(@baseClass, ' topic/author ')]">
                      <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
                    </xsl:apply-templates>     
                    <!-- FIXME: This is a hack to handle index entries in keywords. Need to refine this
                                so it handles keywords explicitly mapped to the prolog.
                      -->
                    <xsl:if test="count($titleIndexEntries) > 0">
                      <metadata>
                        <keywords>
                           <xsl:apply-templates select="$titleIndexEntries" mode="p-content">
                             <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
                           </xsl:apply-templates>                       
                        </keywords>
                      </metadata>
                    </xsl:if>
                    <xsl:apply-templates select="root($firstP)//*[string(@containingTopic) = ('root', 'roottopic') and 
                      string(@topicZone) = 'prolog' and 
                      contains(@baseClass, ' topic/data ')
                      ]">
                      <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
                    </xsl:apply-templates>                        
                  </xsl:element>                  
                </xsl:when>
                <xsl:when test="current-group()[@containingTopic = ('currenttopic')]">
                  <xsl:if test="$doDebug">
                    <xsl:message> + [DEBUG] constructTopic:  Prolog for current topic...</xsl:message>
                  </xsl:if>
                  <xsl:element name="{$prologType}">
                    <xsl:call-template name="generateXtrcAtt">
                      <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
                    </xsl:call-template>
                    <xsl:apply-templates 
                      select="current-group()[@containingTopic = ('currenttopic') and 
                                              string(@topicZone) = 'prolog']">
                      <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
                    </xsl:apply-templates>
                  </xsl:element>
                </xsl:when>
                <xsl:when test="count($titleIndexEntries) > 0">
                  <xsl:if test="$doDebug">
                    <xsl:message> + [DEBUG] constructTopic:  Title index entries</xsl:message>
                  </xsl:if>
                  <xsl:element name="{$prologType}">
                    <xsl:call-template name="generateXtrcAtt"/>
                    <metadata>
                      <keywords>
                        <xsl:apply-templates select="$titleIndexEntries" mode="p-content">
                          <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
                        </xsl:apply-templates>                          
                      </keywords>
                    </metadata>
                  </xsl:element>
                </xsl:when>
                <xsl:otherwise>
                  <!-- Must be only root-level prolog elements in this non-root topic context -->
                  <xsl:if test="$doDebug">
                    <xsl:message> + [DEBUG] constructTopic:  No prolog elements in this group</xsl:message>
                  </xsl:if>
                  
                </xsl:otherwise>
              </xsl:choose>
            </xsl:if>
            <xsl:if test="current-group()[string(@topicZone) = 'body']">
              <xsl:if test="$doDebug">        
                <xsl:message> + [DEBUG] current group is topicZone body</xsl:message>
                <xsl:message> + [DEBUG]   paras=<xsl:value-of select="local:reportParas(current-group())"/></xsl:message>
              </xsl:if>
              <xsl:element name="{$bodyType}">
                <xsl:call-template name="generateXtrcAtt">
                  <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
                </xsl:call-template>
                <xsl:call-template name="handleSectionParas">
                  <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
                  <xsl:with-param name="sectionParas" 
                    select="current-group()[string(@topicZone) = 'body']" as="element()*"/>
                  <xsl:with-param name="initialSectionType" select="$initialSectionType" as="xs:string"/>
                  <xsl:with-param name="resultUrl" as="xs:string" tunnel="yes"
                    select="$topicUrl"/>
                </xsl:call-template>
              </xsl:element>                  
            </xsl:if>
          </xsl:when>
          <xsl:otherwise>
            <xsl:if test="$doDebug">        
              <xsl:message> + [DEBUG] makeTopic(): Not topicZone prolog or body</xsl:message>
            </xsl:if>
          </xsl:otherwise>
        </xsl:choose>        
      </xsl:for-each-group>
      <xsl:if test="$doDebug">        
        <xsl:message> + [DEBUG] makeTopic(): Applying templates to child map and topic items...</xsl:message>
      </xsl:if>
      <xsl:apply-templates select="rsiwp:topic">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
        <xsl:with-param name="topicName" as="xs:string" tunnel="yes" select="''"/>
      </xsl:apply-templates>
    </xsl:element>      
  </xsl:template>
  
  <xsl:template name="handleSectionParas">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="sectionParas" as="element()*"/>
    <xsl:param name="initialSectionType" as="xs:string"/>
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] handleSectionParas: initialSectionType="<xsl:sequence select="$initialSectionType"/>"</xsl:message>
    </xsl:if>
    <xsl:for-each-group select="$sectionParas" group-starting-with="*[string(@structureType) = 'section']">
      <xsl:choose>
        <xsl:when test="current-group()[position() = 1] and string(@structureType) != 'section'">
          <xsl:choose>
            <xsl:when test="$initialSectionType != ''">
              <xsl:element name="{$initialSectionType}">
                <xsl:call-template name="generateXtrcAtt"/>
                <xsl:call-template name="handleBodyParas">
                  <xsl:with-param name="bodyParas" select="current-group()"/>
                </xsl:call-template>
              </xsl:element>
            </xsl:when>
            <xsl:otherwise>
              <xsl:call-template name="handleBodyParas">
                <xsl:with-param name="bodyParas" select="current-group()"/>
              </xsl:call-template>
            </xsl:otherwise>
          </xsl:choose>
          
        </xsl:when>
        <xsl:otherwise>
          <xsl:variable name="sectionType" as="xs:string"
              select="if (@sectionType) then string(@sectionType) else 'section'"
          />
          <xsl:element name="{$sectionType}">
            <xsl:call-template name="generateXtrcAtt"/>
            <xsl:if test="@spectitle != ''">
              <xsl:variable name="spectitle" select="local:constructSpectitle(.)" as="xs:string"/>
              <xsl:attribute name="spectitle" select="$spectitle"/>
            </xsl:if>
            <xsl:variable name="firstSectionPara" as="element()">
              <xsl:choose>
                <xsl:when test="starts-with(@spectitle, '#')">
                  <xsl:sequence select="local:removeSpectitleContent(.)"/>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:sequence select="current-group()[1]"/>
                </xsl:otherwise>
              </xsl:choose>
              
            </xsl:variable>
            <xsl:variable name="bodyParas"
              select="if (string(@useAsTitle) = 'no' or 
                          ((@spectitle != '') and 
                           (not(starts-with(@spectitle, '#')))))
                         then current-group()[position() > 1]
                         else ($firstSectionPara, current-group()[position() > 1])                         
              "
            />
            <xsl:call-template name="handleBodyParas">
              <xsl:with-param name="bodyParas" select="$bodyParas"/>
            </xsl:call-template>
          </xsl:element>
        </xsl:otherwise>
      </xsl:choose>      
    </xsl:for-each-group>
  </xsl:template>
  
  <xsl:template name="handleBodyParas">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="bodyParas" as="element()*"/>
    
    <xsl:for-each-group select="$bodyParas" group-adjacent="boolean(@containerType)">
      <xsl:choose>
        <xsl:when test="@containerType">
          <xsl:variable name="containerGroup" as="element()">
            <containerGroup containerType="{@containerType}">
              <xsl:sequence select="current-group()"/>
            </containerGroup>
          </xsl:variable>
          <xsl:apply-templates select="$containerGroup"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:apply-templates select="current-group()"/>
        </xsl:otherwise>
      </xsl:choose>
      
    </xsl:for-each-group>
  </xsl:template>
  
  <xsl:template match="containerGroup">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:call-template name="processLevelNContainers">
      <xsl:with-param name="context" select="*" as="element()*"/>
      <xsl:with-param name="level" select="1" as="xs:integer"/>
      <xsl:with-param name="currentContainer" select="'body'" as="xs:string"/>
    </xsl:call-template>    
  </xsl:template>
  
  <xsl:template name="generateXtrcAtt">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:attribute name="xtrc" select="@wordLocation"/>
   
  </xsl:template>
  
  <xsl:template name="processLevelNContainers">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="context" as="element()*"/>
    <xsl:param name="level" as="xs:integer"/>
    <xsl:param name="currentContainer" as="xs:string"/>
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] processLevelNContainers, level="<xsl:sequence select="$level"/>"</xsl:message>
      <xsl:message> + [DEBUG]   currentContainer="<xsl:sequence select="$currentContainer"/>"</xsl:message>     
    </xsl:if>
    <xsl:for-each-group select="$context[@level = $level]" group-adjacent="@containerType">
      <xsl:choose>
        <xsl:when test="$currentContainer != string(@containerType)">
          <xsl:element name="{@containerType}">
            <xsl:call-template name="generateXtrcAtt"/>
            <xsl:if test="@containerOutputclass">
              <xsl:attribute name="outputclass" select="string(@containerOutputclass)"/>
            </xsl:if>
            <xsl:for-each select="current-group()">
              <xsl:call-template name="handleGroupSequence">
                <xsl:with-param name="level" select="$level" as="xs:integer"/>
              </xsl:call-template>
            </xsl:for-each>
          </xsl:element>
        </xsl:when>
        <xsl:otherwise>
          <xsl:for-each select="current-group()">
            <xsl:call-template name="handleGroupSequence">
              <xsl:with-param name="level" select="$level" as="xs:integer"/>
            </xsl:call-template>
          </xsl:for-each>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:for-each-group>    
  </xsl:template>
  
  <xsl:template name="handleGroupSequence">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="level" as="xs:integer"/>
    <xsl:choose>
      <xsl:when test="string(@structureType) = 'dt' and @level = $level">
        <xsl:variable name="dlEntryType" as="xs:string"
          select="if (@dlEntryType) then string(@dlEntryType) else 'dlentry'"
        />
        <xsl:element name="{$dlEntryType}"> 
          <xsl:call-template name="generateXtrcAtt"/>
          <xsl:call-template name="transformPara"/>
          <xsl:variable name="followingSibling" as="element()?" select="following-sibling::*[1]"/>
          <xsl:variable name="precedingSibling" as="element()?" select="preceding-sibling::*[1]"/>
          <!-- find position of next <dt> element type -->
          <xsl:variable name="followingSiblingDtPositions" as="item()*">
            <xsl:for-each select="following-sibling::*">
              <xsl:if test="@structureType='dt'">
                <xsl:sequence select="position()"/>
              </xsl:if>
            </xsl:for-each>
          </xsl:variable>
          <xsl:variable name="firstFollowingSiblingDtPosition" as="xs:integer">
            <xsl:choose>
              <xsl:when test="following-sibling::*[@structureType='dt']">
                <xsl:value-of select="$followingSiblingDtPositions[position()=1]"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="0"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:variable>
          <xsl:choose>
            <xsl:when test="$followingSibling/@level &gt; @level">
              <xsl:for-each-group select="following-sibling::*" group-adjacent="@level">
                <xsl:if test="@level &gt; $level">
                  <xsl:element name="{@containerType}">
                    <xsl:call-template name="generateXtrcAtt"/>
                    
                  <xsl:for-each select="current-group()">
                    <xsl:choose>
                      <xsl:when test="string(@structureType) = 'dt'">
                        <xsl:variable name="nestedFollowingSibling" as="element()?" select="following-sibling::*[1]"/>
                        <xsl:variable name="dlEntryType" as="xs:string"
                          select="if (@dlEntryType) then string(@dlEntryType) else 'dlentry'"
                        />
                        <xsl:element name="{$dlEntryType}">  
                          <xsl:call-template name="generateXtrcAtt"/>
                          <xsl:call-template name="transformPara"/>
                          <!-- find position of next <dt> element type -->
                          <xsl:variable name="followingNestedSiblingDtPositions" as="item()*">
                            <xsl:for-each select="following-sibling::*">
                              <xsl:if test="@structureType='dt'">
                                <xsl:sequence select="position()"/>
                              </xsl:if>
                            </xsl:for-each>
                          </xsl:variable>
                          <xsl:variable name="firstFollowingNestedSiblingDtPosition" as="xs:integer">
                            <xsl:choose>
                              <xsl:when test="following-sibling::*[@structureType='dt']">
                                <xsl:value-of select="$followingNestedSiblingDtPositions[position()=1]"/>
                              </xsl:when>
                              <xsl:otherwise>
                                <xsl:value-of select="0"/>
                              </xsl:otherwise>
                            </xsl:choose>
                          </xsl:variable>
                          <xsl:choose>
                            <xsl:when test="following-sibling::*[@structureType='dt']">
                              <xsl:for-each select="following-sibling::*[@structureType='dd'][position() &lt; $firstFollowingNestedSiblingDtPosition]">
                                <xsl:call-template name="transformPara"/>
                              </xsl:for-each>
                            </xsl:when>
                            <xsl:otherwise>
                              <xsl:for-each select="$nestedFollowingSibling">
                                <xsl:call-template name="transformPara"/>
                              </xsl:for-each>
                            </xsl:otherwise>
                          </xsl:choose>
                          <!-- 
                          <xsl:for-each select="$nestedFollowingSibling">
                            <xsl:call-template name="transformPara"/>
                          </xsl:for-each>
                          -->
                        </xsl:element>
                      </xsl:when>
                    </xsl:choose>
                  </xsl:for-each>
                </xsl:element>
                </xsl:if>
              </xsl:for-each-group>
            </xsl:when>
            <xsl:when test="$precedingSibling/@level &lt; @level"/>
              <xsl:otherwise>
                <xsl:choose>
                  <xsl:when test="following-sibling::*[@structureType='dt']">
                    <xsl:for-each select="following-sibling::*[@structureType='dd'][position() &lt; $firstFollowingSiblingDtPosition]">
                      <xsl:call-template name="transformPara"/>
                    </xsl:for-each>
                  </xsl:when>
                  <xsl:otherwise>
                    <xsl:for-each select="following-sibling::*[@structureType='dd']">
                      <xsl:call-template name="transformPara"/>
                    </xsl:for-each>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:otherwise>
          </xsl:choose>
        </xsl:element>
      </xsl:when>
      <xsl:when test="string(@structureType) = 'dd'"/><!-- Handled by dt processing -->
     <xsl:when test="following-sibling::*[1][@level &gt; $level]">
        <xsl:variable name="me" select="." as="element()"/>
       <xsl:element name="{@tagName}">  
         <xsl:call-template name="generateXtrcAtt"/>
         <xsl:apply-templates select="rsiwp:bookmarkStart" mode="generate-para-ids">
           <xsl:with-param name="tagName" select="@tagName" as="xs:string"/>
         </xsl:apply-templates>
         <xsl:call-template name="transformParaContent"/>
          <xsl:call-template name="processLevelNContainers">
            <xsl:with-param name="context" 
              select="following-sibling::*[(@level = $level + 1) and 
              preceding-sibling::*[@level = $level][1][. is $me]]" as="element()*"/>
            <xsl:with-param name="level" select="$level + 1" as="xs:integer"/>
            <xsl:with-param name="currentContainer" select="@tagName" as="xs:string"/>
          </xsl:call-template>
        </xsl:element>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="transformPara"/>
      </xsl:otherwise>
    </xsl:choose>    
  </xsl:template>
  
  <xsl:template mode="generate-para-ids" match="rsiwp:bookmarkStart">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="tagName" as="xs:string" select="../@tagName"/>
    <xsl:attribute name="id" select="concat($tagName, '_', @name)"/>
  </xsl:template>
  
  <xsl:template match="rsiwp:break" mode="p-content">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <br/>
  </xsl:template>
  
  <xsl:template match="rsiwp:break[@xtype = ('page', 'column')]" mode="p-content">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <!-- Suppress by default. The only thing these could safely become would be PIs or comments. -->
  </xsl:template>
  
  <xsl:template match="rsiwp:tab" mode="p-content">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <tab/>
  </xsl:template>
  
  <xsl:template match="rsiwp:b | 
    rsiwp:i | 
    rsiwp:u | 
    rsiwp:sup | 
    rsiwp:sub | 
    rsiwp:ph 
    "
    mode="p-content">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    
    <xsl:variable name="containerType" as="xs:string?"
      select="@containerType"
    />
    <xsl:choose>
      <xsl:when test="$containerType != ''">
        <xsl:element name="{$containerType}">
          <xsl:if test="@containerTypeOutputclass">
            <xsl:attribute name="outputclass" select="@containerTypeOutputclass"/>
          </xsl:if>
          <xsl:element name="{local-name()}">
            <xsl:call-template name="generateXtrcAtt"/>
            <xsl:if test="@outputclass">
              <xsl:attribute name="outputclass" select="string(@outputclass)"/>
            </xsl:if>
            <xsl:if test="@langAttValue != ''">
              <xsl:attribute name="xml:lang" select="string(@langAttValue)"/>
            </xsl:if>
            <xsl:sequence select="./@* except(@langAttValue)"/>            
            <xsl:apply-templates mode="p-content"/>
          </xsl:element>
        </xsl:element>
      </xsl:when>
      <xsl:otherwise>
        <xsl:element name="{local-name()}">
          <xsl:call-template name="generateXtrcAtt"/>
          <xsl:if test="@outputclass">
            <xsl:attribute name="outputclass" select="string(@outputclass)"/>
          </xsl:if>
          <xsl:if test="@langAttValue != ''">
            <xsl:attribute name="xml:lang" select="string(@langAttValue)"/>
          </xsl:if>
          <xsl:apply-templates mode="p-content"/>
        </xsl:element>
      </xsl:otherwise>
    </xsl:choose>
    
  </xsl:template>
  
  <xsl:template mode="p-content" match="rsiwp:bookmarkStart">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:if test="$includeWordBookmarksBoolean">
      <data name="bookmarkStart" value="{@name}" id="bookmark_{@id}">
      </data>
    </xsl:if>
  </xsl:template>
  
  <xsl:template mode="p-content" match="rsiwp:bookmarkEnd">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:if test="$includeWordBookmarksBoolean">
      <data name="bookmarkEnd" value="bookmark_{@id}"/>
    </xsl:if>
  </xsl:template>
  
  <xsl:template match="rsiwp:indexterm" mode="p-content">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="inTitleContext" as="xs:boolean" tunnel="yes"
       select="false()"/>
    <xsl:if test="not($inTitleContext)">
      <xsl:element name="{local-name()}">
        <xsl:sequence select="./@*"/>
        <xsl:apply-templates mode="p-content"/>
      </xsl:element>
    </xsl:if>
  </xsl:template>
  
  <xsl:template match="m:math" mode="p-content">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:choose>
      <xsl:when test="$isDita13">
        <mathml> 
          <xsl:apply-templates select="." mode="fixup-mathml-namespace" />
        </mathml>
      </xsl:when>
      <xsl:otherwise>
        <!-- if not DITA 1.3, then assume 1.2 -->
        <d4p_MathML> 
          <xsl:apply-templates select="." mode="fixup-mathml-namespace" />
        </d4p_MathML>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  
  <!-- We need to make sure the namespace prefix for all MathML elements is "m:", simply having a
  node without a prefix but a @xmlns is not sufficient, for some reason -->
  <xsl:template match="m:*" mode="fixup-mathml-namespace">
    <xsl:element name="m:{local-name()}">
      <xsl:apply-templates select="@*|node()" mode="#current"/>
    </xsl:element>
  </xsl:template>
  
  <xsl:template match="@*|node()" mode="fixup-mathml-namespace">
    <xsl:copy>
      <xsl:apply-templates select="@*|node()" mode="#current"/>
    </xsl:copy>
  </xsl:template>
  
  
  <xsl:template match="rsiwp:fn" mode="p-content">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:element name="{local-name()}">
      <xsl:sequence select="./@*"/>
      <xsl:call-template name="handleSectionParas">
        <xsl:with-param name="sectionParas" select="*" as="element()*"/>
        <xsl:with-param name="initialSectionType" as="xs:string" select="''"/>        
      </xsl:call-template>
    </xsl:element>
  </xsl:template>
  
  <xsl:template match="rsiwp:hyperlink" mode="p-content">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>    
    <xsl:param name="simpleWpDoc" as="document-node()" tunnel="yes"
    />            
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] rsiwp:hyperlink: structureType="<xsl:sequence select="string(@structureType)"/>"</xsl:message>
    </xsl:if>
    <xsl:variable name="tagName" as="xs:string"
      select="if (@tagName) then string(@tagName) else 'xref'"
    />
    <xsl:element name="{$tagName}">
      <!-- Not all Word hyperlinks become DITA hyperlinks: -->
      <xsl:if test="string(@structureType) = 'xref'">
        <xsl:variable name="origHref" select="@href" as="xs:string"/>
        <xsl:variable name="href" as="xs:string">
          <xsl:choose>
            <xsl:when test="matches($origHref, '^\w+:')">
              <!-- URI of some sort, assume external scope -->
              <xsl:sequence select="$origHref"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:sequence select="local:bookmarkRefToDitaRef(., $simpleWpDoc)"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        <xsl:variable name="scope" as="xs:string?">
          <xsl:choose>
            <xsl:when test="matches($origHref, '^\w+:')">
              <xsl:sequence select="'external'"/>
            </xsl:when>
            <xsl:otherwise>
              <!-- scope is defaulted  -->
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        <xsl:variable name="format" as="xs:string?">
          <xsl:choose>
            <xsl:when test="matches($origHref, '^\w+:')">
              <xsl:variable name="extension" select="relpath:getExtension($origHref)"/>
              <xsl:choose>
                <xsl:when test="$extension = ('htm', 'html', 'pgp')">
                  <!-- FIXME: Add to the list of extensions that indicate HTML format -->
                  <xsl:sequence select="'html'"/>
                </xsl:when>
                <xsl:when test="(starts-with($origHref, 'http:') or 
                                 starts-with($origHref, 'https:'))">
                  <!-- If the scheme is HTTP or HTTPS then if there
                       is no extension assum "html" otherwise use the 
                       extension of the resource.
                    -->
                  <xsl:choose>
                    <xsl:when test="$extension = ''">
                      <xsl:sequence select="'html'"/>
                    </xsl:when>
                    <xsl:otherwise>
                      <xsl:sequence select="$extension"/>
                    </xsl:otherwise>
                  </xsl:choose>
                </xsl:when>
                <xsl:otherwise>
                  <!-- Use the URI scheme as the format, since that will usually be
                       correct (e.g., mailto, ftp, etc.)
                    -->
                  <xsl:variable name="scheme" 
                    select="if (starts-with($origHref, 'urn:'))
                               then substring-before(substring-after($origHref, 'urn:'), ':')
                               else substring-before($origHref, ':')
                    "/>
                  <xsl:if test="$doDebug">
                    <xsl:message> + [DEBUG] rsiwp:hyperlink: scheme="<xsl:sequence select="$scheme"/>"</xsl:message>
                  </xsl:if>
                  <xsl:sequence select="$scheme"/>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:when>
            <xsl:otherwise>
              <!-- Format is unspecified -->
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        <xsl:attribute name="href" select="$href"/>
        <xsl:if test="$scope">
          <xsl:attribute name="scope" select="$scope"/>
        </xsl:if>
        <xsl:if test="$format">
          <xsl:attribute name="format" select="$format"/>
        </xsl:if>
        <xsl:if test="@langAttValue != ''">
          <xsl:attribute name="xml:lang" select="string(@langAttValue)"/>
        </xsl:if>
      </xsl:if>
      <xsl:apply-templates mode="#current"/>
    </xsl:element>
  </xsl:template>
  
  <xd:doc xmlns:xd="http://www.oxygenxml.com/ns/doc/xsl">
    <xd:desc>
      <xd:p>Generates a DITA image element. The URL of the image is determined
      by creating a relative path constructed from the value of the @src
      attribute in the simple WP, which should point to the absolute
      location of the image as it will be accessed by the generated XML,
      and the directory containing the result file for the
      topic being generated, so the resulting value is a relative
      path from the containing topic document to the image.</xd:p>
    </xd:desc>
    <xd:param name="resultUrl"></xd:param>
  </xd:doc>
  <xsl:template match="rsiwp:image" mode="p-content">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="resultUrl" as="xs:string" tunnel="yes"/>
        
    <xsl:variable name="resultDir" select="relpath:getParent($resultUrl)"/>
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] rwiwp:image: resultDir="<xsl:value-of select="$resultDir"/>"</xsl:message>
    </xsl:if>
    <xsl:variable name="effectiveImageFilename" as="xs:string" 
      select="local:constructImageFilenameForRsiwpImage(.)"
    />
    <xsl:variable name="imageUrl" as="xs:string"
      select="relpath:getRelativePath($resultDir, $effectiveImageFilename)"
    />
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] rwiwp:image: $effectiveImageFilename="<xsl:value-of select="$effectiveImageFilename"/>"</xsl:message>
      <xsl:message> + [DEBUG] rwiwp:image: imageUrl="<xsl:value-of select="$imageUrl"/>"</xsl:message>
    </xsl:if>
    <image href="{$imageUrl}">
      <xsl:copy-of select="./@width, ./@height"/>
      <alt><xsl:sequence select="$imageUrl"/></alt>
    </image>
  </xsl:template>
  
  <!-- ============================================
       Local functions
       ============================================ -->
  
  <xsl:function name="local:isMap" as="xs:boolean">
    <xsl:param name="context" as="element()"/>
    <xsl:variable name="styleName" as="xs:string"
      select="$context/@style"
    />
    <xsl:choose>
      <xsl:when test="$styleName = '' or $styleName = '[None]'">
        <xsl:sequence select="false()"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:variable name="styleMap" as="element()?"
          select="(key('styleMapsByName', lower-case($styleName), $styleMapDoc)[1],
          key('styleMapsById', $styleName, $styleMapDoc))[1]"
        />
        <xsl:sequence
          select="
          if ($styleMap)
          then ($styleMap/string(@structureType) = ('map', 'mapTitle')) or
               ($styleMap/string(@secondStructureType) = ( 'map', 'mapTitle'))
          else false()
          "
        />
      </xsl:otherwise>
    </xsl:choose>
  </xsl:function>
  
  <xsl:function name="local:isMapRoot" as="xs:boolean">
    <xsl:param name="context" as="element()"/>
    <xsl:variable name="styleName" as="xs:string"
      select="$context/@style"
    />
    <xsl:choose>
      <xsl:when test="$styleName = '' or $styleName = '[None]'">
        <xsl:sequence select="false()"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:variable name="styleMap" as="element()?"
          select="(key('styleMapsByName', lower-case($styleName), $styleMapDoc)[1],
          key('styleMapsById', $styleName, $styleMapDoc))[1]"
        />
        <xsl:sequence
          select="
          if ($styleMap)
          then (string($styleMap/@structureType) = 'map') or
               (string($styleMap/@secondStructureType) = 'map')
          else false()
          "
        />
      </xsl:otherwise>
    </xsl:choose>
  </xsl:function>
  
  <xsl:function name="local:isMapTitle" as="xs:boolean">
    <xsl:param name="context" as="element()"/>
    <xsl:variable name="styleName" as="xs:string"
      select="$context/@style"
    />
    <xsl:choose>
      <xsl:when test="$styleName = '' or $styleName = '[None]'">
        <xsl:sequence select="false()"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:variable name="styleMap" as="element()?"
          select="(key('styleMapsByName', lower-case($styleName), $styleMapDoc)[1],
          key('styleMapsById', $styleName, $styleMapDoc))[1]"
        />
        <xsl:sequence
          select="
          if ($styleMap)
          then (string($styleMap/@structureType) = 'mapTitle') or
               (string($styleMap/@secondStructureType) = 'mapTitle')
          else false()
          "
        />
      </xsl:otherwise>
    </xsl:choose>
  </xsl:function>
  
  <xsl:function name="local:isMapContentItem" as="xs:boolean">
    <!-- Determines if a paragraph generates a map content item,
         meaning a map, map title, topic group, topic head, or
         topic.
         
         This function is used in map generation so any paragraph
         that generates a topic must necessarily generate a topic
         document and a topicref as topics cannot go directly
         in map documents.
       -->
    <xsl:param name="context" as="element()"/>
    <xsl:variable name="isMapContentElement" as="xs:boolean"
      select="string($context/@structureType) = ('map', 'mapTitle', 'topicGroup', 'topicHead') or
              string($context/@secondStructureType) = ('map', 'mapTitle', 'topicGroup', 'topicHead')"
    />
    <xsl:variable name="isTopicref" as="xs:boolean"
      select="$context/@structureType = 'topicTitle' or
              $context/@secondStructureType = 'topicTitle'"
    />
    <xsl:sequence select="$isMapContentElement or $isTopicref"/>
  </xsl:function>
  
  <xsl:function name="local:isRootTopicTitle" as="xs:boolean">
    <xsl:param name="context" as="element()"/>
    <xsl:variable name="styleName" as="xs:string"
      select="$context/@style"
    />
    <xsl:choose>
      <xsl:when test="$styleName = '' or $styleName = '[None]'">
        <xsl:sequence select="false()"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:variable name="styleMap" as="element()?"
          select="
          (key('styleMapsByName', lower-case($styleName), $styleMapDoc)[1],
          key('styleMapsById', $styleName, $styleMapDoc))[1]"
        />
        <xsl:sequence
          select="
          if ($styleMap)
          then (($styleMap/@level = '0') and ($styleMap/@structureType = 'topicTitle'))
          else false()
          "
        />
      </xsl:otherwise>
    </xsl:choose>
    
  </xsl:function>
  
  <xsl:function name="local:isTopicTitle" as="xs:boolean">
    <xsl:param name="context" as="element()"/>
    <xsl:variable name="styleId" as="xs:string"
      select="($context/@styleName, $context/@styleId)"
    />
    <xsl:choose>
      <xsl:when test="not($styleId) or $styleId = '' or $styleId = '[None]'">
        <xsl:sequence select="false()"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:variable name="styleMap" as="element()?"
          select="
          (key('styleMapsByName', lower-case($styleId), $styleMapDoc)[1],
          key('styleMapsById', $styleId, $styleMapDoc)[1])[1]"
        />
        <xsl:sequence
          select="
          if ($styleMap)
            then ($styleMap/@structureType = 'topicTitle' or 
                  $styleMap/@secondStructureType = 'topicTitle')
            else false()
          "
        />
      </xsl:otherwise>
    </xsl:choose>    
  </xsl:function>
  
  <xsl:function name="local:getTopicType" as="xs:string">
    <xsl:param name="context" as="element()"/>
    <xsl:sequence select="local:getTopicType($context, $debugBoolean)"/>
  </xsl:function>
  
  <xsl:function name="local:getTopicType" as="xs:string">
    <xsl:param name="context" as="element()"/>
    <xsl:param name="doDebug" as="xs:boolean"/>
    
    <xsl:variable name="styleId" as="xs:string"
      select="($context/@styleName, $context/@styleId)[1]"
    />
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] local:getTopicType(): styleId="<xsl:value-of select="$styleId"/>"</xsl:message>
    </xsl:if>
    <xsl:choose>
      <xsl:when test="$styleId = '' or $styleId = '[None]'">
        <xsl:sequence select="'unknown-topic-type'"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:variable name="styleMap" as="element()?"
          select="(key('styleMapsByName', lower-case($styleId), $styleMapDoc)[1],
          key('styleMapsById', $styleId, $styleMapDoc)[1])[1]"
        />
        <xsl:if test="$doDebug">
          <xsl:message> + [DEBUG] local:getTopicType(): styleMap="<xsl:sequence select="$styleMap"/>"</xsl:message>
        </xsl:if>
        <xsl:variable name="topicType"
          select="
          if ($styleMap and $styleMap/@topicType)
          then string($styleMap/@topicType)
          else 'unknown-topic-type'
          "
          as="xs:string"
        />
        <xsl:if test="$doDebug">
          <xsl:message> + [DEBUG] local:getTopicType(): returning "<xsl:value-of select="$topicType"/>"</xsl:message>
        </xsl:if>
        <xsl:sequence select="$topicType"/>
      </xsl:otherwise>
    </xsl:choose>
    
  </xsl:function>

  <xsl:function name="local:getMapType" as="xs:string">
    <xsl:param name="context" as="element()"/>
    <xsl:variable name="styleId" as="xs:string"
      select="$context/@style"
    />
    <xsl:choose>
      <xsl:when test="$styleId = '' or $styleId = '[None]'">
        <xsl:sequence select="'unknown-map-type'"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:variable name="styleMap" as="element()?"
          select="(key('styleMapsByName', lower-case($styleId), $styleMapDoc)[1],
          key('styleMapsById', $styleId, $styleMapDoc)[1])[1]"
        />
        <xsl:sequence
          select="
          if ($styleMap and $styleMap/@mapType)
          then string($styleMap/@mapType)
          else 'unknown-map-type'
          "
        />
      </xsl:otherwise>
    </xsl:choose>
    
  </xsl:function>
  
  <xsl:function name="local:getResultUrlForTopic" as="xs:string">
    <xsl:param name="context" as="element()"/>
    <xsl:param name="mapUrl" as="xs:string"/>
    <xsl:param name="topicrefType" as="xs:string?"/>
    <xsl:param name="topicName" as="xs:string"/>
    <xsl:sequence 
      select="local:getResultUrlForTopic(
      $context, 
      $mapUrl, 
      $topicrefType, 
      $topicName, 
      $debugBoolean)"
    />
  </xsl:function>
  
  <xsl:function name="local:getResultUrlForTopic" as="xs:string">
    <xsl:param name="context" as="element()"/>
    <xsl:param name="mapUrl" as="xs:string"/>
    <xsl:param name="topicrefType" as="xs:string?"/>
    <xsl:param name="topicName" as="xs:string"/>
    <xsl:param name="doDebug" as="xs:boolean"/>
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] getResultUrlForTopic(): topicrefType=<xsl:value-of select="$topicrefType"/>, topicName=<xsl:value-of select="$topicName"/></xsl:message>
    </xsl:if>
    <xsl:variable name="topicRelativeUri" as="xs:string+">
      <xsl:apply-templates mode="topic-url" select="$context">
        <xsl:with-param name="topicrefType" as="xs:string?" select="$topicrefType"/>
        <xsl:with-param name="topicName" as="xs:string" select="$topicName"/>   
      </xsl:apply-templates>
    </xsl:variable>
    <!-- mapUrl is the URL of the map document -->
    <xsl:variable name="parentDir" select="relpath:getParent($mapUrl)" as="xs:string"/>
    <xsl:variable name="result" as="xs:string"
      select="relpath:newFile($parentDir, $topicRelativeUri)"
    />
    <xsl:sequence select="$result"/>
  </xsl:function>

  <xsl:function name="local:getResultUrlForMap" as="xs:string">
    <!-- Construct the absolute result URL for the map. -->
    <xsl:param name="context" as="element()"/>
    <xsl:param name="parentMapUrl" as="xs:string"/>
    <xsl:sequence select="local:getResultUrlForMap($context, $parentMapUrl, false())"/>
  </xsl:function>

  <xsl:function name="local:getResultUrlForMap" as="xs:string">
    <!-- Construct the absolute result URL for the map. -->
    <xsl:param name="context" as="element()"/>
    <xsl:param name="parentMapUrl" as="xs:string"/>
    <xsl:param name="doDebug" as="xs:boolean"/>
    
    <xsl:variable name="mapRelativeUri" as="xs:string+">
      <xsl:apply-templates mode="map-url" select="$context">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates>
    </xsl:variable>
    <xsl:variable name="result" as="xs:string"
      select="relpath:newFile(relpath:getParent($parentMapUrl), string-join($mapRelativeUri, ''))"
    />
    <xsl:sequence select="$result"/>
  </xsl:function>

  <xsl:function name="local:debugMessage">
    <xsl:param name="msg" as="xs:string"/>
    <xsl:message> + [DEBUG] <xsl:sequence select="$msg"/></xsl:message>
  </xsl:function>
  
  <xsl:function name="local:reportParas">
    <xsl:param name="paras" as="element()*"/>
    <xsl:for-each select="$paras">
      <xsl:sequence select="local:reportPara(.)"/>
      <xsl:text>&#x0a;</xsl:text>
    </xsl:for-each>
  </xsl:function>
  
  <xsl:function name="local:reportPara">
    <xsl:param name="para" as="element()?"/>
    <xsl:if test="$para">
      <xsl:sequence 
        select="concat('[', 
                       name($para),
                       ' ',
                       ' tagName=',
                       $para/@tagName,
                       if ($para/@level)
                          then concat(' level=', $para/@level)
                          else '',
                       if ($para/@containerType)
                          then concat(' containerType=', $para/@containerType)
                          else '',
                       if ($para/@structureType)
                          then concat(' structureType=', $para/@structureType)
                          else '',
                       if ($para/@secondStructureType)
                          then concat(' secondStructureType=', $para/@secondStructureType)
                          else '',
                       if ($para/@mapType)
                          then concat(' mapType=', $para/@mapType)
                          else '',
                       if ($para/@containerOutputclass)
                          then concat(' containerOutputclass=', $para/@containerOutputclass)
                          else '',
                          ']',
                          if ($para/self::rsiwp:topic)
                             then substring(normalize-space($para/*[1]), 1,40)
                             else substring(normalize-space($para), 1,40)
                       )"
      />
    </xsl:if>
  </xsl:function>
 
  <xsl:function name="local:constructSpectitle">
    <xsl:param name="context" as="element()"/>
    <xsl:variable name="specTitleSpec" select="string($context/@spectitle)"/>    
    <xsl:variable name="spectitle" as="xs:string">
      <xsl:choose>
        <xsl:when test="$specTitleSpec = '#toColon'">
          <xsl:sequence select="substring-before(string($context), ':')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:sequence select="string($context/@spectitle)"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:sequence select="$spectitle"/>
  </xsl:function>
  
  <xsl:function name="local:removeSpectitleContent" as="element()">
    <xsl:param name="context" as="element()"/>
    <xsl:variable name="initialText" as="xs:string" select="$context/text()[count(preceding-sibling::*) = 0]"/>
    <xsl:element name="{name($context)}"
      namespace="{namespace-uri($context)}"
      >
      <xsl:sequence select="$context/@*"/>
      <xsl:choose>
        <xsl:when test="string($context/@spectitle) = '#toColon'">
          <xsl:sequence select="substring-after($initialText, ': ')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:sequence select="string($initialText)"/>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:sequence select="$context/* | $context/text()[count(preceding-sibling::*) gt 0]"/>
    </xsl:element>
  </xsl:function>
  
  <xsl:function name="local:getContainerTypeSiblings" as="node()*">
    <!-- Gets the adjacent siblings following the last member of $sibs
         with the same container type.
         
         If there are no more siblings, returns $sibs.
       -->
    <xsl:param name="sibs" as="element()*"/>
    <xsl:variable name="lastSib" select="$sibs[last()]" as="element()"/>
    <xsl:variable name="nextSib" select="$lastSib/following-sibling::*[1]" as="element()?"/>
    <xsl:variable name="containerType" select="$lastSib/@containerType" as="xs:string?"/>
    <xsl:choose>
      <xsl:when test="$nextSib[@containerType = $containerType]">
        <xsl:sequence select="local:getContainerTypeSiblings(($sibs, $nextSib))"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:sequence select="$sibs"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:function>
  
  <xsl:function name="local:bookmarkRefToDitaRef" as="xs:string">
    <!-- Given a Word Hyperlink that is a reference to a bookmark, 
         convert the bookmark reference to a DITA reference (topic ID or topicID/elementId)
         if at all possible.
         
         Note that at the time we process the hyperlink there is no way to know what
         the final ID or location of the target element will be. Thus the best we can
         do here is to use the @wordlocation value as the reference. That will allow
         post processing to find the target by its word location and correct the
         href value to the appropriate URL and fragment identifier.
      -->
    <xsl:param name="hyperlinkElem" as="element()"/><!-- The SimpleML hyperlink element -->
    <xsl:param name="simpleWpDoc" as="document-node()"/>            
    

    <xsl:variable name="bookmarkName" as="xs:string" select="$hyperlinkElem/@href"/>
    <xsl:variable name="bookmark" select="key('bookmarksByName', $hyperlinkElem/@href, $simpleWpDoc)[1]"/>
<!--    <xsl:message> + [DEBUG] + bookmarkRefToDitaRef(): bookmark=<xsl:sequence select="$bookmark"/></xsl:message>-->
    <xsl:variable name="targetPara" as="element()?"
      select="$bookmark/ancestor::rsiwp:p[1]
      "
    />
<!--    <xsl:message> + [DEBUG] + bookmarkRefToDitaRef(): targetPara=<xsl:sequence select="$targetPara"/></xsl:message>-->
    <xsl:choose>
      <xsl:when test="$targetPara">
        <xsl:sequence select="concat('urn:wordlocation:', $targetPara/@wordLocation)"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:sequence select="concat('unresolvable reference to name ', $hyperlinkElem/@href)"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:function>
  
  <xsl:function name="local:constructImageFilenameForRsiwpImage" as="xs:string">
    <xsl:param name="imageElem" as="element(rsiwp:image)"/>
    <!-- Construct the image filename to use when constructing image references from
         rsiwp:image elements. 
      -->
    <xsl:variable name="result" select="$imageElem/@src" as="xs:string"/>
    <xsl:sequence select="$result"/>
  </xsl:function>

  <xsl:function name="local:constructColumnName" as="xs:string">
    <xsl:param name="colElement" as="element(rsiwp:col)"/>
    <xsl:variable name="result" select="concat('col', count($colElement/preceding-sibling::rsiwp:col) + 1)" as="xs:string"/>
    <xsl:sequence select="$result"/>
  </xsl:function>

  <!-- ============================================
       Catch-all templates
       ============================================ -->
  
  <xsl:template match="rsiwp:*" priority="-0.5" mode="p-content">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:message> + [WARNING] simple2dita[p-content]: Unhandled element <xsl:sequence select="name(..)"/>/<xsl:sequence select="name(.)"/></xsl:message>
  </xsl:template>
  
  <xsl:template match="rsiwp:*" priority="-0.5">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:message> + [WARNING] simple2dita[default mode]: Unhandled element <xsl:sequence select="name(..)"/>/<xsl:sequence select="name(.)"/></xsl:message>
  </xsl:template>
  
</xsl:stylesheet>
