<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:xs="http://www.w3.org/2001/XMLSchema"
  xmlns:xd="http://www.oxygenxml.com/ns/doc/xsl"
  xmlns:local="urn:functions:local"
  xmlns:xhtml="http://www.w3.org/1999/xhtml"
  xmlns="http://www.w3.org/1999/xhtml"
  xmlns:m="http://www.w3.org/1998/Math/MathML"
  xmlns:relpath="http://dita2indesign/functions/relpath"
  exclude-result-prefixes="xs xd local m xhtml relpath"
  version="2.0">
  <!-- Transform unnamespaced HTML docs into namespaced XHTML docs are required by the epub spec. 
  
       Also cleans up anything generated by the base Toolkit HTML transforms that is not allowed
       in EPUB XHTML.
      -->
  
  <xsl:template match="html | HTML | xhtml:xhtml" mode="html2xhtml" priority="10">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="topicref" as="element()?" tunnel="yes"/>
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] html2xhtml: root element <xsl:value-of select="name(.)"/></xsl:message>
      <xsl:message> + [DEBUG] html2xhtml:   Namespace: "<xsl:value-of select="namespace-uri(.)"/>"</xsl:message>
    </xsl:if>

    <xsl:variable name="lang" select="if ($topicref) then root($topicref)/*/@xml:lang else ''"/>
    <html>
      <xsl:choose>
        <xsl:when test="$lang != ''">
          <xsl:attribute name="xml:lang" select="$lang"/>
        </xsl:when>
        <xsl:otherwise/><!-- No lang attribute -->
      </xsl:choose>
      <xsl:apply-templates mode="#current" select="@*">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates>
      <xsl:if test="$doDebug">
        <xsl:comment> Generated by html2xhtml mode <xsl:value-of select="current-dateTime()"/></xsl:comment>
      </xsl:if>
      <xsl:apply-templates mode="#current" select="node()">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates>
    </html>
  </xsl:template>
  
  <xsl:template mode="html2xhtml" match="meta/@scheme">
    <!-- Suppress @scheme on <meta> because @scheme is not allowed there per epubcheck -->
  </xsl:template>
  
  <xsl:template mode="html2xhtml" match="meta[@http-equiv = 'Content-Type'] | xhtml:meta[@http-equiv = 'Content-Type']">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] html2xhtml: Suppressing meta element: <xsl:sequence select="."/></xsl:message>
    </xsl:if>
  </xsl:template>
  
  <xsl:template mode="html2xhtml" match="link[@rel = 'stylesheet' and  @type='text/css']">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="resultUri" as="xs:string" tunnel="yes"/>
        
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] html2xhtml: handling a link to a CSS style sheet...</xsl:message>
    </xsl:if>
    
    <!-- Rewrite the CSS pointer to reflect the actual location of this file. -->

    <xsl:variable name="oldHref" as="xs:string" select="@href"/>
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] html2xhtml:   oldHref="<xsl:value-of select="$oldHref"/>"</xsl:message>
    </xsl:if>
    <xsl:variable name="absPathToCssDir" as="xs:string"
      select="relpath:newFile($outdir, $cssOutDir)"
    />
    
    <xsl:variable name="relPathToCssDir" as="xs:string"
      select="relpath:getRelativePath(relpath:getParent($resultUri), $absPathToCssDir)"
    />
    <xsl:variable name="newHref" as="xs:string"
      select="if ($relPathToCssDir != '') 
                 then relpath:newFile($relPathToCssDir, relpath:getName($oldHref))
                 else $oldHref"
    />
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] html2xhtml:   newHref="<xsl:value-of select="$newHref"/>"</xsl:message>
    </xsl:if>
    <xsl:element name="{name(.)}">
      <xsl:apply-templates select="@* except @href" mode="#current"/>
      <xsl:attribute name="href" select="$newHref"/>
      <xsl:apply-templates mode="#current" select="node()"/>
    </xsl:element>
  </xsl:template>

  <xsl:template mode="html2xhtml" match="script/@src | xhtml:script/@src">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:param name="resultUri" as="xs:string" tunnel="yes"/>
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] html2xhtml: handling a script @src attribute: "<xsl:value-of select="."/>"</xsl:message>
    </xsl:if>
    
    <!-- Rewrite the pointer to reflect the actual location of this file. -->
    
    <xsl:variable name="oldSrc" as="xs:string" select="."/>
    <xsl:variable name="relPathToJsDir" as="xs:string"
      select="relpath:getRelativePath(relpath:getParent($resultUri), $javascriptOutputPath)"
    />
    <xsl:variable name="newSrc" as="xs:string"
      select="if ($relPathToJsDir != '') 
      then relpath:newFile($relPathToJsDir, relpath:getName($oldSrc))
      else $oldSrc"
    />
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] html2xhtml:   newSrc="<xsl:value-of select="$newSrc"/>"</xsl:message>
    </xsl:if>
    <xsl:attribute name="{name(.)}" select="$newSrc"/>
  </xsl:template>
  

  <xsl:template mode="html2xhtml" match="img[not(@alt)] | xhtml:img[not(@alt)]">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>

    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] html2xhtml: img with no @alt</xsl:message>
    </xsl:if>

    <xsl:element name="{name(.)}">
      <xsl:attribute name="alt" select="@src"/>
      <xsl:apply-templates select="@*,node()" mode="#current">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates>
    </xsl:element>
  </xsl:template>
  
  <xsl:template mode="html2xhtml" match="@xml:base"/><!-- Don't include in output -->

  <xsl:template mode="html2xhtml" match="math | m:math" xmlns="http://www.w3.org/1998/Math/MathML">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>

    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] html2xhtml: math or m:math element</xsl:message>
    </xsl:if>

    <xsl:element name="{name(.)}">
      <xsl:apply-templates select="@*,node()" mode="#current">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates>
    </xsl:element>
  </xsl:template>
  
  <xsl:template match="m:*" mode="html2xhtml" priority="-0.5" >
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] html2xhtml: <xsl:value-of select="concat(name(..), '/', name(.))"/></xsl:message>
    </xsl:if>
     <xsl:element name="{name(.)}" namespace="http://www.w3.org/1998/Math/MathML">
        <xsl:apply-templates select="@*,node()" mode="#current">
          <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
        </xsl:apply-templates>
     </xsl:element>    
  </xsl:template>
  
  <!-- <a> elements used for IDs are not used in XHTML -->
  <xsl:template mode="html2xhtml" 
    match="*[      not(@id) and a[@name and not(@href)]] | 
                       xhtml:*[not(@id) and xhtml:a[@name and not(@href)]]">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] </xsl:message>
    </xsl:if>
    <!-- Move anchor ID to containing element. -->
    <xsl:element name="{name(.)}">
      <xsl:apply-templates select="@*" mode="#current"/>
      <xsl:attribute name="id" 
        select="(xhtml:a[@name and not(@href)][1] | a[@name and not(@href)][1])[1]/@name"
      />
      <xsl:apply-templates mode="#current" select="node()"/>
    </xsl:element>
  </xsl:template>
  
  <xsl:template match="a[@name and not(@href)] | xhtml:a[@name and not(@href)]" priority="10" mode="html2xhtml">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
  </xsl:template>
  
  <xsl:template match="a/@name | xhtml:a/@name" mode="html2xhtml" priority="10">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:attribute name="id" select="string(.)"/>
  </xsl:template>
  
  
  <xsl:template match="a[@href] | xhtml:a[@href]" priority="20" mode="html2xhtml">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:variable name="newHref" select="@href" as="xs:string"/>
    <a>
      <xsl:attribute name="href" select="$newHref"/>
      <xsl:apply-templates select="@*,node()" mode="#current">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates>
    </a>
  </xsl:template>
  
  <xsl:template match="blockquote | xhtml:blockquote" priority="20" mode="html2xhtml">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>

    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] html2xhtml: math or m:math element</xsl:message>
    </xsl:if>

    <blockquote>
      <xsl:apply-templates select="@*" mode="#current">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates>
      <xsl:for-each-group select="*|text()" 
        group-adjacent="local:getBlockOrInlineGroupingKey(.)">
        <xsl:choose>
          <xsl:when test="current-grouping-key() = 'inline'">
            <xsl:if test="normalize-space(.) != '' and normalize-space(.) != ' '">
              <p><xsl:apply-templates select="current-group()" mode="#current">
                <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
              </xsl:apply-templates></p>
            </xsl:if>
          </xsl:when>
          <xsl:otherwise>
            <xsl:apply-templates select="current-group()" mode="#current">
              <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
            </xsl:apply-templates>
          </xsl:otherwise>
        </xsl:choose>        
      </xsl:for-each-group>
    </blockquote>
  </xsl:template>  
  
  <xsl:template mode="html2xhtml" match="*" priority="-2">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>

    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] html2xhtml: * : <xsl:sequence select="concat(name(..), '/', name(.))"/></xsl:message>
    </xsl:if>
    <xsl:element name="{name(.)}">
      <xsl:apply-templates select="@*,node()" mode="#current">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates>
    </xsl:element>
  </xsl:template>
  
  <xsl:template mode="html2xhtml" match="xhtml:*" priority="-1">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>

    <xsl:if test="$doDebug">
      <xsl:message> + [DEBUG] html2xhtml: xhtml:* : <xsl:sequence select="concat(name(..), '/', name(.))"/></xsl:message>
    </xsl:if>
    <xsl:element name="{name(.)}">
      <xsl:apply-templates select="@*,node()" mode="#current">
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates>
    </xsl:element>
  </xsl:template>
  
  <xsl:template match="u | xhtml:u" priority="10" mode="html2xhtml">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <!-- DITA <u> (underline element) -->
    <span class="underline" style="text-decoration: underline"><xsl:apply-templates mode="#current">
      <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
    </xsl:apply-templates></span>
  </xsl:template>
  
  <xsl:template match="span/p | xhtml:span/xhtml:p" priority="10" mode="html2xhtml">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <!-- Paragraphs not allowed within span -->
    <xsl:apply-templates mode="#current">
      <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
    </xsl:apply-templates>
  </xsl:template>
  
  <xsl:template match="p/div | xhtml:p/xhtml:div" priority="10" mode="html2xhtml">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <span>
      <xsl:apply-templates select="@*,node()" mode="#current"
      >
        <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
      </xsl:apply-templates></span>
  </xsl:template>
  
  <xsl:template match="tt | xhtml:tt" mode="html2xhtml">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <code><xsl:apply-templates mode="#current">
      <xsl:with-param name="doDebug" as="xs:boolean" tunnel="yes" select="$doDebug"/>
    </xsl:apply-templates></code>
  </xsl:template>
  
  <xsl:template mode="html2xhtml" match="@headers">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <!-- Ensure no leading or trailing whitespace, which throws off epubcheck. -->
    <xsl:attribute name="{name(.)}" select="normalize-space(.)"/>
  </xsl:template>
  
  <xsl:template  mode="html2xhtml" 
    match="img/@width | img/@height | xhtml:img/@width | xhtml:img/@height" priority="100">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <!--  Suppress for now because of issue with ImgUtils not working and generating
          bad values for height and width. -->
    <xsl:variable name="length" as="xs:string" select="."/>
    <xsl:choose>
      <xsl:when test="starts-with($length, '-')">
        <xsl:message> + [WARN] Value "<xsl:sequence select="$length"/>" for <xsl:sequence select="name(..)"/>/@<xsl:sequence select="name(.)"/> is negative. This reflects a bug in the Open Toolkit.</xsl:message>
        <xsl:message> + [WARN]   Suppressing attribute in HTML output.</xsl:message>
      </xsl:when>
      <xsl:otherwise>
        <!-- GitHub issue #75: For EPUB there should be no units on pixel values. 
            
             Alternative would be to put the width and height into @style. Not
             trying to do that for the initial solution.
        -->
        <xsl:attribute name="{name(.)}" select="$length"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  
  <xsl:template  mode="html2xhtml" match="video/@width | xhtml:video/@width" priority="20">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:copy/>  
  </xsl:template>
  
  <xsl:template  mode="html2xhtml" match="script/@type | xhtml:script/@type" priority="30">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
  	<xsl:copy/>
  </xsl:template>
  
  <xsl:template  mode="html2xhtml" match="@type[.='text/css']" priority="20">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:copy/>
  </xsl:template>
  
  <xsl:template  mode="html2xhtml" match="
    @lang |
    @target |
    @compact |
    @width |
    @type |
    @align |
    @valign |
    @rules |
    @frame |
    @summary |
    @cellspacing |
    @cellpadding |
    @xxx
    " priority="10">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>    
  </xsl:template>
  
  <xsl:template mode="html2xhtml" match="@*|text()|processing-instruction()|comment()">
    <xsl:param name="doDebug" as="xs:boolean" tunnel="yes" select="false()"/>
    <xsl:copy-of select="."/>
  </xsl:template>
  
  <xsl:function name="local:isBlock" as="xs:boolean">
    <xsl:param name="context" as="node()"/>
    <xsl:variable name="result" as="xs:boolean"
      select="
      $context/self::address or
      $context/self::blockquote or
      $context/self::del or
      $context/self::div or
      $context/self::dl or
      $context/self::fieldset or
      $context/self::form or
      $context/self::h1 or
      $context/self::hr or
      $context/self::ins or
      $context/self::noscript or
      $context/self::ol or
      $context/self::p or
      $context/self::pre or
      $context/self::script or
      $context/self::table or
      $context/self::ul
      "
    />
    <xsl:sequence select="$result"/>    
  </xsl:function>
    
  <xsl:function name="local:getBlockOrInlineGroupingKey" as="xs:string">
    <xsl:param name="context" as="node()"/>
    <xsl:choose>
      <xsl:when 
        test="local:isBlock($context)">
        <xsl:sequence select="'block'"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:sequence select="'inline'"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:function>
  
  <xsl:template match="nav | section | figure | article" mode="html2xhtml" priority="20">
    <div>
      <xsl:apply-templates select="@* except (@role, @aria-labelledby) | node()" mode="html2xhtml"/>
    </div>
  </xsl:template>
  
  <!-- Group for root document node does not need extra XHTML div -->
  <xsl:template match="main/article" mode="html2xhtml" priority="30">
    <xsl:apply-templates select="node()" mode="html2xhtml"/>
  </xsl:template>
  
  <xsl:template match="header | footer | main" mode="html2xhtml" priority="20">
    <xsl:apply-templates select="node()" mode="html2xhtml"/>
  </xsl:template>
  
  <xsl:template match="div/@role" mode="html2xhtml" priority="10"/>
</xsl:stylesheet>
