<?xml version="1.0" encoding="UTF-8"?>
<!-- 
    Generates different FO elements based on the css:display attribute:
    
    css:display="block" -> fo:block
    css:display="inline" -> fo:inline
    etc..

	The @css:link attribute is examined to generate a wrapping element around the fo:inline and fo:block.
        
-->
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema" 
    xmlns:fo="http://www.w3.org/1999/XSL/Format"
    xmlns:oxy="http://www.oxygenxml.com/css2fo" 
    xmlns:css="http://www.w3.org/1998/CSS"
    xmlns:fox="http://xmlgraphics.apache.org/fop/extensions"
    exclude-result-prefixes="xs oxy css fox" version="2.0">

    <!-- Fast search for IDs. -->
    <xsl:key name="ids" match="*" use="@id | @xml:id"/>
    <xsl:key name="html-names" match="*:html//*:a" use="@name"/>

    <!-- 
    We can have links only in inline context, on text, or other static content. 
    Simply wrap the elements in a "fo:basic-links" if their parent has a @css:link attribute. 
    However, we cannot use "fo:basic-link" to wrap the "fo:footnote-call" because "fo:basic-link" is not a valid child of "fo:footnote"
    -->
    <xsl:template match="
            css:*[../@css:link][not(../@css:float = 'footnote')]
            |
            text()[../@css:link and string-length(normalize-space(.)) != 0]" priority="5">

        <xsl:choose>
            <xsl:when test="string-length(normalize-space(../@css:link)) > 0">
                <fo:basic-link>
                    <xsl:call-template name="generate-link-attributes"/>
                    <xsl:next-match/>
                </fo:basic-link>
            </xsl:when>
            <xsl:otherwise>
                <!-- Empty link -->
                <xsl:next-match/>
            </xsl:otherwise>
        </xsl:choose>

    </xsl:template>

    <xsl:template match="
            css:*[../@css:link][not(../@css:float = 'footnote')]
            |
            text()[../@css:link and string-length(normalize-space(.)) != 0]" priority="10">

        <xsl:variable name="nm">
            <xsl:next-match/>
        </xsl:variable>
        <xsl:apply-templates select="$nm" mode="add-alt-text"/>
    </xsl:template>

    <xsl:template match="node() | @*" mode="add-alt-text">
        <xsl:copy>
            <xsl:apply-templates select="node() | @*" mode="#current"/>
        </xsl:copy>
    </xsl:template>

    <xsl:template match="fo:basic-link[not(ancestor::fo:basic-link)]" mode="add-alt-text">
        <xsl:copy>
            <xsl:copy-of select="@*"/>
            <xsl:if test="not(./@fox:alt-text)">
                <xsl:choose>
                    <xsl:when test=".//@fox:alt-text">
                        <!-- children have alt-text they will be used -->
                        <xsl:variable name="alt-text-from-children" select="normalize-space(string-join(.//@fox:alt-text, ' '))"/>
                        <xsl:attribute name="fox:alt-text" select="
                                if (string-length($alt-text-from-children) != 0) then
                                    $alt-text-from-children
                                else
                                    ' '"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <!-- children does not have alt-text we use the text from the node -->
                        <xsl:variable name="text-from-node" select="normalize-space(string-join(.//text()))"/>
                        <xsl:attribute name="fox:alt-text" select="
                                if (string-length($text-from-node) != 0) then
                                    $text-from-node
                                else
                                    ' '"/>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:if>
            <xsl:apply-templates mode="#current"/>
        </xsl:copy>
    </xsl:template>

    <!-- A link to a footnote should result in a footnote call. -->
    <xsl:template match="*[@css:link][key('ids', oxy:adjust-if-local-link(@css:link))/@css:float = 'footnote']">
        <xsl:variable name="fn-link" select="oxy:adjust-if-local-link(@css:link)"/>
        <xsl:variable name="fn-is-first-link" select="count(./preceding::*[oxy:adjust-if-local-link(@css:link) = $fn-link]) &lt; 1"/>
        <xsl:for-each select="key('ids', oxy:adjust-if-local-link(@css:link))">
            <xsl:call-template name="generate-footnote-call-for-link">
                <xsl:with-param name="fn-is-first-link" select="$fn-is-first-link" tunnel="yes"/>
            </xsl:call-template>
        </xsl:for-each>
    </xsl:template>

    <!-- 
        Generates the fo:basic-link attributes.
        
        Context: Any node that has a @link attribute.
    -->
    <xsl:template name="generate-link-attributes">
        <xsl:call-template name="generate-link-attributes-generic">
            <xsl:with-param name="doc" select="/"/>
            <xsl:with-param name="link-attr-val" select="../@css:link"/>
            <xsl:with-param name="link-alt-text-attr-val" select="../@css:oxy-alt-text"/>
        </xsl:call-template>
    </xsl:template>

    <!-- 
        Generates the fo:basic-link attributes.
        
        Context: none
    -->
    <xsl:template name="generate-link-attributes-generic">
        <xsl:param name="doc"/>
        <xsl:param name="link-attr-val"/>
        <xsl:param name="link-alt-text-attr-val"/>

        <xsl:attribute name="show-destination" select="'new'"/>
        <xsl:attribute name="indicate-destination" select="'true'"/>

        <xsl:if test="$link-alt-text-attr-val">
            <xsl:attribute name="fox:alt-text" select="$link-alt-text-attr-val"/>
        </xsl:if>

        <!-- Deals with special case in which the link is local, and starts with a hash. -->
        <xsl:variable name="link" select="oxy:adjust-if-local-link($link-attr-val)"/>

        <xsl:choose>
            <xsl:when test="$doc/key('ids', $link) or $doc/key('html-names', $link)">
                <!-- Is an internal link, there is an ID matching it in the document. -->
                <xsl:attribute name="internal-destination" select="$link"/>
            </xsl:when>
            <xsl:otherwise>
                <!-- Is an external link. -->
                <xsl:attribute name="external-destination" select="$link"/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <!--
  	  If the parameter is a local link, removes the # from start, if any.
  	  
  	  @param  link-attr-val The value of the link attribute
  	  @return a string, the link value.
  	-->
    <xsl:function name="oxy:adjust-if-local-link" as="xs:string?">
        <xsl:param name="link-attr-val"/>
        <xsl:value-of select="
            if (starts-with($link-attr-val, '#')) then
                substring($link-attr-val, 2)
            else
                $link-attr-val"/>
    </xsl:function>

</xsl:stylesheet>
