<?xml version="1.0" encoding="UTF-8"?>
<!--
    This stylesheet deals with footnotes. 
    The elements with the float: footnote are converted to a fo:footnote element.    
-->
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    xmlns:css="http://www.w3.org/1998/CSS"
    xmlns:fo="http://www.w3.org/1999/XSL/Format" 
    xmlns:oxy="http://www.oxygenxml.com/css2fo"
    
    exclude-result-prefixes="xs oxy"
    version="2.0">
    
    <!-- Do not copy the 'float' attribute. This is converted into XSL-FO footnote structure. -->
    <xsl:template match="@css:float[.='footnote']" priority="4"/>
    
    <!-- 
      This template is similar to template matching the footnote, but generates only
      the footnote call. This template is called from the links that point to a footnote.
      The content of the link element is discarded, replaced with the call. The context of this 
      template is the footnote element. 
     -->    
    <xsl:template name="generate-footnote-call-for-link">
    
        <xsl:variable name="number-and-id">
          <xsl:call-template name="get-footnote-number-and-id"/>
        </xsl:variable>
        
        <xsl:variable name="fn-id" select="$number-and-id/fn-id"/>
        <xsl:variable name="fn-number" select="$number-and-id/fn-number"/>                    
      
        <xsl:for-each select="css:footnote-call">
            <xsl:call-template name="generate-footnote-call">
                <xsl:with-param name="fn-id" select="$fn-id"/>            
                <xsl:with-param name="fn-number" select="$fn-number" tunnel="yes"/>                    
                <xsl:with-param name="fn-call-from-link" select="true()"/>                    
            </xsl:call-template>
        </xsl:for-each>
    </xsl:template>
    
    
    <xsl:template match="*[@css:float='footnote'][@css:display != 'none']" >

		<!-- 
			Forces the output of the footnote, even if it is in an 
		    illegal place in the DOM. The output will be placed after the table.
		-->
    	<xsl:param name="generate-footnotes-within-table-structure" tunnel="yes"/>
        
        <xsl:choose>
            <xsl:when test="$generate-footnotes-within-table-structure or not(oxy:is-table-structure(..))">

                <fo:footnote 
                	text-align="start" 
                	text-align-last="start" 
                	font-style="normal" 
                	font-weight="400" 
                	start-indent="0" 
                	end-indent="0">

                    <xsl:variable name="number-and-id">
                      <xsl:call-template name="get-footnote-number-and-id"/>
                    </xsl:variable>
                    
                    <xsl:variable name="fn-id" select="$number-and-id/fn-id"/>
                    <xsl:variable name="fn-number" select="$number-and-id/fn-number"/>                    
                  
                    <!--  If is a referred footnote, the footnote call will be presented by the reference. -->
                    <xsl:for-each select="css:footnote-call">
                        <xsl:call-template name="generate-footnote-call">
                            <xsl:with-param name="fn-id" select="$fn-id"/>    
                            <xsl:with-param name="fn-number" select="$fn-number" tunnel="yes"/>
                        </xsl:call-template>
                    </xsl:for-each>
                    
                    <xsl:variable name="list-style-position" select="oxy:get-list-style-position(css:footnote-marker)"/>
                    
                    <xsl:variable name="fn-marker-width" select="oxy:get-footnote-marker-width(., $fn-number)"/>
                    <fo:footnote-body>
                        <fo:list-block 
                            relative-align="baseline"
                            provisional-label-separation="{if ($list-style-position = 'outside') then '0pt' else '0pt'}"
                            provisional-distance-between-starts="{if ($list-style-position = 'outside') then $fn-marker-width else '0'}">                                        
                            
                            <xsl:call-template name="list-item">
                                <xsl:with-param name="list-item" select="."/>
                                <xsl:with-param name="list-style-position" select="$list-style-position"/>
                                <xsl:with-param name="marker">
                                    <xsl:for-each select="css:footnote-marker">
                                        <xsl:call-template name="generate-footnote-marker">
                                            <xsl:with-param name="fn-number" select="$fn-number" tunnel="yes"/>
                                            <xsl:with-param name="fn-id" select="$fn-id"/>
                                            <xsl:with-param name="as-block" select="$list-style-position = 'outside'"/>
                                        </xsl:call-template>                                
                                    </xsl:for-each>
                                </xsl:with-param>
                                <xsl:with-param name="content">
                                    <xsl:choose>
                                        <xsl:when test="@css:display = 'table'">
                                            <!-- Generate the table. -->
                                            <xsl:next-match/>
                                        </xsl:when>
                                        <xsl:otherwise>
                                            <xsl:apply-templates select="*|text()"/>
                                        </xsl:otherwise>
                                    </xsl:choose>
                                </xsl:with-param>                        
                            </xsl:call-template>
                            
                        </fo:list-block>
                    </fo:footnote-body>        
                </fo:footnote>
                
                
            </xsl:when>
            <xsl:otherwise>
                <!--  The footnote will be placed after the table, at a latter time.  -->                
            </xsl:otherwise>
        </xsl:choose>

    </xsl:template>
    
    <!--
        Gets a structure with the footnote number and its id.
         
        We need the footnote number and its format in order to 
        provision space for the footnote marker.
         
        For this, select all pages-sequences or elements that have a counter-reset, 
        take the one that is closest in the document order to the footnote,
        and count the footnotes between them and the footnote.         
     -->
    <xsl:template name="get-footnote-number-and-id">
                   
          <!-- All pages-sequences or elements that have a counter-reset -->
          <xsl:variable name="one-that-resets-counter" select="
              (
              preceding::css:page-sequence[oxy:get-footnote-counter-reset-from-page(/,@page) >=0 ] union
              preceding::*[@css:counter-reset][oxy:get-footnote-counter-reset-from-element(@css:counter-reset) >=0 ] union
              ancestor-or-self::css:page-sequence[oxy:get-footnote-counter-reset-from-page(/,@page) >= 0] union
              ancestor-or-self::*[@css:counter-reset][oxy:get-footnote-counter-reset-from-element(@css:counter-reset) >=0 ]
              )
              [position() = last()]"/>

                  
          <fn-number>
              <xsl:choose>
                  <xsl:when test="$one-that-resets-counter">
                      
                      <!-- Count all the footnotes that are between the the page-sequence or the document element 
                       that resets the counter and the current footnote, including the current footnote. -->
                      <xsl:variable name="fn" select="."/>
                      <xsl:variable name="no">
                          <xsl:number count="*[@css:float='footnote']
                              [not(following::*[generate-id(.) = generate-id($one-that-resets-counter)])]
                              [not(preceding::*[@css:float='footnote'][generate-id(.) = generate-id($fn)])]" 
                              level="any"/>
                      </xsl:variable>
                      
                      
                      <xsl:choose>
                          <xsl:when test="local-name($one-that-resets-counter) = 'page-sequence'">
                              <!-- Extract the reset value from the page definition, and add it to the number of counted footnotes. -->
                              <xsl:value-of select="oxy:get-footnote-counter-reset-from-page(/,$one-that-resets-counter/@page) + $no - 1"/>                               
                          </xsl:when>
                          <xsl:otherwise>
                              <!-- Extract the reset value from the element styles, and add it to the number of counted footnotes. -->
                              <xsl:value-of select="oxy:get-footnote-counter-reset-from-element($one-that-resets-counter/@css:counter-reset) + $no - 1"/>
                          </xsl:otherwise>
                      </xsl:choose>
                      
                  </xsl:when>
                  <xsl:otherwise>
                      <xsl:number count="*[@css:float='footnote']" level="any"/>                        
                  </xsl:otherwise>
              </xsl:choose>
          </fn-number>
            
          <!-- We do not use the footnote number as an id, because it can be reset, so we can get duplicated id-s. -->
          <fn-id>
              <xsl:number count="*[@css:float='footnote']" level="any"/>
          </fn-id>            
                    
     </xsl:template>
    
    <!-- 
        Gets the 'footnote' counter reset value set on a page declaration. 
    
        @param doc The document
        @param page-name The name of the page.
        @return a number if there is a counter-reset property that refers to the 'footnote' counter, 
        or an empty string if the page does not reset the counters.
    -->
    <xsl:function name="oxy:get-footnote-counter-reset-from-page" as="xs:integer">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>

        <xsl:sequence select="oxy:get-counter-reset-from-page-def($doc, $page-name, 'footnote')"/>
    </xsl:function>
    
    
    <!--        
        Gets the 'footnote' counter reset value set on an element. 
    
        @param counter-reset The value of the 'counter-reset' property. 
        @param counter-name The name of the counter. May be 'page', or 'footnote', etc.. 
        @return a number if there is a counter-reset property that refers to the 'footnote' counter, 
                or -1 otherwise.
    -->
    <xsl:function name="oxy:get-footnote-counter-reset-from-element" as="xs:integer">
        <xsl:param name="counter-reset-prop-val" />        
        <xsl:sequence select="oxy:get-counter-reset($counter-reset-prop-val, 'footnote')"/>
    </xsl:function>
    
    
    <!-- 
    
        Generates the :footnote-call elements.
    
    -->
    <xsl:template match="css:footnote-call"/>
    <xsl:template name="generate-footnote-call">
        <xsl:param name="fn-id" as="xs:string"/>
        <xsl:param name="fn-call-from-link" as="xs:boolean" select="false()"/>
        <xsl:param name="fn-is-first-link" as="xs:boolean" select="true()" tunnel="yes"/>

        <!-- Link between the call and the marker -->

        <!-- The link is placed into an inline, it cannot be added in the flow directly. 
             And we need the link outside the superscript text, because otherwise it increases the line
             even if the @line-height-shift-adjustment="disregard-shifts" is set on the page-sequence.-->
        <fo:inline>
            <xsl:if test="@css:display != 'none' or $fn-call-from-link">
                <fo:basic-link internal-destination="footnote-marker-{$fn-id}">
                    <xsl:if test="$fn-is-first-link">
                        <xsl:attribute name="id">footnote-call-<xsl:value-of select="$fn-id"/></xsl:attribute>
                    </xsl:if>
                    <!-- The real "foonote-call" -->
                    <fo:inline>
                        <xsl:apply-templates select="@*"/>
                        <xsl:apply-templates/>
                    </fo:inline>
                </fo:basic-link>
            </xsl:if>
        </fo:inline>
          
    </xsl:template>

    <!-- 
    
        Generates the :footnote-marker elements, with a link to the footnote call.
        
        @param as-block If this is true, the generated element will be a fo:block.
        @param fn-number The footnote number.
    
    -->
    <xsl:template match="css:footnote-marker"/>        
    <xsl:template name="generate-footnote-marker">
        <xsl:param name="fn-id" as="xs:string"/>
        <xsl:param name="as-block" as="xs:boolean"/>
        
        
        <xsl:choose>
            <xsl:when test="$as-block">
                <fo:block>
				    <xsl:apply-templates select="../@css:*" mode="copy-marker-properties-from-parent"/>
				    
				    <!-- Deal with vertical align by setting it on an inline. Otherwise has no effect. -->
                    <xsl:apply-templates select="@* except @css:vertical-align"/>
                    
			        <!-- Link between the marker and the call -->
			        <fo:basic-link internal-destination="footnote-call-{$fn-id}" id="footnote-marker-{$fn-id}">
			        
	                    <xsl:choose>
	                    	<xsl:when test="@css:vertical-align">
			                    <fo:inline>
			                    	<!-- Place here the vertical align. -->
				                    <xsl:apply-templates select="@css:vertical-align"/>
				                    <xsl:apply-templates/>
			                    </fo:inline>
			            	</xsl:when>
			            	<xsl:otherwise>
				                    <xsl:apply-templates/>
			            	</xsl:otherwise>
	                    </xsl:choose>
	                    
			        </fo:basic-link>

                </fo:block>
            </xsl:when>
            <xsl:otherwise>
                <fo:inline>
                    <xsl:apply-templates select="@*"/>
	                
	                <!-- Link between the marker and the call -->
	                <fo:basic-link internal-destination="footnote-call-{$fn-id}" id="footnote-marker-{$fn-id}">
	                    <xsl:apply-templates/>
	                </fo:basic-link>
                </fo:inline>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    
    <!-- 
        Evaluates the footnotes counter.
        
        @param fn-number The footnote number. This is set from the element that has 
        float:footnote. If there is no such parent, then the footnote counter was used 
        directly from some arbitrary static content, not necessarily linked to the footnotes. 
        In this case, simply count all the footnotes.
    -->
    <xsl:template match="css:counter[@name='footnote']">
        <xsl:param name="fn-number" as="xs:string" tunnel="yes" select="@value"/>
        
        <xsl:choose>
            <xsl:when test="$fn-number">
                <xsl:value-of select="$fn-number"/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:choose>
                    <xsl:when test="@format">
                        <xsl:number count="*[@css:float='footnote']" format="{oxy:get-number-format(@format)}" level="any"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:number count="*[@css:float='footnote']" level="any"/>
                    </xsl:otherwise>
                </xsl:choose>                    
            </xsl:otherwise>
        </xsl:choose>        
    
    </xsl:template>
    
    
    
    <!--
        Estimates the footnote marker width by computing the number of symbols it will need to display,
        and computing the margin - border - padding around the text.
        
        @param fn The footnote element.
        @param fn-number The footnote number.
        
        @return An expression like "4pt + 1em + 2pt".
    -->
    <xsl:function name="oxy:get-footnote-marker-width" as="xs:string">
        <xsl:param name="fn"/>
        <xsl:param name="fn-number"/>
        
        <!-- The footnote format. -->
        <xsl:variable name="fn-format" select="$fn/css:footnote-marker//counter[@name='footnote']/@format"/>
        <xsl:variable name="fn-format" select="if ($fn-format) then $fn-format else 'decimal'"/>
        
        <!-- Get the padding. -->
        <xsl:variable name="fn-marker-mbp">
            <xsl:for-each select="
                $fn/css:footnote-marker/@css:margin-left | 
                $fn/css:footnote-marker/@css:margin-right | 
                $fn/css:footnote-marker/@css:padding-left | 
                $fn/css:footnote-marker/@css:padding-right">
                
                <xsl:value-of select="."/>
                <xsl:if test="not(position() = last())"> + </xsl:if>
            </xsl:for-each>
        </xsl:variable>
        
        <!-- Use the marker width from CSS, otherwise compute one. -->
        <xsl:variable name="fn-marker-width" select="
            if ($fn/css:footnote-marker/@css:width) then  
                $fn/css:footnote-marker/@css:width  
            else 
                if (string-length(normalize-space($fn-marker-mbp)) > 0) then 
                    concat(oxy:get-number-width($fn-number, $fn-format),' + ', $fn-marker-mbp) 
                else 
                    oxy:get-number-width($fn-number, $fn-format)" />
        
        <xsl:value-of select="$fn-marker-width"/>
    </xsl:function>
    
    
    
    <!-- 
      Checks if the display of the node is related to table structure.
       
      @param node The node to check its display.
      @retrurn true If the displaty is related to table structures (rows, groups). 
      If the node is a 'table-cell', returns false.     
    -->
    <xsl:function name="oxy:is-table-structure" as="xs:boolean">
      <xsl:param name="node" as="item()"/>
      <xsl:variable name="display"  select="$node/@css:display"/>
      <xsl:choose>
        <xsl:when test="
            $display = 'table-column' or 
            $display = 'table-column-group' or
            $display = 'table-footer-group' or
            $display = 'table-header-group' or
            $display = 'table-row' or 
            $display = 'table-row-group' or  
            $display = 'table'">
          <xsl:sequence select="true()"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:sequence select="false()"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:function>
  
  
    <!--       
    	We do not support footnotes in tables, between rows, or table sections. 	
    	We move them after the table.
     -->
	 <xsl:template match="*[@css:display = 'table'][.//*[@css:float='footnote'][oxy:is-table-structure(..)]]" priority="10">
	 	<xsl:variable name="table" select="."/>
	 
	 	 <xsl:next-match>
	     	<xsl:with-param name="generate-footnotes-within-table-structure" select="false()" tunnel="yes"/>
	     </xsl:next-match>
	      
		 <xsl:for-each select=".//*[@css:float='footnote'][oxy:is-table-structure(..)][ancestor::*[@css:display = 'table'][1] = $table]">
		  	  <xsl:message>WARNING Moving footnote "<xsl:value-of select="text()"/>" after the parent table. Footnotes within table structure elements are not supported.</xsl:message>
			  <xsl:apply-templates select=".">
			  	<xsl:with-param name="generate-footnotes-within-table-structure" select="true()" tunnel="yes"/>      
			  </xsl:apply-templates>
              <xsl:text> </xsl:text>
		 </xsl:for-each>		  
	 </xsl:template>
	     
</xsl:stylesheet>