/*
 * Copyright (c) 2000 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 *
 * $Id$
 */
package org.w3c.flute.parser.selectors;

import org.w3c.css.sac.CSSException;
import org.w3c.flute.css.sac.AttributeCondition;
import org.w3c.flute.css.sac.CombinatorCondition;
import org.w3c.flute.css.sac.Condition;
import org.w3c.flute.css.sac.ConditionFactory;
import org.w3c.flute.css.sac.ContentCondition;
import org.w3c.flute.css.sac.HasCondition;
import org.w3c.flute.css.sac.LangCondition;
import org.w3c.flute.css.sac.NegativeCondition;
import org.w3c.flute.css.sac.Selector;
import org.w3c.flute.parser.selectors.spc.EmptySPCCondition;
import org.w3c.flute.parser.selectors.spc.FirstChildSPCCondition;
import org.w3c.flute.parser.selectors.spc.FirstOfTypeSPCCondition;
import org.w3c.flute.parser.selectors.spc.LastChildSPCCondition;
import org.w3c.flute.parser.selectors.spc.LastOfTypeSPCCondition;
import org.w3c.flute.parser.selectors.spc.NthChildSPCCondition;
import org.w3c.flute.parser.selectors.spc.NthLastChildSPCCondition;
import org.w3c.flute.parser.selectors.spc.NthLastOfTypeSPCCondition;
import org.w3c.flute.parser.selectors.spc.NthOfTypeSPCCondition;
import org.w3c.flute.parser.selectors.spc.OnlyChildSPCCondition;
import org.w3c.flute.parser.selectors.spc.OnlyOfTypeSPCCondition;
import org.w3c.flute.parser.selectors.spc.RootSPCCondition;

/**
 * Factory for {@link Condition}s.
 * 
 * @version $Revision$
 * @author  Philippe Le Hegaret
 * @author Dan Caprioara
 */
public class ConditionFactoryImpl implements ConditionFactory {

  /**
   * The levels for the :before(N) pseudo elements. Not <code>null</code> and always starts with the value "1". It is ordered.
   */
  private int[] levelsForBefores;

  /**
   * The levels for the :after(N) pseudo elements. Not <code>null</code> and always starts with the value "1". It is ordered.
   */
  private int[] levelsForAfters;

  /**
   * Constructor.
   */
  public ConditionFactoryImpl() {
    levelsForAfters = new int[]{1};
    levelsForBefores = new int[]{1};
  }
  
  /**
   * Creates an and condition
   *
   * @param first the first condition
   * @param second the second condition
   * @return A combinator condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public CombinatorCondition createAndCondition(Condition first, Condition second)
      throws CSSException {
    return new AndConditionImpl(first, second);
  }

  /**
   * Creates an or condition
   *
   * @param first the first condition
   * @param second the second condition
   * @return A combinator condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public CombinatorCondition createOrCondition(Condition first, Condition second)
      throws CSSException {
    throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
  }

  /**
   * Creates a negative condition
   *
   * @param condition the condition
   * @return A negative condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public NegativeCondition createNegativeCondition(Condition condition) throws CSSException {
    throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
  }

  /**
   * Creates a negative condition, over a selector. 
   * Dan: In CSS3 there is support to negate a simple selector, not just a condition.
   *
   * @param selector the condition
   * @return A negative condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public NegativeCondition createNegativeCondition(Selector selector) throws CSSException {
    return new NegativeConditionImpl(selector);
  }

  /**
   * Creates a :has condition, over a selector. 
   *
   * @param selector the relative descendant selector
   * @return A :has condition
   */
  @Override
  public HasCondition createHasCondition(Selector selector){
    return new HasConditionImpl(selector);
  }

  /**
   * creates an attribute condition
   *
   * @param localName the localName of the attribute
   * @param namespaceURI the namespace URI of the attribute
   * @param specified <code>true</code> if the attribute must be specified
   *                  in the document.
   * @param value the value of this attribute.
   * @return An attribute condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public AttributeCondition createAttributeCondition(String localName, String namespaceURI,
      boolean specified, String value) throws CSSException {
    if (specified) {
      throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
    } else {
      return new AttributeConditionImpl(namespaceURI, localName, value);
    }
  }

  /**
   * Creates an id condition
   *
   * @param value the value of the id.
   * @return An Id condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public AttributeCondition createIdCondition(String value) throws CSSException {
    return new IdConditionImpl(value);
  }

  /**
   * Creates a lang condition
   *
   * @return A lang condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public LangCondition createLangCondition(String lang) throws CSSException {
    return new LangConditionImpl(lang);
  }

  /**
   * Creates a "one of" attribute condition
   *
   * @param localName the localName of the attribute
   * @param namespaceURI the namespace URI of the attribute
   * @param specified <code>true</code> if the attribute must be specified
   *                  in the document.
   * @param value the value of this attribute.
   * @return A "one of" attribute condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public AttributeCondition createOneOfAttributeCondition(String localName, String namespaceURI,
      boolean specified, String value) throws CSSException {
    if (specified) {
      throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
    } else {
      return new OneOfAttributeConditionImpl(namespaceURI, localName, value);
    }
  }

  /**
   * Creates a "begin hyphen" attribute condition
   *
   * @param localName the localName of the attribute
   * @param namespaceURI the namespace URI of the attribute
   * @param specified <code>true</code> if the attribute must be specified
   *                  in the document.
   * @param value the value of this attribute.
   * @return A "begin hyphen" attribute condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public AttributeCondition createBeginHyphenAttributeCondition(String localName,
      String namespaceURI, boolean specified, String value) throws CSSException {
    if (specified) {
      throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
    } else {
      return new BeginHyphenAttributeConditionImpl(namespaceURI, localName, value);
    }
  }

  /**
   * Creates a "begin with" attribute condition:
   * E[foo^="bar"]  an E element whose "foo" attribute value begins exactly with the string "bar"   Attribute selectors   3
   *
   * @param localName the localName of the attribute
   * @param namespaceURI the namespace URI of the attribute
   * @param specified <code>true</code> if the attribute must be specified
   *                  in the document.
   * @param value the value of this attribute.
   * @return A "begin hyphen" attribute condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public AttributeCondition createBeginWithAttributeCondition(String localName,
      String namespaceURI, boolean specified, String value) throws CSSException {
    if (specified) {
      throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
    } else {
      return new BeginsWithAttributeCondition(namespaceURI, localName, value);
    }
  }

  /**
   * Create condition for:
   * 
   * E[foo$="bar"]  an E element whose "foo" attribute value ends exactly with the string "bar"   Attribute selectors   3
   * 
   * @param localName the localName of the attribute
   * @param namespaceURI the namespace URI of the attribute
   * @param specified <code>true</code> if the attribute must be specified
   *                  in the document.
   * @param value the value of this attribute.
   * @return A "begin hyphen" attribute condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public AttributeCondition createEndWithAttributeCondition(String localName, String namespaceURI,
      boolean specified, String value) throws CSSException {
    if (specified) {
      throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
    } else {
      return new EndsWithAttributeCondition(namespaceURI, localName, value);
    }
  }

  /**
   * Create condition for:
   * 
   * E[foo*="bar"]  an E element whose "foo" attribute value contains the substring "bar" 
   * @param localName the localName of the attribute
   * @param namespaceURI the namespace URI of the attribute
   * @param specified <code>true</code> if the attribute must be specified
   *                  in the document.
   * @param value the value of this attribute.
   * @return A "begin hyphen" attribute condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public AttributeCondition createContainsAttributeCondition(String localName, String namespaceURI,
      boolean specified, String value) throws CSSException {
    if (specified) {
      throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
    } else {
      return new ContainsAttributeCondition(namespaceURI, localName, value);
    }
  }

  /**
   * Creates a class condition
   *
   * @param namespaceURI the namespace URI of the attribute
   * @param value the name of the class.
   * @return A class condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public AttributeCondition createClassCondition(String namespaceURI, String value)
      throws CSSException {
    return new ClassConditionImpl(value);
  }

  /**
   * Creates a pseudo class condition
   *
   * @param namespaceURI the namespace URI of the attribute
   * @param value the name of the pseudo class
   * @param pseudoLevel The level of the pseudo element. This is <code>1</code> for the 
   * simple :before and :after and the <code>N</code> value from :before(N), :after(N).
   * @return A pseudo class condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public AttributeCondition createPseudoClassCondition(String namespaceURI, String value, int pseudoLevel)
      throws CSSException {
    AttributeCondition cd;

    value = value.toLowerCase();
    if ("empty".equals(value)) {
      cd = new EmptySPCCondition();
    } else if ("first-child".equals(value)) {
      cd = new FirstChildSPCCondition();
    } else if ("first-of-type".equals(value)) {
      cd = new FirstOfTypeSPCCondition();
    } else if ("last-child".equals(value)) {
      cd = new LastChildSPCCondition();
    } else if ("last-of-type".equals(value)) {
      cd = new LastOfTypeSPCCondition();
    } else if ("only-child".equals(value)) {
      cd = new OnlyChildSPCCondition();
    } else if ("only-of-type".equals(value)) {
      cd = new OnlyOfTypeSPCCondition();
    } else if ("root".equals(value)) {
      cd = new RootSPCCondition();
    } else {
      // :before(N) or :after(N).
      cd = new PseudoClassConditionImpl(value, pseudoLevel);
      if (pseudoLevel != 1){
        if ("before".equals(value)){
          addBeforeLevel(pseudoLevel);
        } else {
          addAfterLevel(pseudoLevel);
        }
      }
    }

    return cd;
  }

  
  /**
   * Adds a level for the :before(s).
   *  
   * @param pseudoLevel The pseudo element to add.
   */
  public void addBeforeLevel(int pseudoLevel) {
    levelsForBefores = PseudoLevelMerger.addPseudoLevel(levelsForBefores, pseudoLevel);
  }

  
  /**
   * Adds a level for the :after(s).
   *  
   * @param pseudoLevel The pseudo element to add.
   */
  public void addAfterLevel(int pseudoLevel) {
    levelsForAfters = PseudoLevelMerger.addPseudoLevel(levelsForAfters, pseudoLevel);
  }

  /**
   * Creates a "only one" child condition
   *
   * @return A "only one" child condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public Condition createOnlyChildCondition() throws CSSException {
    throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
  }

  /**
   * Creates a "only one" type condition
   *
   * @return A "only one" type condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public Condition createOnlyTypeCondition() throws CSSException {
    throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
  }

  /**
   * Creates a content condition
   *
   * @param data the data in the content
   * @return A content condition
   * @exception CSSException if this exception is not supported.
   */
  @Override
  public ContentCondition createContentCondition(String data) throws CSSException {
    throw new CSSException(CSSException.SAC_NOT_SUPPORTED_ERR);
  }

  /**
   * Creates a nth-child condition. The parameters may be one of the constants: "odd", "even", 
   * or an expression like "1", "+3", or "5n+2". For the last one, the first number represents 
   * the cycle, the second is the delta.
   * 
   * For more information, please read here: http://www.w3.org/TR/css3-selectors/#nth-child-pseudo
   * 
   * @param pseudoFunction The name of the pseudo-function, like "nth-child".
   * @param cycle The cycle after which the condition is triggered again. Can be zero for no cycle.
   * @param delta The displacement in the cycle after which the condition is triggered.
   * @return A nth-child type condition.
   * @throws CSSException 
   */
  @Override
  public Condition createStructuralCondition(String pseudoFunction, int cycle, int delta)
      throws CSSException {
    Condition cd;

    if (pseudoFunction.startsWith("nth-child")) {
      cd = new NthChildSPCCondition(cycle, delta);
    } else if (pseudoFunction.startsWith("nth-last-child")) {
      cd = new NthLastChildSPCCondition(cycle, delta);
    } else if (pseudoFunction.startsWith("nth-of-type")) {
      cd = new NthOfTypeSPCCondition(cycle, delta);
    } else if (pseudoFunction.startsWith("nth-last-of-type")) {
      cd = new NthLastOfTypeSPCCondition(cycle, delta);
    } else {
      throw new CSSException("The pseudo function is not supported: " + pseudoFunction);
    }

    return cd;
  }

  /**
   * Gets the possible levels for the :before pseudo elements.
   * 
   * @return An array with all the :before pseudo levels. For instance when 
   * parsing a stylesheet having pseudo element selectors: x:before, z:before(500), 
   * y:before(200), the pseudo levels list will contain: 1, 200, 500.
   * Not <code>null</code> and always starts with the value "1". It is ordered.
   */
  public int[] getLevelsForBefores() {
    return levelsForBefores;
  }

  
  /**
   * Gets the possible levels for the :after pseudo elements.
   * 
   * @return An array with all the :after pseudo levels. For instance when
   * parsing a stylesheet having pseudo element selectors: x:after, z:after(500), 
   * y:after(200), the pseudo levels list will contain: 1, 200, 500.
   * Not <code>null</code> and always starts with the value "1". It is ordered.
   */
  public int[] getLevelsForAfters() {
    return levelsForAfters;
  }
  

}
