/*
 * Copyright (c) 2000 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 *
 * $Id$
 */
package org.w3c.flute.parser.selectors;

import java.util.Map;
import java.util.Set;

import org.w3c.flute.css.sac.AttributeCondition;
import org.w3c.flute.css.sac.Condition;

/**
 * @version $Revision$
 * @author  Philippe Le Hegaret
 */
public class AttributeConditionImpl implements AttributeCondition {

  /**
   * The local name.
   */
  String localName;

  /**
   * The value of the attribute.
   */
  String value;

  /**
   * The namespace of the uri.
   */
  String nsURI;

  /**
   * Creates a new AttributeConditionImpl
   * 
   * @param nsURI The namespace of the attribute
   * @param localName The local name of the attribute
   * @param value The value of the attribute
   * 
   */
  public AttributeConditionImpl(String nsURI, String localName, String value) {
    this.nsURI = nsURI;
    this.localName = localName;
    this.value = value;
  }

  /**
   * An integer indicating the type of <code>Condition</code>.
   */
  @Override
  public short getConditionType() {
    return Condition.SAC_ATTRIBUTE_CONDITION;
  }

  /**
   * Returns the
   * <a href="http://www.w3.org/TR/REC-xml-names/#dt-NSName">namespace
   * URI</a> of this attribute condition.
   * <p><code>NULL</code> if :
   * <ul>
   * <li>this attribute condition can match any namespace.
   * <li>this attribute is an id attribute.
   * </ul>
   */
  @Override
  public String getNamespaceURI() {
    return nsURI;
  }

  /**
   * Returns the
   * <a href="http://www.w3.org/TR/REC-xml-names/#NT-LocalPart">local part</a>
   * of the
   * <a href="http://www.w3.org/TR/REC-xml-names/#ns-qualnames">qualified
   * name</a> of this attribute.
   * <p><code>NULL</code> if :
   * <ul>
   * <li><p>this attribute condition can match any attribute.
   * <li><p>this attribute is a class attribute.
   * <li><p>this attribute is an id attribute.
   * <li><p>this attribute is a pseudo-class attribute.
   * </ul>
   */
  @Override
  public String getLocalName() {
    return localName;
  }

  /**
   * Returns <code>true</code> if the attribute must have an explicit value
   * in the original document, <code>false</code> otherwise.
   */
  @Override
  public boolean getSpecified() {
    return false;
  }

  /**
   * Returns the value of the attribute.
   * If this attribute is a class or a pseudo class attribute, you'll get
   * the class name (or psedo class name) without the '.' or ':'.
   */
  @Override
  public String getValue() {
    return value;
  }
  
  /**
   * Will return something line: [uri|local=val]
   * @see org.w3c.flute.css.sac.Condition#getSerialization(Map)
   */
  @Override
  public String getSerialization(Map<String, String> proxyNamespaceMapping) {
    return getSerialization("=", proxyNamespaceMapping);
  }

  
  /**
   * Gets the condition serialization.
   * 
   * @param oper The value operator: |=, ^=, $=, ~=, etc..
   * @param proxyNamespaceMapping The namespace mapping.
   * @return The serialization of the condition.
   */
  protected String getSerialization(String oper, Map<String, String> proxyNamespaceMapping) {
    StringBuilder sb = new StringBuilder();
    sb.append("[");
    if (nsURI != null && nsURI.length() > 0){
      String prefix = getPrefix(proxyNamespaceMapping);
      if (prefix != null){
        sb.append(prefix);
      } else {
        sb.append(nsURI);
      }
      sb.append("|");
    }
    sb.append(localName);
    if (value != null){
      sb.append(oper);
      sb.append("'");
      sb.append(value);
      sb.append("'");
    }
    sb.append("]");
    
    return sb.toString();
  }
  
  /**
   * Gets the prefix for the current namespace URI.
   * 
   * @param proxyNamespaceMapping The prefix to namespace mapping. 
   * @return The prefix, or null if not found.
   */
  private String getPrefix(Map<String, String> proxyNamespaceMapping) {	  
    String prefix = null;
    if(proxyNamespaceMapping != null){
      Set<Map.Entry<String, String>> entrySet = proxyNamespaceMapping.entrySet();
      for (Map.Entry<String, String> entry : entrySet) {
        if (nsURI.equals(entry.getValue())){
          prefix = entry.getKey();
        }          
      }
    }
    return prefix;
  }
}
