/*
 * Copyright (c) 2018 Syncro Soft SRL - All Rights Reserved.
 *
 * This file contains proprietary and confidential source code.
 * Unauthorized copying of this file, via any medium, is strictly prohibited.
 */
package org.w3c.flute.parser;

import java.util.Map;

/**
 * Utility for NSs in Parser.
 */
public class ParserUtil {
  /**
   * Any namespace.
   */
  public static final String ANY_NAMESPACE = "<ANY_NAMESPACE>";
  
  /**
   * Get the NS URI and local name for the attr qname
   * http://www.w3.org/TR/2005/WD-css3-selectors-20051215/#typenmsp
   * 
   * 
   * CSS examples:
   * <pre>    
   * {@literal @}namespace foo "http://www.example.com";
   * [foo|att=val] { color: blue }
   * [*|att] { color: yellow }
   * [|att] { color: green }
   * [att] { color: green }
   * </pre>
   * The first rule will match only elements with the attribute att in the "http://www.example.com" namespace with the value "val".
   * The second rule will match only elements with the attribute att regardless of the namespace of the attribute (including no declared namespace).
   * The last two rules are equivalent and will match only elements with the attribute att where the attribute is not declared to be in a namespace.
   * 
   * @param name Attr name
   * @param proxyNSMapping Proxy to NS mapping
   * @return The NS uri and local name
   */
  public static String[] getAttrNSURIAndLocal(String name, Map proxyNSMapping) {
    String[] prefixAndLocal = extractPrefixAndLocalName(name);
    String nsURI = "";
    if(
        // attribute in no namespace.
        prefixAndLocal[0] == null) {
      nsURI = "";
    } else if(prefixAndLocal[0].length() == 0) {
      //No namespace. |E - attributes with name E without any declared namespace
      nsURI = "";
    } else if (
        // "*|name"
        prefixAndLocal[0].length() == 1 && prefixAndLocal[0].charAt(0) == '*') {
      //Any namespace
      nsURI = ANY_NAMESPACE;
    } else {
      if("xml".equals(prefixAndLocal[0])) {
        //EXM-20071 XML prefix is reserved
        nsURI = "http://www.w3.org/XML/1998/namespace";
      } else {
        //Get from mapping.
        nsURI = (String) proxyNSMapping.get(prefixAndLocal[0]);
      }
    }
    return new String[] {nsURI, prefixAndLocal[1]};
  }
  
  /**
   * Get from an element QName the NS URI and local name
   * @param elQname Element QName
   * @param proxyNSMapping Proxy to NS mapping
   * @return the NS URI and local name
   */
  public static String[] getElementNSURIAndLocal(String elQname, Map proxyNSMapping) {
    String[] prefixAndLocal = extractPrefixAndLocalName(elQname);
    String nsURI = "";
    if(prefixAndLocal[0] == null) {
      //No "|", in default NS or ANY_NAMESPACE
      nsURI = (String)proxyNSMapping.get("");
      if(nsURI == null) {
        //No default NS -> ANY_NAMESPACE
        nsURI = ANY_NAMESPACE;
      }
    } else {
      if(prefixAndLocal[0] != null 
          && prefixAndLocal[0].length() == 1 
          && prefixAndLocal[0].charAt(0) == '*') {
        //This is any namespace.
        nsURI = ANY_NAMESPACE;
      } else if(prefixAndLocal[0] != null && prefixAndLocal[0].length() == 0){
        //This is no namespace.
        nsURI = "";
      } else {
        nsURI = (String)proxyNSMapping.get(prefixAndLocal[0]);
      }
    }
    return new String[] {nsURI, prefixAndLocal[1]};
  }
  
  /**
   * Extract prefix and local name from CSS QName.
   * 
   * @param selectorElemQName The QName
   * @return The prefix and local name.
   */
  public static String[] extractPrefixAndLocalName(String selectorElemQName) {
    String prefix = null;
    String localName = null;
    int sepIndex = selectorElemQName.indexOf('|');
    if(sepIndex != -1) {
      prefix = selectorElemQName.substring(0, sepIndex);
      localName = selectorElemQName.substring(sepIndex + 1);
    } else {
      //The usual approach, no prefix specified.
      localName = selectorElemQName;
    }
    return new String[] {prefix, localName};
  }
}
