/*
 * Copyright (c) 1999 World Wide Web Consortium
 * (Massachusetts Institute of Technology, Institut National de Recherche
 *  en Informatique et en Automatique, Keio University).
 * All Rights Reserved. http://www.w3.org/Consortium/Legal/
 *
 * $Id$
 */
package org.w3c.flute.parser;

import org.w3c.css.sac.LexicalUnit;

/**
 * @version $Revision$
 * @author  Philippe Le Hegaret
 */
public class LexicalUnitImpl implements LexicalUnitEx {
  /**
   * Previous LU
   */
  LexicalUnit prev;

  /**
   * Next LU
   */
  LexicalUnit next;

  /**
   * LU Type 
   */
  short type;

  /**
   * LU line
   */
  int line;

  /**
   * LU column
   */
  int column;

  /**
   * Integer value
   */
  int i;

  /**
   * Float value
   */
  float f;

  /**
   * Dimension
   */
  short dimension;

  /**
   * String dimension
   */
  String sdimension;

  /**
   * String value
   */
  String s;

  /**
   * Function name
   */
  String fname;

  /**
   * Params
   */
  LexicalUnitImpl params;

  /**
   * Constructor.
   * @param type Type
   * @param line Line
   * @param column Column
   * @param p Previous LU
   */
  protected LexicalUnitImpl(short type, int line, int column, LexicalUnitImpl p) {
    if (p != null) {
      prev = p;
      p.next = this;
    }
    this.line = line;
    this.column = column - 1;
    this.type = type;
  }

  /**
   * Constructor
   * @param line Line 
   * @param column Col
   * @param previous Previous LU
   * @param i Integer value
   */
  LexicalUnitImpl(int line, int column, LexicalUnitImpl previous, int i) {
    this(SAC_INTEGER, line, column, previous);
    this.i = i;
  }

  /**
   * Constructor
   * @param line Line
   * @param column Column
   * @param previous Previous LU
   * @param dimension Dimension
   * @param sdimension String dim
   * @param f Float dim
   */
  LexicalUnitImpl(int line, int column, LexicalUnitImpl previous, short dimension,
      String sdimension, float f) {
    this(dimension, line, column, previous);
    this.f = f;
    this.dimension = dimension;
    this.sdimension = sdimension;
  }

  /**
   * Constructor.
   * @param line Line
   * @param column Col
   * @param previous Prev
   * @param type Type
   * @param s String val
   */
  protected LexicalUnitImpl(int line, int column, LexicalUnitImpl previous, short type, String s) {
    this(type, line, column, previous);
    this.s = s;
  }

  /**
   * Constructor
   * @param type Type 
   * @param line Line
   * @param column Col
   * @param previous Prev
   * @param fname Function name
   * @param params Params
   */
  LexicalUnitImpl(short type, int line, int column, LexicalUnitImpl previous, String fname,
      LexicalUnitImpl params) {
    this(type, line, column, previous);
    this.fname = fname;
    this.params = params;
  }

  /**
   * @return Line number
   */
  public int getLineNumber() {
    return line;
  }

  /**
   * @return Column number
   */
  public int getColumnNumber() {
    return column;
  }

  @Override
  public short getLexicalUnitType() {
    return type;
  }

  @Override
  public LexicalUnit getNextLexicalUnit() {
    return next;
  }

  @Override
  public LexicalUnit getPreviousLexicalUnit() {
    return prev;
  }

  @Override
  public int getIntegerValue() {
    return i;
  }

  /**
   * Set int value
   * @param i The int value
   */
  void setIntegerValue(int i) {
    this.i = i;
  }

  /**
   * get float value
   * @see org.w3c.css.sac.LexicalUnit#getFloatValue()
   */
  @Override
  public float getFloatValue() {
    return f;
  }

  /**
   * Set float val
   * @param f Float val
   */
  void setFloatValue(float f) {
    this.f = f;
  }

  @Override
  public String getDimensionUnitText() {
    switch (type) {
      case SAC_PERCENTAGE:
        return "%";
      case SAC_DPI:
        return "dpi";
      case SAC_EM:
        return "em";
      case SAC_REM:
        return "rem";
      case SAC_EX:
        return "ex";
      case SAC_PIXEL:
        return "px";
      case SAC_CENTIMETER:
        return "cm";
      case SAC_MILLIMETER:
        return "mm";
      case SAC_INCH:
        return "in";
      case SAC_POINT:
        return "pt";
      case SAC_PICA:
        return "pc";
      case SAC_DEGREE:
        return "deg";
      case SAC_RADIAN:
        return "rad";
      case SAC_GRADIAN:
        return "grad";
      case SAC_MILLISECOND:
        return "ms";
      case SAC_SECOND:
        return "s";
      case SAC_HERTZ:
        return "Hz";
      case SAC_KILOHERTZ:
        return "kHz";
      case SAC_DIMENSION:
        return sdimension;
      default:
        throw new IllegalStateException("invalid dimension " + type);
    }
  }

  @Override
  public String getStringValue() {
    return s;
  }

  @Override
  public String getFunctionName() {
    return fname;
  }

  @Override
  public org.w3c.css.sac.LexicalUnit getParameters() {
    return params;
  }

  @Override
  public org.w3c.css.sac.LexicalUnit getSubValues() {
    return params;
  }

  @Override
  public String toString() {
    return serialize();
  }
  
  /**
   * Builds a string representation of the entire structure. Recursively iterates over the
   * {@link #getNextLexicalUnit()} and {@link #getParameters()}
   * @return
   */
  public String serialize() {
    StringBuilder b = new StringBuilder();
    serializeInternal(b);
    return  b.toString();
  }
  
  /**
   * Builds a string representation of the entire structure. Recursively iterates over the
   * {@link #getNextLexicalUnit()} and {@link #getParameters()}
   */
  private void serializeInternal(StringBuilder b) {
    toStringNoNext(b);
    if (next instanceof LexicalUnitImpl) {
      b.append(' ');
      ((LexicalUnitImpl) next).serializeInternal(b);
    }
  }

  /**
   * Gets the string representation of this lexical unit, without passing through 
   * the next lexical units.
   * 
   * @param b Builder where to put the string representation.
   */
  public void toStringNoNext(StringBuilder b) {
    switch (type) {
      case SAC_LPARAN:
        b.append("(");
        break;
      case SAC_RPARAN:
        b.append(")");
        break;
      case SAC_OPERATOR_COMMA:
        b.append(",");
        break;
      case SAC_OPERATOR_PLUS:
        b.append("+");
        break;
      case SAC_OPERATOR_MINUS:
        b.append("-");
        break;
      case SAC_OPERATOR_MULTIPLY:
        b.append("*");
        break;
      case SAC_OPERATOR_SLASH:
        b.append("/");
        break;
      case SAC_OPERATOR_MOD:
        b.append("%");
        break;
      case SAC_OPERATOR_EXP:
        b.append("^");
        break;
      case SAC_OPERATOR_LT:
        b.append("<");
        break;
      case SAC_OPERATOR_GT:
        b.append(">");
        break;
      case SAC_OPERATOR_LE:
        b.append("<=");
        break;
      case SAC_OPERATOR_GE:
        b.append("=>");
        break;
      case SAC_OPERATOR_TILDE:
        b.append("~");
        break;
      case SAC_INHERIT:
        b.append("inherit");
        break;
      case SAC_INITIAL:
        b.append("initial");
        break;
      case SAC_UNSET:
        b.append("unset");
        break;
      case SAC_INTEGER:
        b.append(Integer.toString(i, 10));
        break;
      case SAC_REAL:
        b.append(f);
        break;
      case SAC_DPI:
      case SAC_EM:
      case SAC_REM:
      case SAC_EX:
      case SAC_PIXEL:
      case SAC_INCH:
      case SAC_CENTIMETER:
      case SAC_MILLIMETER:
      case SAC_POINT:
      case SAC_PICA:
      case SAC_PERCENTAGE:
      case SAC_DEGREE:
      case SAC_GRADIAN:
      case SAC_RADIAN:
      case SAC_MILLISECOND:
      case SAC_SECOND:
      case SAC_HERTZ:
      case SAC_KILOHERTZ:
      case SAC_DIMENSION:
        int i = (int) f;
        if (i == f) {
          b.append(i).append(getDimensionUnitText());
        } else {
          b.append(f).append(getDimensionUnitText());
        }
        break;
      case SAC_URI:
        b.append("url(").append(s).append(")");
        break;        
      case SAC_RGBCOLOR:        
        // Deal with the RGB color separately, we can have RGB or RGBA depending on the
    	// number of parameters.
        LexicalUnit p = params;
        int cnt = 0;
        while(p != null) {
          cnt ++;
          p = p.getNextLexicalUnit();
        }
        if (cnt == 7 ) {
          // RGBA
          b.append("rgba(");
          params.serializeInternal(b);
          b.append(")");
        } else {
          // RGB (5 parameters)
          b.append("rgb(");
          if (params != null) {
            params.serializeInternal(b);
          }
          b.append(")");
        }
        break;
      case SAC_COUNTER_FUNCTION:
      case SAC_COUNTERS_FUNCTION:
      case SAC_RECT_FUNCTION:
      case SAC_FUNCTION:
        if (params == null){        
          // The functions with no parameters should not get a null as param.
          b.append(getFunctionName()).append("()");
        } else {
          b.append(getFunctionName()).append("(");
          params.serializeInternal(b);
          b.append(")");
        }
        break;
      case SAC_IDENT:
        b.append(getStringValue());
        break;
      case SAC_STRING_VALUE:
        // @@SEEME. not exact
    	// Escaping the quotes
        b.append("\"").append(getStringValue().replace("\"", "\\\"")).append("\"");
        break;
      case SAC_ATTR:
        b.append("attr(").append(getStringValue()).append(")");
        break;
      case SAC_UNICODERANGE:
        b.append("@@TODO");
        break;
      case SAC_SUB_EXPRESSION:
        ((LexicalUnitImpl) getSubValues()).serializeInternal(b);
        break;
      default:
        b.append("@unknown");
        break;
    }
  }

  // here some useful function for creation
  /**
   * Create number LU
   * @param line  Line 
   * @param column  Col
   * @param previous  Prev
   * @param v value
   * @return  The number LU
   */
  public static LexicalUnitImpl createNumber(int line, int column, LexicalUnitImpl previous, float v) {
    int i = (int) v;
    if (v == i) {
      return new LexicalUnitImpl(line, column, previous, i);
    } else {
      return new LexicalUnitImpl(line, column, previous, SAC_REAL, "", v);
    }
  }

  /**
   * Create an integer LU
   * @param line Line
   * @param column Col
   * @param previous Prev
   * @param i Int value
   * @return The LU.
   */
  public static LexicalUnitImpl createInteger(int line, int column, LexicalUnitImpl previous, int i) {
    return new LexicalUnitImpl(line, column, previous, i);
  }

  /**
   * Create percentage
   * @param line Line 
   * @param column Col
   * @param previous Prev LU
   * @param v Value
   * @return The LU
   */
  public static LexicalUnitImpl createPercentage(int line, int column, LexicalUnitImpl previous, float v) {
    return new LexicalUnitImpl(line, column, previous, SAC_PERCENTAGE, null, v);
  }

  /**
   * Create em size.
   * @param line Line
   * @param column Col
   * @param previous Prev LU
   * @param v value
   * @return The LU
   */
  public static LexicalUnitImpl createEMS(int line, int column, LexicalUnitImpl previous, float v) {
    return new LexicalUnitImpl(line, column, previous, SAC_EM, null, v);
  }

  /**
   * Create rem size.
   * @param line Line
   * @param column Col
   * @param previous Prev LU
   * @param v value
   * @return The LU
   */
  public static LexicalUnitImpl createRMS(int line, int column, LexicalUnitImpl previous, float v) {
    return new LexicalUnitImpl(line, column, previous, SAC_REM, null, v);
  }

  /**
   * Create x size
   * @param line Line
   * @param column Col
   * @param previous Prev LU
   * @param v value
   * @return The LU.
   */
  public static LexicalUnitImpl createEXS(int line, int column, LexicalUnitImpl previous, float v) {
    return new LexicalUnitImpl(line, column, previous, SAC_EX, null, v);
  }

  /**
   * Create pixel size.
   * @param line Line
   * @param column Column
   * @param previous Previous
   * @param v Value
   * @return The LU
   */
  public static LexicalUnitImpl createPX(int line, int column, LexicalUnitImpl previous, float v) {
    return new LexicalUnitImpl(line, column, previous, SAC_PIXEL, null, v);
  }

  /**
   * Create centimeters.
   * @param line Line
   * @param column Col
   * @param previous Prev LU
   * @param v Value
   * @return The LU
   */
  public static LexicalUnitImpl createCM(int line, int column, LexicalUnitImpl previous, float v) {
    return new LexicalUnitImpl(line, column, previous, SAC_CENTIMETER, null, v);
  }

  /**
   * Create milimiters
   * @param line Line
   * @param column Col
   * @param previous Prev LU
   * @param v value
   * @return the LU
   */
  public static LexicalUnitImpl createMM(int line, int column, LexicalUnitImpl previous, float v) {
    return new LexicalUnitImpl(line, column, previous, SAC_MILLIMETER, null, v);
  }

  /**
   * Create inch value
   * @param line Line 
   * @param column Col
   * @param previous Prev LU
   * @param v Value
   * @return The lexical unit
   */
  public static LexicalUnitImpl createIN(int line, int column, LexicalUnitImpl previous, float v) {
    return new LexicalUnitImpl(line, column, previous, SAC_INCH, null, v);
  }

  /**
   * Create point.
   * @param line Line
   * @param column Val
   * @param previous Prev LU
   * @param v The value
   * @return The LU
   */
  public static LexicalUnitImpl createPT(int line, int column, LexicalUnitImpl previous, float v) {
    return new LexicalUnitImpl(line, column, previous, SAC_POINT, null, v);
  }

  /**
   * Create PICA
   * @param line Line 
   * @param column Col
   * @param previous Prev LU
   * @param v Value
   * @return The LU
   */
  public static LexicalUnitImpl createPC(int line, int column, LexicalUnitImpl previous, float v) {
    return new LexicalUnitImpl(line, column, previous, SAC_PICA, null, v);
  }

  /**
   * Create degrees
   * @param line Line
   * @param column Col
   * @param previous Prev LU
   * @param v Value
   * @return The LU
   */
  public static LexicalUnitImpl createDEG(int line, int column, LexicalUnitImpl previous, float v) {
    return new LexicalUnitImpl(line, column, previous, SAC_DEGREE, null, v);
  }

  /**
   * Create radians LU
   * @param line Line 
   * @param column Col
   * @param previous Prev LU
   * @param v Value
   * @return The LU
   */
  public static LexicalUnitImpl createRAD(int line, int column, LexicalUnitImpl previous, float v) {
    return new LexicalUnitImpl(line, column, previous, SAC_RADIAN, null, v);
  }

  /**
   * Create gradient value
   * @param line Line 
   * @param column Col
   * @param previous Prev
   * @param v Value
   * @return The grad LU
   */
  public static LexicalUnitImpl createGRAD(int line, int column, LexicalUnitImpl previous, float v) {
    return new LexicalUnitImpl(line, column, previous, SAC_GRADIAN, null, v);
  }

  /**
   * Create millis LU
   * @param line Line
   * @param column Col
   * @param previous Prev LU
   * @param v Value
   * @return The millis LU
   */
  public static LexicalUnitImpl createMS(int line, int column, LexicalUnitImpl previous, float v) {
    if (v < 0) {
      throw new ParseException("Time values may not be negative");
    }
    return new LexicalUnitImpl(line, column, previous, SAC_MILLISECOND, null, v);
  }

  /**
   * Create seconds LU
   * @param line Line
   * @param column Col
   * @param previous Prev
   * @param v Value
   * @return The LU
   */
  public static LexicalUnitImpl createS(int line, int column, LexicalUnitImpl previous, float v) {
    if (v < 0) {
      throw new ParseException("Time values may not be negative");
    }
    return new LexicalUnitImpl(line, column, previous, SAC_SECOND, null, v);
  }

  /**
   * Create Hertz LU
   * @param line Line
   * @param column Column
   * @param previous Prev LU
   * @param v The float value
   * @return The LU
   */
  public static LexicalUnitImpl createHZ(int line, int column, LexicalUnitImpl previous, float v) {
    if (v < 0) {
      throw new ParseException("Frequency values may not be negative");
    }
    return new LexicalUnitImpl(line, column, previous, SAC_HERTZ, null, v);
  }

  /**
   * Create Kilohertz
   * @param line Line
   * @param column Col
   * @param previous Prev LU
   * @param v Float val
   * @return The LU
   */
  public static LexicalUnitImpl createKHZ(int line, int column, LexicalUnitImpl previous, float v) {
    if (v < 0) {
      throw new ParseException("Frequency values may not be negative");
    }
    return new LexicalUnitImpl(line, column, previous, SAC_KILOHERTZ, null, v);
  }

  static LexicalUnitImpl createDimen(int line, int column, LexicalUnitImpl previous, float v,
      String s) {
    return new LexicalUnitImpl(line, column, previous, SAC_DIMENSION, s, v);
  }

  public static LexicalUnitImpl createInherit(int line, int column, LexicalUnitImpl previous) {
    return new LexicalUnitImpl(line, column, previous, SAC_INHERIT, "inherit");
  }
  
  public static LexicalUnitImpl createInitial(int line, int column, LexicalUnitImpl previous) {
    return new LexicalUnitImpl(line, column, previous, SAC_INITIAL, "initial");
  }
  
  static LexicalUnitImpl createUnset(int line, int column, LexicalUnitImpl previous) {
    return new LexicalUnitImpl(line, column, previous, SAC_UNSET, "unset");
  }

  public static LexicalUnitImpl createIdent(int line, int column, LexicalUnitImpl previous, String s) {
    return new LexicalUnitImpl(line, column, previous, SAC_IDENT, s);
  }

  /**
   * Create simple string LU
   * @param line Line
   * @param column Col
   * @param previous Prev
   * @param s String val
   * @return The string LU
   */
  public static LexicalUnitImpl createString(int line, int column, LexicalUnitImpl previous, String s) {
    return new LexicalUnitImpl(line, column, previous, SAC_STRING_VALUE, s);
  }

  /**
   * Create an URL LU
   * @param line Line
   * @param column Col
   * @param previous Prev
   * @param s String val
   * @return The URL LU
   */
  public static LexicalUnitImpl createURL(int line, int column, LexicalUnitImpl previous, String s) {
    return new LexicalUnitImpl(line, column, previous, SAC_URI, s);
  }

  static LexicalUnitImpl createAttr(int line, int column, LexicalUnitImpl previous, String s) {
    return new LexicalUnitImpl(line, column, previous, SAC_ATTR, s);
  }

  static LexicalUnitImpl createCounter(int line, int column, LexicalUnitImpl previous,
      LexicalUnit params) {
    return new LexicalUnitImpl(SAC_COUNTER_FUNCTION, line, column, previous, "counter",
        (LexicalUnitImpl) params);
  }

  static LexicalUnitImpl createCounters(int line, int column, LexicalUnitImpl previous,
      LexicalUnit params) {
    return new LexicalUnitImpl(SAC_COUNTERS_FUNCTION, line, column, previous, "counters",
        (LexicalUnitImpl) params);
  }
  

  /**
   * Create a DPI lexical unit.
   * 
   * @param line The line of the token. 
   * @param column The column of the token.
   * @param previous Previous LU
   * @param v The value
   * @return The created LU
   */
  public static LexicalUnitImpl createDPI(int line, int column, LexicalUnitImpl previous, float v) {
    return new LexicalUnitImpl(line, column, previous, SAC_DPI, null, v);
  }

  /**
   * Create RGB color
   * 
   * @param line The line
   * @param column The column
   * @param previous The previous LU
   * @param params The params
   * @return The LU
   */
  public static LexicalUnitImpl createRGBColor(int line, int column, LexicalUnitImpl previous,
      LexicalUnit params) {
    return new LexicalUnitImpl(SAC_RGBCOLOR, line, column, previous, "color",
        (LexicalUnitImpl) params);
  }

  static LexicalUnitImpl createRect(int line, int column, LexicalUnitImpl previous,
      LexicalUnit params) {
    return new LexicalUnitImpl(SAC_RECT_FUNCTION, line, column, previous, "rect",
        (LexicalUnitImpl) params);
  }

  public static LexicalUnitImpl createFunction(int line, int column, LexicalUnitImpl previous,
      String fname, LexicalUnit params) {
    return new LexicalUnitImpl(SAC_FUNCTION, line, column, previous, fname,
        (LexicalUnitImpl) params);
  }

  static LexicalUnitImpl createUnicodeRange(int line, int column, LexicalUnit previous,
      LexicalUnit params) {
    // @@ return new LexicalUnitImpl(line, column, previous, null, SAC_UNICODERANGE, params);
    return null;
  }

  /**
   * Create comma LU
   * @param line Line
   * @param column Col
   * @param previous Prev LU
   * @return The LU.
   */
  public static LexicalUnitImpl createComma(int line, int column, LexicalUnitImpl previous) {
    return new LexicalUnitImpl(SAC_OPERATOR_COMMA, line, column, previous);
  }

  public static LexicalUnitImpl createSlash(int line, int column, LexicalUnitImpl previous) {
    return new LexicalUnitImpl(SAC_OPERATOR_SLASH, line, column, previous);
  }
  
  public static LexicalUnitImpl createMinus(int line, int column, LexicalUnitImpl previous) {
    return new LexicalUnitImpl(SAC_OPERATOR_MINUS, line, column, previous);
  }
  
  public static LexicalUnitImpl createPlus(int line, int column, LexicalUnitImpl previous) {
      return new LexicalUnitImpl(SAC_OPERATOR_PLUS, line, column, previous);
  }

  public static LexicalUnitImpl createMultiply(int line, int column, LexicalUnitImpl previous) {
    return new LexicalUnitImpl(SAC_OPERATOR_MULTIPLY, line, column, previous);
  }

  public static LexicalUnitImpl createLParan(int line, int column, LexicalUnitImpl previous) {
    return new LexicalUnitImpl(SAC_LPARAN, line, column, previous);
  }
  
  public static LexicalUnitImpl createRParan(int line, int column, LexicalUnitImpl previous) {
    return new LexicalUnitImpl(SAC_RPARAN, line, column, previous);
  }

  /**
   * @param prev The prev to set.
   */
  public void setPrev(LexicalUnit prev) {
    this.prev = prev;
  }

  /**
   * @param next The next to set.
   */
  public void setNext(LexicalUnit next) {
    this.next = next;
  }
  
  /**
   * Clone this property decl LU deep.
   * @return The deep cloned LU.
   */
  public LexicalUnitImpl clonePropertyDeclLU() {
    LexicalUnitImpl clone = cloneLU();
    
    //Now navigate the list to the right and clone all.
    if(this.next != null) {
      //Clone the next handler.
      clone.next = ((LexicalUnitImpl) this.next).cloneLU();
      
      //Navigate...
      LexicalUnitImpl cloneNextIter = (LexicalUnitImpl) clone.next;
      LexicalUnitImpl nextIter = (LexicalUnitImpl) next;
      while (nextIter != null) {
        nextIter = (LexicalUnitImpl) nextIter.getNextLexicalUnit();
        if(nextIter != null) {
          LexicalUnitImpl clonedNext = nextIter.cloneLU();
          cloneNextIter.setNext(clonedNext);
          clonedNext.setPrev(cloneNextIter);
          cloneNextIter = clonedNext;
        }
      }
    }
    return clone;
  }
  
  /**
   * Clone this LU deep.
   * 
   * @return The deep cloned LU.
   */
  private LexicalUnitImpl cloneLU() {
    LexicalUnitImpl clone = new LexicalUnitImpl(line, column, null, i);
    clone.dimension = this.dimension;
    clone.f = this.f;
    clone.fname = this.fname;
    clone.s = this.s;
    clone.sdimension = this.sdimension;
    clone.type = this.type;
    if(this.params != null) {
      //Iterate the params list and clone each of them.
      clone.params = this.params.cloneLU();
      LexicalUnitImpl clonedParamsIter = clone.params;
      LexicalUnitImpl paramsIter = params;
      while (paramsIter != null) {
        paramsIter = (LexicalUnitImpl) paramsIter.getNextLexicalUnit();
        if(paramsIter != null) {
          LexicalUnitImpl clonedParam = paramsIter.cloneLU();
          clonedParamsIter.setNext(clonedParam);
          clonedParam.setPrev(clonedParamsIter);
          clonedParamsIter = clonedParam;
        }
      }
    }
    
    return clone;
  }

  /**
   * Creates a list of String lexical units out of a String array.
   * 
   * @param line The line of the lexical unit.
   * @param column The column of the lexical unit.
   * @param previous The previous lexical unit.
   * @param strings The array of strings. Can be <code>null</code>. 
   * @return The linked list of {@link LexicalUnit}, all of type {@link LexicalUnit#SAC_STRING_VALUE}. May be <code>null</code> if the array is empty.
   */
  public static LexicalUnit createString(int line, int column, LexicalUnitImpl previous, String[] strings) {
    
    LexicalUnit first = null;
    LexicalUnitImpl lu = previous;
    
    int length = strings.length;
    for (int i = 0; i < length; i++) {
      String string = strings[i];
      
      lu =  createString(line, column, lu, string);
      if (first == null) {
        first = lu;
      }

      if (i < length - 1) {
        lu = createComma(line, column, lu);
      }                
    }
    return first;
    
  }
}
