/*
 * Copyright (c) 1999 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 *
 * $Id: SelectorFactory.java,v 1.3 2000/07/27 21:19:21 plehegar Exp $
 */
package org.w3c.flute.css.sac;

import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.Locator;
import org.w3c.css.sac.NegativeSelector;

/**
 * @version $Revision: 1.3 $
 * @author  Philippe Le Hegaret
 * @see org.w3c.flute.css.sac.Selector
 */
public interface SelectorFactory {

  /**
   * Creates a conditional selector.
   * 
   * @param selector a selector
   * @param condition a condition
   * @param locator The current parser locator. Can be used to 
   * access the URI, line and column of the selector currently built.
   * Note that this locator may be later reused, so it is not recommended
   * to keep a reference to it.
   * 
   * @return the conditional selector.
   * @exception CSSException If this selector is not supported.
   */
  ConditionalSelector createConditionalSelector(SimpleSelector selector, Condition condition, Locator locator)
      throws CSSException;

  /**
   * Creates an any node selector.
   * 
   * @param locator The current parser locator. Can be used to 
   * access the URI, line and column of the selector currently built.
   * Note that this locator may be later reused, so it is not recommended
   * to keep a reference to it.
   * @return the any node selector.
   * @exception CSSException If this selector is not supported.
   */
  SimpleSelector createAnyNodeSelector(Locator locator) throws CSSException;

  /**
   * Creates an root node selector.
   * 
   * @param locator The current parser locator. Can be used to 
   * access the URI, line and column of the selector currently built.
   * Note that this locator may be later reused, so it is not recommended
   * to keep a reference to it.
   * @return the root node selector.
   * @exception CSSException If this selector is not supported.
   */
  SimpleSelector createRootNodeSelector(Locator locator) throws CSSException;

  /**
   * Creates an negative selector.
   * 
   * @param selector a selector.
   * @param locator The current parser locator. Can be used to 
   * access the URI, line and column of the selector currently built.
   * Note that this locator may be later reused, so it is not recommended
   * to keep a reference to it.
   * @return the negative selector.
   * @exception CSSException If this selector is not supported.
   */
  NegativeSelector createNegativeSelector(SimpleSelector selector, Locator locator) throws CSSException;

  /**
   * Creates an element selector.
   * 
   * @param namespaceURI the <a href="http://www.w3.org/TR/REC-xml-names/#dt-NSName">namespace
   *                     URI</a> of the element selector.
   * @param tagName the <a href="http://www.w3.org/TR/REC-xml-names/#NT-LocalPart">local
   *        part</a> of the element name. <code>NULL</code> if this element
   *        selector can match any element.
   * @param locator The current parser locator. Can be used to 
   * access the URI, line and column of the selector currently built.
   * Note that this locator may be later reused, so it is not recommended
   * to keep a reference to it.
   * @return the element selector
   * @exception CSSException If this selector is not supported.
   */
  ElementSelector createElementSelector(String namespaceURI, String tagName, Locator locator) throws CSSException;

  /**
   * Creates a pseudo element selector.
   * 
   * @param namespaceURI The namespace of the element.
   * @param pseudoName the pseudo element name. <code>NULL</code> if this
   *                   element selector can match any pseudo element.
   *
   * 
   * @param locator The current parser locator. Can be used to 
   * access the URI, line and column of the selector currently built.
   * Note that this locator may be later reused, so it is not recommended
   * to keep a reference to it.
   * @return the element selector
   * @exception CSSException If this selector is not supported.
   */
  ElementSelector createPseudoElementSelector(String namespaceURI, String pseudoName, Locator locator)
      throws CSSException;

  /**
   * Creates a sibling selector.
   *
   * @param nodeType the type of nodes in the siblings list.
   * @param child the child selector
   * @param directAdjacent The last part of the operator.
   * @param locator The current parser locator. Can be used to 
   * access the URI, line and column of the selector currently built.
   * Note that this locator may be later reused, so it is not recommended
   * to keep a reference to it.
   * @return the sibling selector with nodeType 
             equals to org.w3c.dom.Node.ELEMENT_NODE
   * @exception CSSException If this selector is not supported.
   */
  SiblingSelector createDirectAdjacentSelector(short nodeType, Selector child,
      SimpleSelector directAdjacent, Locator locator) throws CSSException;

  /**
   * Creates a child selector.
   *
   * @param parent the parent selector
   * @param child the child selector
   * @param isSubject true if the simple selector has been marked as the subject. http://dev.w3.org/csswg/selectors4/#subject
   * @param locator The current parser locator. Can be used to 
   * access the URI, line and column of the selector currently built.
   * Note that this locator may be later reused, so it is not recommended
   * to keep a reference to it.
   * @return the combinator selector.
   * @exception CSSException If this selector is not supported.
   */
  DescendantSelector createChildSelector(Selector parent, SimpleSelector child, boolean isSubject, Locator locator)
      throws CSSException;

  /**
   * Creates a descendant selector.
   *
   * @param parent the parent selector
   * @param descendant the descendant selector
   * @param isSubject true if the simple selector has been marked as the subject. http://dev.w3.org/csswg/selectors4/#subject
   * @param locator The current parser locator. Can be used to 
   * access the URI, line and column of the selector currently built.
   * Note that this locator may be later reused, so it is not recommended
   * to keep a reference to it.
   * @return the combinator selector.
   * @exception CSSException If this selector is not supported.
   */
  DescendantSelector createDescendantSelector(Selector parent, SimpleSelector descendant,
      boolean isSubject, Locator locator) throws CSSException;

  /**
   * Creates a direct adjacent selector.
   *
   * @param nodeType The type of the node.
   * @param child the child selector
   * @param directAdjacent the direct adjacent selector
   * @param locator The current parser locator. Can be used to 
   * access the URI, line and column of the selector currently built.
   * Note that this locator may be later reused, so it is not recommended
   * to keep a reference to it.
   * @return the combinator selector.
   * @exception CSSException If this selector is not supported.
   */
  SiblingSelector createGeneralAdjacentSelector(short nodeType, Selector child,
      SimpleSelector directAdjacent, Locator locator) throws CSSException;
}
