/*
 * Copyright (c) 1999 World Wide Web Consortium
 * (Massachusetts Institute of Technology, Institut National de Recherche
 *  en Informatique et en Automatique, Keio University).
 * All Rights Reserved. http://www.w3.org/Consortium/Legal/
 *
 * $Id: DocumentHandler.java,v 1.4 2000/02/16 21:29:35 plehegar Exp $
 */
package org.w3c.flute.css.sac;

import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.InputSource;
import org.w3c.css.sac.LexicalUnit;
import org.w3c.flute.parser.MediaQueryList;

/**
 * This is the main interface that most CSS applications implement: if the
 * application needs to be informed of basic parsing events, it implements this
 * interface and registers an instance with the CSS parser using the
 * setCSSHandler method.
 *
 * @version $Revision: 1.4 $
 * @author Philippe Le Hegaret 
 */
public interface DocumentHandler {
    
    /**
     * Receive notification of the beginning of a style sheet.
     *
     * The CSS parser will invoke this method only once, before any other
     * methods in this interface.
     *
     * @param source The source of the style sheet. @@TODO can be NULL ! (inline style sheet)
     * @exception CSSException Any CSS exception, possibly wrapping another
     *                         exception.  
     */
    void startDocument(InputSource source)
        throws CSSException;
    
    /**
     * Receive notification of the end of a document. 
     *
     * The CSS parser will invoke this method only once, and it will be the
     * last method invoked during the parse. The parser shall not invoke this
     * method until it has either abandoned parsing (because of an
     * unrecoverable error) or reached the end of input.  
     *
     * @param source The source of the style sheet.
     * @exception CSSException Any CSS exception, possibly wrapping another
     *                         exception.  
     */
    void endDocument(InputSource source) throws CSSException;

    /**
     * Receive notification of a comment.
     * If the comment appears in a declaration (e.g. color: /* comment * / blue;),
     * the parser notifies the comment before the declaration.
     *
     * @param text The comment.
     * @exception CSSException Any CSS exception, possibly wrapping another
     *                         exception.  
     */
    void comment(String text) throws CSSException;

    /**
     * Receive notification of an unknown rule t-rule not supported by this
     * parser.
     *
     * @param atRule The complete ignored at-rule.
     * @exception CSSException Any CSS exception, possibly wrapping another
     *                         exception.  
     */
    void ignorableAtRule(String atRule) throws CSSException;

    /**
     * Receive notification of an unknown rule t-rule not supported by this
     * parser.
     *
     * @param prefix <code>null</code> if this is the default namespace
     * @param uri The URI for this namespace.
     * @exception CSSException Any CSS exception, possibly wrapping another
     *                         exception.  
     */
    void namespaceDeclaration(String prefix, String uri) 
	throws CSSException;

    /**
     * Start media.
     * 
     * @param mediaQueryList The media query list.
     */
    void startMedia(MediaQueryList mediaQueryList) ;

    /**
     * End media.
     * 
     * @param mediaQueryList The media query list.
     */
    void endMedia(MediaQueryList mediaQueryList);

    /**
     * Import a CSS for a certain media.
     * 
     * @param path The path.
     * @param forMedia The media query list.
     * @param namespaceURI The namespace URI.
     * @throws CSSException
     */
    void importStyle(String path, MediaQueryList forMedia, String namespaceURI) throws CSSException;

    /**
     * Receive notification of the beginning of a page statement.
     *
     * The Parser will invoke this method at the beginning of every page
     * statement in the style sheet. there will be a corresponding endPage()
     * event for every startPage() event.
     *
     * @param pageSelectorList list of the page selectors. Can be <code>null</code> if 
     * the page rule is generic (no specific name and no pseudo).
     * @exception CSSException Any CSS exception, possibly wrapping another
     *                         exception.
     *                         
     * @see #startPageMargin(String)
     * @see #endPage(PageSelectorList)                          
     */    
    void startPage(PageSelectorList pageSelectorList) throws CSSException;
    
    /**
     * Receive notification of the end of a page statement.
     *
     * @param pageSelectorList list of the page selectors. Can be <code>null</code> if 
     * the page rule is generic (no specific name and no pseudo).
     * @exception CSSException Any CSS exception, possibly wrapping another
     *                         exception.
     * @see #startPage(PageSelectorList)                         
     *                           
     */    
    void endPage(PageSelectorList pageSelectorList) throws CSSException;

    /**
     * Receive notification of the beginning of a font face statement.
     *
     * The Parser will invoke this method at the beginning of every font face
     * statement in the style sheet. there will be a corresponding endFontFace()
     * event for every startFontFace() event.
     *
     * @exception CSSException Any CSS exception, possibly wrapping another
     *                         exception.
     */
    void startFontFace() throws CSSException;

    /**
     * Receive notification of the end of a font face statement.
     *
     * @exception CSSException Any CSS exception, possibly wrapping another
     *                         exception.  
     */
    void endFontFace() throws CSSException;

    /**
     * Receive notification of the beginning of a rule statement.
     *
     * @param selectors All intended selectors for all declarations.
     * @exception CSSException Any CSS exception, possibly wrapping another
     *                         exception.
     */
    void startSelector(SelectorList selectors) throws CSSException;

    /**
     * Receive notification of the end of a rule statement.
     *
     * @param selectors All intended selectors for all declarations.
     * @exception CSSException Any CSS exception, possibly wrapping another
     *                         exception.
     */
    void endSelector(SelectorList selectors) throws CSSException;

    /**
     * Receive notification of a declaration.
     *
     * @param name the name of the property.
     * @param value the value of the property. All whitespace are stripped.
     * @param important is this property important ?
     * @exception CSSException Any CSS exception, possibly wrapping another
     *                         exception.
     */
    void property(String name, LexicalUnit value, boolean important)
        throws CSSException;
    

    /**
     * Receive notification of a start of a page margin.
     * Example:
     * <pre>
     *  {@literal @}page {
     *  
     *    {@literal @}top-left-corner {
     *        background-color: green;
     *    }
     *    
     *    {@literal @}bottom-left-corner {
     *        margin: 1em;
     *    }
     *    
     *  }
     * </pre> 
     * 
     * The sequence of the events are:
     * <table>
     * <caption>Events</caption>
     * <tr><td>{@link #startPage(PageSelectorList)}</td><td><code>null</code>, <code>null</code></td></tr>
     * <tr><td>{@link #startPageMargin(String)}</td><td><code>top-left-corner</code></td></tr>
     * <tr><td>{@link #endPageMargin(String)}</td><td><code>top-left-corner</code></td></tr>
     * <tr><td>{@link #startPageMargin(String)}</td><td><code>bottom-left-corner</code></td></tr>
     * <tr><td>{@link #endPageMargin(String)}</td><td><code>bottom-left-corner</code></td></tr>
     * <tr><td>{@link #endPage(PageSelectorList)}</td><td><code>null</code>, <code>null</code></td></tr>
     * </table>
     * 
     * @param pageMarginName The name of the page margin. In our example would be called twice,
     * one for 'top-left-corner', and one for 'bottom-left-corner'; 
     */
    public void startPageMargin(String pageMarginName);
    
    /**
     * Receive notification of an end of a page margin.
     * 
     * @param pageMarginName The name of the page margin. 
     * @see #startPageMargin(String)
     */
    public void endPageMargin(String pageMarginName);

    
    
}
