<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    exclude-result-prefixes="xs util s"
    xmlns:util="http://www.oxygenxml.com/ns/framework/extend/util"
    xmlns:s="http://www.oxygenxml.com/ns/framework/extend"
    version="2.0">
    
    <xd:doc scope="stylesheet" xmlns:xd="http://www.oxygenxml.com/ns/doc/xsl">
        <xd:desc>
            <xd:p>Utility functions.</xd:p>
        </xd:desc>
    </xd:doc>
    
    
    <xsl:param name="TOOLBAR_TYPE_SUBTOOLBAR">3</xsl:param>
    <xsl:param name="TOOLBAR_TYPE_MAIN">2</xsl:param>
    
    <xsl:param name="CC_WINDOW_TARGET">0</xsl:param>
    <xsl:param name="CC_POP_UP_MENUS_TARGET">1</xsl:param>
    <xsl:param name="CC_ELEMENT_VIEWS_TARGET">2</xsl:param>
    <xsl:param name="CC_ENTITY_VIEWS_TARGET">3</xsl:param>
    
    <xd:doc xmlns:xd="http://www.oxygenxml.com/ns/doc/xsl">
        <xd:desc>
            <xd:p>Wraps the given value in a <xd:b>String</xd:b> element</xd:p>
        </xd:desc>
        <xd:param name="value"/>
    </xd:doc>
    <xsl:function name="util:stringValue" as="node()">
        <xsl:param name="value" as="xs:string?"/>
        <String><xsl:value-of select="$value"/></String>
    </xsl:function>
    
    
    <xd:doc xmlns:xd="http://www.oxygenxml.com/ns/doc/xsl">
        <xd:desc>
            <xd:p>If the value represents a relative path will rewrite it with $framework and $frameworkDir</xd:p>
        </xd:desc>
        <xd:param name="path">The path to serialize.</xd:param>
        <xd:param name="relativePathRewritePolicy">
            <xd:ul>
                <xd:li>none - no rewriting</xd:li>
                <xd:li>dir - put a $frameworkDir in front.</xd:li>
                <xd:li>url - put a $framework in front.</xd:li>
            </xd:ul>
        </xd:param>
    </xd:doc>
    <xsl:function name="util:serializePathValue" as="node()">
        <xsl:param name="path" as="xs:string?"/>
        <xsl:param name="relativePathRewritePolicy" as="xs:string"></xsl:param>
        
        <xsl:variable name="isFilePath" as="xs:boolean" select="$relativePathRewritePolicy = 'dir'"/>
        
        <xsl:choose>
            <!-- The path is already written with variables or with a protocol. Leave it like it is. -->
            <xsl:when test="
                $relativePathRewritePolicy = 'none'
                or starts-with($path, '${') 
                or starts-with($path, '/')
                or matches($path, '[a-z]+:.+')"><String><xsl:value-of select="util:replaceBackwardSlash($path)"/></String></xsl:when>
            <!-- put $frameworkDir in front of it -->
            <xsl:when test="$isFilePath"><String><xsl:value-of select="concat('${frameworkDir}/', util:replaceBackwardSlash($path))"/></String></xsl:when>
            <!-- put $framework in front of it -->
            <xsl:otherwise><String><xsl:value-of select="concat('${framework}/', util:replaceBackwardSlash($path))"/></String></xsl:otherwise>
        </xsl:choose>
    </xsl:function>
    
    <xd:doc xmlns:xd="http://www.oxygenxml.com/ns/doc/xsl">
        <xd:desc>Replaces all \ with /</xd:desc>
        <xd:param name="path">The path to process.</xd:param>
    </xd:doc>
    <xsl:function name="util:replaceBackwardSlash" as="xs:string">
        <xsl:param name="path" as="xs:string"></xsl:param>
        
        <xsl:value-of select="replace($path, '\\', '/')"/>
    </xsl:function>
    
    <xd:doc xmlns:xd="http://www.oxygenxml.com/ns/doc/xsl">
        <xd:desc>If the entry name is in the list of skipped entries, returns true.</xd:desc>
        <xd:param name="entryName">Name of the entry to test.</xd:param>
        <xd:param name="skippedEntriesNames">The list of skipped entries names.</xd:param>
    </xd:doc>
    <xsl:function name="util:isSkippedEntry" as="xs:boolean">
        <xsl:param name="entryName" as="xs:string"></xsl:param>
        <xsl:param name="skippedEntriesNames" as="xs:string*" />
        
        <xsl:sequence select="not(empty(index-of($skippedEntriesNames, $entryName)))"/>
    </xsl:function>
    
    <xd:doc xmlns:xd="http://www.oxygenxml.com/ns/doc/xsl">
        <xd:desc>
            Replaces ${framework(BASE)} and ${baseFramework} with ${framework}
            Replaces ${frameworkDir(BASE)} and ${baseFrameworkDir} with ${frameworkDir}</xd:desc>
        <xd:param name="paths">The list of strings to process.</xd:param>
    </xd:doc>
    <xsl:function name="util:replaceBaseWithFramework" as="xs:string*">
        <xsl:param name="paths" as="xs:string*" />
        <xsl:choose>
            <!-- We have a base framework. -->
            <xsl:when test="($script/@base and string-length($script/@base))">
                <xsl:for-each select="$paths">
                    <xsl:variable name="step1" select="replace(., concat('\$\{framework\(', $script/@base, '\)\}'), '\${framework}')"/>
                    <xsl:variable name="step2" select="replace($step1, concat('\$\{frameworkDir\(', $script/@base, '\)\}'), '\${frameworkDir}')"/>
                    
                    <xsl:variable name="step3" select="replace($step2, '\$\{baseFramework\}', '\${framework}')"/>
                    <xsl:variable name="step4" select="replace($step3, '\$\{baseFrameworkDir\}', '\${frameworkDir}')"/>
                    
                    <xsl:copy-of select="$step4"/>
                </xsl:for-each>
            </xsl:when>
            <xsl:otherwise><xsl:copy-of select="$paths"/></xsl:otherwise>
        </xsl:choose>
        
    </xsl:function>  
    
    <xd:doc xmlns:xd="http://www.oxygenxml.com/ns/doc/xsl">
        <xd:desc>Creates a CSS framework entry over the given details.</xd:desc>
        <xd:param name="path">The path to the CSS.</xd:param>
        <xd:param name="title">The CSS title.</xd:param>
        <xd:param name="alternate">The CSS alternate flag. true or false.</xd:param>
    </xd:doc>
    <xsl:function name="util:createCSSEntry" as="node()">
        <xsl:param name="path" as="xs:string"/>
        <xsl:param name="title" as="xs:string?"/>
        <xsl:param name="alternate" as="xs:string?"/>
        
        <cssFile>
            <field name="href">
                <xsl:sequence select="util:serializePathValue($path, 'url')"></xsl:sequence>
            </field>
            <field name="title">
                <String><xsl:value-of select="$title"/></String>
            </field>
            <field name="alternate">
                <Boolean><xsl:value-of select="exists($alternate) and xs:boolean($alternate)"/></Boolean>
            </field>
        </cssFile>
    </xsl:function>
    
    <xd:doc xmlns:xd="http://www.oxygenxml.com/ns/doc/xsl">
        <xd:desc>Creates an association rule.</xd:desc>
        <xd:param name="rule">The rule.</xd:param>
    </xd:doc>
    <xsl:function name="util:createAssociationRule" as="node()">
        <xsl:param name="rule" as="element()"/>
        
        <documentTypeRule>
            <field name="namespace">
                <xsl:sequence select="util:getAssociationValue($rule/@namespace)"/>
            </field>
            <field name="rootElem">
                <xsl:sequence select="util:getAssociationValue($rule/@rootElementLocalName)"/>
            </field>
            <field name="fileName">
                <xsl:sequence select="util:getAssociationValue($rule/@fileName)"/>
            </field>
            <field name="publicID">
                <xsl:sequence select="util:getAssociationValue($rule/@publicID)"/>
            </field>
            <field name="javaRuleClass">
                <String><xsl:value-of select="$rule/@javaRuleClass"/></String>
            </field>
            <field name="attributeLocalName">
                <xsl:sequence select="util:getAssociationValue($rule/@attributeLocalName)"/>
            </field>
            <field name="attributeNamespace">
                <xsl:sequence select="util:getAssociationValue($rule/@attributeNamespace)"/>
            </field>
            <field name="attributeValue">
                <xsl:sequence select="util:getAssociationValue($rule/@attributeValue)"/>
            </field>
        </documentTypeRule>
    </xsl:function>
    
    <xsl:function name="util:getAssociationValue">
        <xsl:param name="value"/>
        <xsl:sequence select="util:stringValue(if($value) then $value else '*')"></xsl:sequence>
    </xsl:function>
    
    <xd:doc xmlns:xd="http://www.oxygenxml.com/ns/doc/xsl">
        <xd:desc>Creates a new entry in the content completion configuration.</xd:desc>
        <xd:param name="newAction">The new item configuration.</xd:param>
    </xd:doc>
    <xsl:function name="util:createCCEntry">
        <xsl:param name="newAction"/>
        
        <contextItem>
            <field name="targets">
                <list>
                    <xsl:if test="$newAction/xs:boolean(@inCCWindow)">
                        <Integer><xsl:value-of select="$CC_WINDOW_TARGET"/></Integer>
                    </xsl:if>
                    <xsl:if test="$newAction/xs:boolean(@inMenus)">
                        <Integer><xsl:value-of select="$CC_POP_UP_MENUS_TARGET"/></Integer>
                    </xsl:if>
                    <xsl:if test="$newAction/xs:boolean(@inElementsView)">
                        <Integer><xsl:value-of select="$CC_ELEMENT_VIEWS_TARGET"/></Integer>
                    </xsl:if>
                </list>
            </field>
            <field name="actionID">
                <String><xsl:value-of select="$newAction/@id"/></String>
            </field>
            <field name="alias">
                <String><xsl:value-of select="$newAction/@alias"/></String>
            </field>
            <field name="replacementFor">
                <xsl:choose>
                    <xsl:when test="$newAction/@replacedElement">
                        <String><xsl:value-of select="$newAction/@replacedElement"/></String>
                    </xsl:when>
                    <xsl:otherwise>
                        <null/>
                    </xsl:otherwise>
                </xsl:choose>
            </field>
            <field name="replacedElementNs">
                <xsl:choose>
                    <xsl:when test="$newAction/@replacedElementNs">
                        <String><xsl:value-of select="$newAction/@replacedElementNs"/></String>
                    </xsl:when>
                    <xsl:otherwise>
                        <null/>
                    </xsl:otherwise>
                </xsl:choose>
            </field>
            <field name="displayOnlyWhenElementAllowed">
                <xsl:choose>
                    <xsl:when test="$newAction/@displayOnlyWhenElementAllowed">
                        <Boolean><xsl:value-of select="$newAction/xs:boolean(@displayOnlyWhenElementAllowed)"/></Boolean>
                    </xsl:when>
                    <xsl:otherwise>
                        <Boolean>false</Boolean>
                    </xsl:otherwise>
                </xsl:choose>
            </field>
            <field name="useReplaceElementName">
                <xsl:choose>
                    <xsl:when test="$newAction/@useReplaceElementName">
                        <Boolean><xsl:value-of select="$newAction/xs:boolean(@useReplaceElementName)"/></Boolean>
                    </xsl:when>
                    <xsl:otherwise>
                        <Boolean>false</Boolean>
                    </xsl:otherwise>
                </xsl:choose>
            </field>
        </contextItem>
    </xsl:function>
    
    <xd:doc xmlns:xd="http://www.oxygenxml.com/ns/doc/xsl">
        <xd:desc>Updates the content completion configuration. Specifies which entries to remove from the defaults.</xd:desc>
        <xd:param name="removeEntryConfig">Remove entry configuration.</xd:param>
    </xd:doc>
    <xsl:function name="util:createRemoveCCEntry">
        <xsl:param name="removeEntryConfig"/>
        <xsl:variable name="removeAll" select="not($removeEntryConfig/@fromCCWindow  or $removeEntryConfig/@fromElementsView  or $removeEntryConfig/@fromEntitiesView  or $removeEntryConfig/@fromMenusfromMenus)"/>
        <contextRemoveItem>
            <field name="targets">
                <list>
                    <xsl:if test="$removeAll or $removeEntryConfig/xs:boolean(@fromCCWindow)">
                        <Integer><xsl:value-of select="$CC_WINDOW_TARGET"/></Integer>
                    </xsl:if>
                    <xsl:if test="$removeAll or $removeEntryConfig/xs:boolean(@fromMenus)">
                        <Integer><xsl:value-of select="$CC_POP_UP_MENUS_TARGET"/></Integer>
                    </xsl:if>
                    <xsl:if test="$removeAll or $removeEntryConfig/xs:boolean(@fromElementsView)">
                        <Integer><xsl:value-of select="$CC_ELEMENT_VIEWS_TARGET"/></Integer>
                    </xsl:if>
                    <xsl:if test="$removeAll or $removeEntryConfig/xs:boolean(@fromEntitiesView)">
                        <Integer><xsl:value-of select="$CC_ENTITY_VIEWS_TARGET"/></Integer>
                    </xsl:if>
                </list>
            </field>
            <field name="renderName">
                <String><xsl:value-of select="$removeEntryConfig/@renderName"/></String>
            </field>
        </contextRemoveItem>
    </xsl:function>
  
    <xd:doc xmlns:xd="http://www.oxygenxml.com/ns/doc/xsl">
        <xd:desc>Updates the toolbar configuration.</xd:desc>
        <xd:param name="removeEntryConfig">Remove entry configuration.</xd:param>
    </xd:doc>  
    <xsl:function name="util:createToolbarItem">
        <xsl:param name="addActionElement"/>
        
        <toolbarItem>
            <field name="actionID">
                <String><xsl:value-of select="$addActionElement/@id"/></String>
            </field>
        </toolbarItem>
    </xsl:function>
    
    <xsl:function name="util:createToolbarSeparator">
        <xsl:param name="separator"/>
        
        <toolbarSeparator/>
    </xsl:function>

    
    <xsl:function name="util:createToolbar">
        <xsl:param name="addGroupElement"/>
        
        <xsl:sequence select="util:createToolbar($addGroupElement, $TOOLBAR_TYPE_SUBTOOLBAR)"/>
    </xsl:function>
    
    <xsl:function name="util:createMainToolbar">
        <xsl:param name="addGroupElement"/>
        
        <xsl:sequence select="util:createToolbar($addGroupElement, $TOOLBAR_TYPE_MAIN)"/>
    </xsl:function>
    
    <xsl:function name="util:createToolbar">
        <xsl:param name="addGroupElement"/>
        <xsl:param name="type" as="xs:integer"/>
        
        <toolbar>
            <field name="id">
                <String><xsl:value-of select="$addGroupElement/@name"/></String>
            </field>
            <field name="type">
                <Integer><xsl:value-of select="$type"/></Integer>
            </field>
            <field name="largeIconPath">
                <String><xsl:value-of select="if ($addGroupElement/@largeIconPath) then $addGroupElement/@largeIconPath else ()"/></String>
            </field>
            <field name="smallIconPath">
                <String><xsl:value-of select="if ($addGroupElement/@smallIconPath) then $addGroupElement/@smallIconPath else ()"/></String>
            </field>
            <field name="toolbarEntriesDescriptorList">
                <toolbarEntry-array>
                    <xsl:for-each select="$addGroupElement/*">
                        <xsl:sequence select="util:createToolbarEntry(.)"/>
                    </xsl:for-each>
                </toolbarEntry-array>
            </field>
        </toolbar>
    </xsl:function>
    
    <xsl:function name="util:createToolbarEntry">
        <xsl:param name="directive"/>
        <xsl:choose>
            <xsl:when test="$directive[self::s:addAction]">
                <xsl:sequence select="util:createToolbarItem($directive)"/>
            </xsl:when>
            <xsl:when test="$directive[self::s:group]">
                <xsl:sequence select="util:createToolbar($directive)"/>
            </xsl:when>
            <xsl:when test="$directive[self::s:separator]">
                <xsl:sequence select="util:createToolbarSeparator($directive)"/>
            </xsl:when>
        </xsl:choose>
    </xsl:function>
    
    
    <!-- 
        
        MENUS
    
    -->
    
    <xsl:function name="util:createMenuEntry">
        <xsl:param name="directive"/>
        <xsl:choose>
            <xsl:when test="$directive[self::s:addAction]">
                <xsl:sequence select="util:createMenuItem($directive)"/>
            </xsl:when>
            <xsl:when test="$directive[self::s:submenu]">
                <xsl:sequence select="util:createSubmenu($directive)"/>
            </xsl:when>
            <xsl:when test="$directive[self::s:separator]">
                <xsl:sequence select="util:createMenuSeparator($directive)"/>
            </xsl:when>
        </xsl:choose>
    </xsl:function>
    
    <xsl:function name="util:createMenuItem">
        <xsl:param name="addActionElement"/>
        
        <menuItem>
            <field name="actionID">
                <String><xsl:value-of select="$addActionElement/@id"/></String>
            </field>
        </menuItem>
    </xsl:function>
    
    <xsl:function name="util:createMenuSeparator">
        <xsl:param name="separator"/>
        
        <menuSeparator/>
    </xsl:function>
    
    <xsl:function name="util:createSubmenu">
        <xsl:param name="addGroupElement"/>
        
        <menu>
            <field name="label">
                <String><xsl:value-of select="$addGroupElement/@name"/></String>
            </field>
            <field name="accessKey">
                <String></String>
            </field>
            <field name="iconPath">
                <String><xsl:value-of select="$addGroupElement/@iconPath"/></String>
            </field>
            <field name="menuEntriesDescriptorList">
                <menuEntry-array>
                    <xsl:for-each select="$addGroupElement/*">
                        <xsl:sequence select="util:createMenuEntry(.)"/>
                    </xsl:for-each>
                </menuEntry-array>
            </field>
            <field name="context">
                <null/>
            </field>
        </menu>
        
    </xsl:function>
    
    <xsl:function name="util:isPresentInBaseToolbars" as="xs:boolean">
        <!-- A toolbarEntry-array -->
        <xsl:param name="toolbarEntryArray"/>
        <xsl:param name="name" as="xs:string?"/>
        
        <xsl:variable name="existing" select="$toolbarEntryArray/toolbar/field[@name='id']/String/text()"></xsl:variable>
        
        <xsl:value-of select="$name and $existing and not(empty(index-of($existing, $name)))"/>
    </xsl:function>
    
    <xsl:function name="util:isPresentInBaseMenus" as="xs:boolean">
        <!-- A toolbarEntry-array -->
        <xsl:param name="menuEntryArray"/>
        <xsl:param name="name" as="xs:string?"/>
        
        <xsl:variable name="existing" select="$menuEntryArray/menu/field[@name='label']/String/text()"></xsl:variable>
        
        <xsl:value-of select="$name and $existing and not(empty(index-of($existing, $name)))"/>
    </xsl:function>
    
</xsl:stylesheet>