<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet 
    xmlns:xsl="http://www.w3.org/1999/XSL/Transform" 
    xmlns:xs="http://www.w3.org/2001/XMLSchema" 
    xmlns:fo="http://www.w3.org/1999/XSL/Format" 
    xmlns:oxy="http://www.oxygenxml.com/css2fo"
    xmlns:chfox="http://www.oxygenxml.com/chemistry/extensions/fop"
    xmlns:css="http://www.w3.org/1998/CSS" 
    xmlns:fox="http://xmlgraphics.apache.org/fop/extensions"
    exclude-result-prefixes="xs oxy" version="2.0"
    xpath-default-namespace="http://www.w3.org/1998/CSS">
    
    <!-- Default page size is US-LETTER with 1in margins -->
    <xsl:variable name="const-def-margin" select="'1in'"/>
    <xsl:variable name="const-def-page-width" select="'8.5in'"/>
    <xsl:variable name="const-def-page-height" select="'11in'"/>
    
    <xsl:variable name="debug" select="false()"/>
    
    <!-- 
        Generates the page sequence master based on the CSS @page definitions  
        To accomodate all the left/right/first pseudo page combinations, the page 
        master uses an alternate of simple page masters.
        
        Context: Document.
    -->
    <xsl:template name="generate-page-masters">
        <xsl:variable name="doc" select="/"/>
        
        <xsl:for-each select="distinct-values(('css2fo-default', /root/pages/page/@name))">
            <xsl:variable name="page-name" select="."/>
            <xsl:comment> Page: <xsl:value-of select="$page-name"/> </xsl:comment>
            
            <!-- Generate the master references combinations -->
            <fo:page-sequence-master master-name="{$page-name}">
                <fo:repeatable-page-master-alternatives>
                    <xsl:for-each select="oxy:get-used-page-sides($doc, $page-name)">
                        <xsl:variable name="page-side" select="."/>
                        <xsl:for-each select="oxy:get-used-page-positions($doc, $page-name)">
                            <xsl:variable name="page-position" select="."/>
                            <xsl:for-each select="oxy:get-used-page-blanks($doc, $page-name)">
                                <xsl:variable name="page-blank" select="."/>

                                <fo:conditional-page-master-reference master-reference="spm-{oxy:generate-name($page-name, $page-side, $page-position, $page-blank)}">
                                    <!-- Page side condition -->
                                    <xsl:attribute name="odd-or-even">
                                        <xsl:choose>
                                            <xsl:when test="$page-side = 'left'">
                                                <xsl:value-of select="'even'"/>
                                            </xsl:when>
                                            <xsl:when test="$page-side = 'right'">
                                                <xsl:value-of select="'odd'"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="'any'"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:attribute>
                                    <!-- Page position condition -->
                                    <xsl:attribute name="page-position">
                                        <xsl:choose>
                                            <xsl:when test="$page-position = 'first'">
                                                <xsl:value-of select="'first'"/>
                                            </xsl:when>
                                            <xsl:when test="$page-position = 'last'">
                                                <xsl:value-of select="'last'"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="'any'"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:attribute>
                                    <!-- Page blank condition -->
                                    <xsl:attribute name="blank-or-not-blank">
                                        <xsl:choose>
                                            <xsl:when test="$page-blank = 'blank'">
                                                <xsl:value-of select="'blank'"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="'any'"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:attribute>
                                </fo:conditional-page-master-reference>
                            </xsl:for-each>
                        </xsl:for-each>
                    </xsl:for-each>
                </fo:repeatable-page-master-alternatives>
            </fo:page-sequence-master>
            
            <xsl:for-each select="oxy:get-used-page-sides($doc, $page-name)">
                <xsl:variable name="page-side" select="."/>
                <xsl:for-each select="oxy:get-used-page-positions($doc, $page-name)">
                    <xsl:variable name="page-position" select="."/>
                    <xsl:for-each select="oxy:get-used-page-blanks($doc, $page-name)">
                        <xsl:variable name="page-blank" select="."/>
                        
                        <!-- Distribute page properties either to the page master, or the region body. -->
                        <fo:simple-page-master master-name="spm-{oxy:generate-name($page-name, $page-side, $page-position, $page-blank)}">
                            
                            <!-- The width and height.-->
                            <xsl:variable name="pw" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'width')"/>
                            <xsl:variable name="ph" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'height')"/>
                            <xsl:if test="$ph">
                                <xsl:attribute name="page-width" select="$pw"/>
                            </xsl:if>
                            <xsl:if test="$ph">
                                <xsl:attribute name="page-height" select="$ph"/>
                            </xsl:if>
                            <!-- Add the foreground element if exists -->
                            <xsl:variable name="page-foreground-properties" select="oxy:get-page-foreground-properties($doc, $page-name, $page-side, $page-position, $page-blank)"/>
                            <xsl:if test="$page-foreground-properties">
                                <xsl:attribute name="chfox:foreground-image" select="$page-foreground-properties/@value"/>
                            </xsl:if>
                            
                            <!-- Prepress properties. -->
                            <xsl:variable name="bleed" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'bleed')"/>
                            <xsl:if test="$bleed">
                                <xsl:attribute name="fox:bleed" select="$bleed"/>
                            </xsl:if>
                            <xsl:variable name="crop-offset" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'oxy-crop-offset')"/>
                            <xsl:if test="$crop-offset">
                                <xsl:attribute name="fox:crop-offset" select="$crop-offset"/>
                            </xsl:if>
                            <xsl:variable name="crop-box" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'oxy-crop-box')"/>
                            <xsl:if test="$crop-box">
                                <xsl:attribute name="fox:crop-box" select="$crop-box"/>
                            </xsl:if>
                            <xsl:variable name="scale" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'oxy-scale')"/>
                            <xsl:if test="$scale">
                                <xsl:attribute name="fox:scale" select="$scale"/>
                            </xsl:if>
                            <xsl:variable name="marks" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'marks')"/>
                            <xsl:if test="$marks">
                                <xsl:attribute name="chfox:marks" select="$marks"/>
                            </xsl:if>
                            
                            <xsl:variable name="page-top-margin" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'margin-top')"/>
                            <xsl:variable name="page-bottom-margin" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'margin-bottom')"/>
                            <xsl:variable name="page-left-margin" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'margin-left')"/>
                            <xsl:variable name="page-right-margin" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'margin-right')"/>
                            
                            <fo:region-body>
                                
                                <!-- The border and padding works if FOP is not in strict validation mode. -->
                                <xsl:call-template name="copy-properties-by-prefix">
                                    <xsl:with-param name="doc" select="$doc"/>
                                    <xsl:with-param name="page-name" select="$page-name"/>
                                    <xsl:with-param name="page-side" select="$page-side"/>
                                    <xsl:with-param name="page-position" select="$page-position"/>
                                    <xsl:with-param name="page-blank" select="$page-blank"/>
                                    
                                    <xsl:with-param name="props" xml:space="preserve">padding, border</xsl:with-param>
                                </xsl:call-template>
                                
                                <!-- Other properties -->
                                <xsl:call-template name="copy-properties-by-prefix">
                                    <xsl:with-param name="doc" select="$doc"/>
                                    <xsl:with-param name="page-name" select="$page-name"/>
                                    <xsl:with-param name="page-side" select="$page-side"/>
                                    <xsl:with-param name="page-position" select="$page-position"/>
                                    <xsl:with-param name="page-blank" select="$page-blank"/>
                                    
                                    <xsl:with-param name="props" xml:space="preserve">color, column-count, column-gap</xsl:with-param>
                                </xsl:call-template>
                                
                                <!-- The page margins, taken from the CSS, or from defaults -->
                                <xsl:attribute name="margin-top" select="$page-top-margin"/>
                                <xsl:attribute name="margin-bottom" select="$page-bottom-margin"/>
                                <xsl:attribute name="margin-left" select="$page-left-margin"/>
                                <xsl:attribute name="margin-right" select="$page-right-margin"/>
                                
                            </fo:region-body>
                            
                            <!-- The extent must be equal to the page top margin-->
                            <fo:region-before region-name="{oxy:generate-name('sc-before', $page-side, $page-position, $page-blank)}">
                                <xsl:if test="string-length($page-top-margin) > 0">
                                    <xsl:attribute name="extent" select="$page-top-margin"/>
                                </xsl:if>
                            </fo:region-before>
                            
                            <!-- The extent must be equal to the page bottom margin-->
                            <fo:region-after region-name="{oxy:generate-name('sc-after', $page-side, $page-position, $page-blank)}">
                                <xsl:if test="string-length($page-bottom-margin) > 0">
                                    <xsl:attribute name="extent" select="$page-bottom-margin"/>
                                </xsl:if>
                            </fo:region-after>
                            
                            <!-- The extent of the must be equal to the page left margin-->
                            <fo:region-start region-name="{oxy:generate-name('sc-start', $page-side, $page-position, $page-blank)}">
                                <xsl:if test="string-length($page-left-margin) > 0">
                                    <xsl:attribute name="extent" select="$page-left-margin"/>
                                </xsl:if>
                            </fo:region-start>
                            
                            <!-- The extent of the must be equal to the page right margin-->
                            <fo:region-end region-name="{oxy:generate-name('sc-end', $page-side, $page-position, $page-blank)}">
                                <xsl:if test="string-length($page-right-margin) > 0">
                                    <xsl:attribute name="extent" select="$page-right-margin"/>
                                </xsl:if>
                            </fo:region-end>
                            
                        </fo:simple-page-master>
                    </xsl:for-each>
                </xsl:for-each>
            </xsl:for-each>
        </xsl:for-each>
    </xsl:template>
    
    <!-- 
        Generates a name. Concatenates all the parameters, adding only the ones that have 'any' empty values.
        
        @return a string.
    -->
    <xsl:function name="oxy:generate-name" as="xs:string">
        <xsl:param name="param0"/>
        <xsl:param name="param1"/>
        <xsl:param name="param2"/>
        <xsl:param name="param3"/>
        
        <xsl:variable name="name" select="''"/>
        <xsl:variable name="name" select="
            if ($param0 != 'any') then
                concat($name, $param0)
            else
                $name"/>
        <xsl:variable name="name" select="
            if ($param1 != 'any') then
                concat($name, '-', $param1)
            else
                $name"/>
        <xsl:variable name="name" select="
            if ($param2 != 'any') then
                concat($name, '-', $param2)
            else
                $name"/>
        <xsl:variable name="name" select="
            if ($param3 != 'any') then
                concat($name, '-', $param3)
            else
                $name"/>
        
        <xsl:value-of select="$name"/>
    </xsl:function>
    
    <!-- 
        This template populates all the regions that were declared in the generate-page-masters template with the static content
        extracted from the CSS page margin boxes definition.
        
        For example, having:
        
        @page intro {
        @top-left-corner { content: "Introduction" }
        }
        
        will result in a fo:inline put in a block position at the top of the fo-region named "start".
        
        The static regions must be defined every time content is switching page name. 
        
        Context: css:page-sequence
    -->
    <xsl:template name="generate-page-static">
        <xsl:param name="doc" required="yes"/>
        <xsl:param name="page-name" required="yes"/>
        
        <xsl:for-each select="oxy:get-used-page-sides($doc, $page-name)">
            <xsl:variable name="page-side" select="."/>
            <xsl:for-each select="oxy:get-used-page-positions($doc, $page-name)">
                <xsl:variable name="page-position" select="."/>
                <xsl:for-each select="oxy:get-used-page-blanks($doc, $page-name)">
                    <xsl:variable name="page-blank" select="."/>
                    
                    <!-- We use 'any' so in the end it matches only the generic pages, without any pseudo for position or side.  -->
                    <xsl:variable name="page-top-margin" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'margin-top', $const-def-margin)"/>
                    <xsl:variable name="page-bottom-margin" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'margin-bottom', $const-def-margin)"/>
                    <xsl:variable name="page-left-margin" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'margin-left', $const-def-margin)"/>
                    <xsl:variable name="page-right-margin" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'margin-right', $const-def-margin)"/>
                    <!-- The default is processed at a lower level. -->
                    <xsl:variable name="page-width" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'width')"/>
                    <xsl:variable name="page-height" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'height')"/>
                    
                    <!-- 
                        ======================================
                        TOP
                        ======================================
                    -->
                    <xsl:if test="
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'top-left') or
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'top-center') or
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'top-right') or
                        oxy:get-page-background-properties($doc, $page-name, $page-side, $page-position, $page-blank)
                        ">
                        <xsl:call-template name="generate-page-statics-top">
                            <xsl:with-param name="doc" select="$doc"/>
                            <xsl:with-param name="page-name" select="$page-name"/>
                            <xsl:with-param name="page-side" select="$page-side"/>
                            <xsl:with-param name="page-position" select="$page-position"/>
                            <xsl:with-param name="page-blank" select="$page-blank"/>
                            
                            <xsl:with-param name="page-top-margin" select="$page-top-margin"/>
                            <xsl:with-param name="page-left-margin" select="$page-left-margin"/>
                            <xsl:with-param name="page-right-margin" select="$page-right-margin"/>
                            <xsl:with-param name="page-width" select="$page-width"/>
                            <!-- Needed for the page background. -->
                            <xsl:with-param name="page-height" select="$page-height"/>
                        </xsl:call-template>
                    </xsl:if>
                    
                    <!-- 
                        ======================================
                        BOTTOM
                        ======================================
                    -->
                    <xsl:if test="
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-left') or
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-center') or
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-right')
                        ">
                        <xsl:call-template name="generate-page-statics-bottom">
                            <xsl:with-param name="doc" select="$doc"/>
                            <xsl:with-param name="page-name" select="$page-name"/>
                            <xsl:with-param name="page-side" select="$page-side"/>
                            <xsl:with-param name="page-position" select="$page-position"/>
                            <xsl:with-param name="page-blank" select="$page-blank"/>
                            
                            <xsl:with-param name="page-left-margin" select="$page-left-margin"/>
                            <xsl:with-param name="page-bottom-margin" select="$page-bottom-margin"/>
                            <xsl:with-param name="page-right-margin" select="$page-right-margin"/>
                            <xsl:with-param name="page-width" select="$page-width"/>
                        </xsl:call-template>
                    </xsl:if>
                    
                    <!-- 
                        ======================================
                        LEFT
                        ======================================
                    -->
                    
                    <xsl:if test="
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'top-left-corner') or
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'left-top') or
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'left-middle') or
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'left-bottom') or
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-left-corner')
                        ">
                        <xsl:call-template name="generate-page-statics-left">
                            <xsl:with-param name="doc" select="$doc"/>
                            <xsl:with-param name="page-name" select="$page-name"/>
                            <xsl:with-param name="page-side" select="$page-side"/>
                            <xsl:with-param name="page-position" select="$page-position"/>
                            <xsl:with-param name="page-blank" select="$page-blank"/>
                            
                            <xsl:with-param name="page-top-margin" select="$page-top-margin"/>
                            <xsl:with-param name="page-left-margin" select="$page-left-margin"/>
                            <xsl:with-param name="page-bottom-margin" select="$page-bottom-margin"/>
                            <xsl:with-param name="page-right-margin" select="$page-right-margin"/>
                            <xsl:with-param name="page-height" select="$page-height"/>
                        </xsl:call-template>
                    </xsl:if>
                    
                    <!-- 
                        ======================================
                        RIGHT
                        ======================================
                    -->
                    <xsl:if test="
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'top-right-corner') or
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'right-top') or
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'right-middle') or
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'right-bottom') or
                        oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-right-corner')
                        ">
                        <xsl:call-template name="generate-page-statics-right">
                            <xsl:with-param name="doc" select="$doc"/>
                            <xsl:with-param name="page-name" select="$page-name"/>
                            <xsl:with-param name="page-side" select="$page-side"/>
                            <xsl:with-param name="page-position" select="$page-position"/>
                            <xsl:with-param name="page-blank" select="$page-blank"/>
                            
                            <xsl:with-param name="page-top-margin" select="$page-top-margin"/>
                            <xsl:with-param name="page-left-margin" select="$page-left-margin"/>
                            <xsl:with-param name="page-bottom-margin" select="$page-bottom-margin"/>
                            <xsl:with-param name="page-right-margin" select="$page-right-margin"/>
                            <xsl:with-param name="page-height" select="$page-height"/>
                        </xsl:call-template>
                    </xsl:if>
                    
                </xsl:for-each>
            </xsl:for-each>
        </xsl:for-each>
        
        <!--
            ======================================
            FOOTNOTES
            ======================================
        -->
        <xsl:if test="
            oxy:exist-page-margin-box($doc, $page-name, 'any', 'any', 'any', 'footnote')
            ">
            <xsl:call-template name="generate-page-statics-footnote">
                <xsl:with-param name="doc" select="$doc"/>
                <xsl:with-param name="page-name" select="$page-name"/>
                <xsl:with-param name="page-side" select="'any'"/>
                <xsl:with-param name="page-position" select="'any'"/>
                <xsl:with-param name="page-blank" select="'any'"/>
            </xsl:call-template>
        </xsl:if>
        
    </xsl:template>
    
    <!-- 
        Gets a not empty sequence with all the page side pseudos used in the page definitions.
        
        @param doc The document.
        @param page-name The name of the page we are interested in its pseudos. The default page is taken into account as well.
        @return ('left', 'right') , ('left','any') or other combinations.    
    -->
    
    <xsl:function name="oxy:get-used-page-sides" as="xs:anyAtomicType*">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:choose>
            <xsl:when test="
                $doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'left')] and
                $doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'right')]">
                <xsl:sequence select="('left', 'right')"/>
            </xsl:when>
            <xsl:when test="
                $doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'left')] and
                not($doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'right')])">
                <xsl:sequence select="('left', 'any')"/>
            </xsl:when>
            <xsl:when test="
                not($doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'left')]) and
                $doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'right')]">
                <xsl:sequence select="('right', 'any')"/>
            </xsl:when>
            <xsl:when test="
                not($doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'left')]) and
                not($doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'right')])">
                <xsl:sequence select="('any')"/>
            </xsl:when>
        </xsl:choose>
    </xsl:function>
    
    <!-- 
        Gets a not empty sequence with all the position pseudos used in the page definitions.
        
        @param doc The document.
        @param page-name The name of the page we are interested in its pseudos. The default page is taken into account as well.
        
        @return ('first', 'last', 'any') or other combinations.    
    -->
    <xsl:function name="oxy:get-used-page-positions" as="xs:anyAtomicType*">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:choose>
            <xsl:when test="
                $doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'first')] and
                $doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'last')]">
                <xsl:sequence select="('first', 'last', 'any')"/>
            </xsl:when>
            <xsl:when test="
                $doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'first')] and
                not($doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'last')])">
                <xsl:sequence select="('first', 'any')"/>
            </xsl:when>
            <xsl:when test="
                not($doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'first')]) and
                $doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'last')]">
                <xsl:sequence select="('last', 'any')"/>
            </xsl:when>
            <xsl:when test="
                not($doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'first')]) and
                not($doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'last')])">
                <xsl:sequence select="('any')"/>
            </xsl:when>
        </xsl:choose>
    </xsl:function>
    
    <!-- 
        Gets a not empty sequence with all the blank pseudos used in the page definitions.
        
        @param doc The document.
        @param page-name The name of the page we are interested in its pseudos. The default page is taken into account as well.
        
        @return ('blank', 'any') or ('any')    
    -->
    <xsl:function name="oxy:get-used-page-blanks" as="xs:anyAtomicType*">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:choose>
            <xsl:when test="
                $doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'blank')]">
                <xsl:sequence select="('blank', 'any')"/>
            </xsl:when>
            <xsl:when test="
                not($doc/root/pages/page
                [@name = $page-name or not(@name) or @name = 'css2fo-default']
                [contains(@pseudo, 'blank')])">
                <xsl:sequence select="('any')"/>
            </xsl:when>
        </xsl:choose>
    </xsl:function>
    
    <!-- 
        Generates in the "end" FO region, the static extracted from:
        @top-right-corner, @right-top, @right-middle, @right-bottom, @bottom-right-corner     
    -->
    <xsl:template name="generate-page-statics-right">
        <xsl:param name="doc" required="yes"/>
        <xsl:param name="page-name" required="yes"/>
        <xsl:param name="page-side" required="yes"/>
        <xsl:param name="page-position" required="yes"/>
        <xsl:param name="page-blank" required="yes"/>
        
        <xsl:param name="page-top-margin" required="yes"/>
        <xsl:param name="page-left-margin" required="yes"/>
        <xsl:param name="page-bottom-margin" required="yes"/>
        <xsl:param name="page-right-margin" required="yes"/>
        <xsl:param name="page-height" required="yes"/>
        
        <xsl:variable name="fullH" select="concat('(', $page-height, ' - ', $page-top-margin, ' - ', $page-bottom-margin, ')')"/>
        <xsl:variable name="thirdH" select="concat($fullH, ' div 3')"/>
        <xsl:variable name="halfH" select="concat($fullH, ' div 2')"/>
        
        <xsl:variable name="nrOfBoxes" select="oxy:count-page-margin-boxes-with-content($doc, $page-name, $page-side, $page-position, $page-blank, 'top-right-corner, right-top, right-middle, right-bottom, bottom-right-corner')"/>
        <xsl:if test="$nrOfBoxes > 0">
            <fo:static-content flow-name="{oxy:generate-name('sc-end', $page-side, $page-position, $page-blank)}" role="artifact">
                <!-- TOP-RIGHT corner -->
                <xsl:comment>TOP-RIGHT-CORNER</xsl:comment>
                
                <xsl:variable name="margin-box-width" select="oxy:adjust-page-margin-box-width($page-right-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'top-right-corner')"/>
                <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($page-top-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'top-right-corner')"/>
                
                <fo:block-container width="{$margin-box-width}" height="{$margin-box-height}" margin="0">
                    <!-- TODO what to do with the text alignment? These should be only defaults, more important are the ones from the CSS. We should treat them for all boxes. -->
                    <xsl:choose>
                        <xsl:when test="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'top-right-corner')">
                            <xsl:call-template name="generate-properties-for-margin-box">
                                <xsl:with-param name="doc" select="$doc"/>
                                <xsl:with-param name="page-name" select="$page-name"/>
                                <xsl:with-param name="page-side" select="$page-side"/>
                                <xsl:with-param name="page-position" select="$page-position"/>
                                <xsl:with-param name="page-blank" select="$page-blank"/>
                                <xsl:with-param name="margin-box-name" select="'top-right-corner'"/>
                                <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                
                                <xsl:with-param name="default-text-align" select="'left'"/>
                                <xsl:with-param name="default-display-align" select="'center'"/>
                            </xsl:call-template>
                            <xsl:call-template name="generate-content-for-page-margin-box">
                                <xsl:with-param name="doc" select="$doc"/>
                                <xsl:with-param name="page-name" select="$page-name"/>
                                <xsl:with-param name="page-side" select="$page-side"/>
                                <xsl:with-param name="page-position" select="$page-position"/>
                                <xsl:with-param name="page-blank" select="$page-blank"/>
                                <xsl:with-param name="margin-box-name" select="'top-right-corner'"/>
                                <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                            </xsl:call-template>
                        </xsl:when>
                        <xsl:otherwise>
                            <fo:block/>
                        </xsl:otherwise>
                    </xsl:choose>
                </fo:block-container>
                
                <!-- This container holds the "right-top", "right-middle", "right-bottom" boxes, has the height of the page minus the top and bottom margins. -->
                <fo:block-container width="{$page-right-margin}" height="{$fullH}">
                    <xsl:variable name="nrOfBoxes" select="oxy:count-page-margin-boxes-with-content($doc, $page-name, $page-side, $page-position, $page-blank, 'right-top, right-middle, right-bottom')"/>
                    <xsl:choose>
                        <xsl:when test="$nrOfBoxes > 0">
                            
                            <xsl:variable name="exists-top" select="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'right-top')"/>
                            <xsl:variable name="exists-middle" select="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'right-middle')"/>
                            <xsl:variable name="exists-bottom" select="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'right-bottom')"/>
                            
                            <!-- The boxes span an equal height. -->
                            <xsl:variable name="boxHeight">
                                <xsl:choose>
                                    <xsl:when test="$nrOfBoxes = 1">
                                        <xsl:value-of select="$fullH"/>
                                    </xsl:when>
                                    <xsl:when test="$nrOfBoxes = 2 and not($exists-middle)">
                                        <xsl:value-of select="$halfH"/>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:value-of select="$thirdH"/>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </xsl:variable>
                            
                            <xsl:if test="$exists-top or ($exists-middle and $exists-bottom)">
                                <xsl:comment>RIGHT-TOP</xsl:comment>
                                
                                <xsl:variable name="margin-box-width" select="oxy:adjust-page-margin-box-width($page-right-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'right-top')"/>
                                <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($boxHeight, $doc, $page-name, $page-side, $page-position, $page-blank, 'right-top')"/>
                                
                                <fo:block-container width="{$margin-box-width}" height="{$margin-box-height}" margin="0">
                                    <xsl:call-template name="generate-properties-for-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'right-top'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                        
                                        <xsl:with-param name="default-text-align" select="'center'"/>
                                        <xsl:with-param name="default-display-align" select="'before'"/>
                                    </xsl:call-template>
                                    <xsl:call-template name="generate-content-for-page-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'right-top'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                    </xsl:call-template>
                                </fo:block-container>
                            </xsl:if>
                            
                            <xsl:if test="$exists-middle">
                                <xsl:comment>RIGHT-MIDDLE</xsl:comment>
                                
                                <xsl:variable name="margin-box-width" select="oxy:adjust-page-margin-box-width($page-right-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'right-middle')"/>
                                <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($boxHeight, $doc, $page-name, $page-side, $page-position, $page-blank, 'right-middle')"/>
                                
                                <fo:block-container width="{$margin-box-width}" height="{$margin-box-height}" margin="0">
                                    <xsl:call-template name="generate-properties-for-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'right-middle'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                        
                                        <xsl:with-param name="default-text-align" select="'center'"/>
                                        <xsl:with-param name="default-display-align" select="'center'"/>
                                    </xsl:call-template>
                                    <xsl:call-template name="generate-content-for-page-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'right-middle'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                    </xsl:call-template>
                                </fo:block-container>
                            </xsl:if>
                            
                            <xsl:if test="$exists-bottom">
                                <xsl:comment>RIGHT-BOTTOM</xsl:comment>
                                
                                <xsl:variable name="margin-box-width" select="oxy:adjust-page-margin-box-width($page-right-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'right-bottom')"/>
                                <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($boxHeight, $doc, $page-name, $page-side, $page-position, $page-blank, 'right-bottom')"/>
                                
                                <fo:block-container width="{$margin-box-width}" height="{$margin-box-height}" margin="0">
                                    <xsl:call-template name="generate-properties-for-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'right-bottom'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                        
                                        <xsl:with-param name="default-text-align" select="'center'"/>
                                        <xsl:with-param name="default-display-align" select="'after'"/>
                                    </xsl:call-template>
                                    <xsl:call-template name="generate-content-for-page-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'right-bottom'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                    </xsl:call-template>
                                </fo:block-container>
                            </xsl:if>
                            
                        </xsl:when>
                        <xsl:otherwise>
                            <fo:block/>
                        </xsl:otherwise>
                    </xsl:choose>
                </fo:block-container>
                
                <!-- BOTTOM-RIGHT corner -->
                <xsl:comment>BOTTOM-RIGHT-CORNER</xsl:comment>
                
                <xsl:variable name="margin-box-width" select="oxy:adjust-page-margin-box-width($page-right-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-right-corner')"/>
                <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($page-bottom-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-right-corner')"/>
                
                <fo:block-container width="{$margin-box-width}" height="{$margin-box-height}" margin="0">
                    <xsl:choose>
                        <xsl:when test="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-right-corner')">
                            <xsl:call-template name="generate-properties-for-margin-box">
                                <xsl:with-param name="doc" select="$doc"/>
                                <xsl:with-param name="page-name" select="$page-name"/>
                                <xsl:with-param name="page-side" select="$page-side"/>
                                <xsl:with-param name="page-position" select="$page-position"/>
                                <xsl:with-param name="page-blank" select="$page-blank"/>
                                <xsl:with-param name="margin-box-name" select="'bottom-right-corner'"/>
                                <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                
                                <xsl:with-param name="default-text-align" select="'left'"/>
                                <xsl:with-param name="default-display-align" select="'center'"/>
                            </xsl:call-template>
                            <xsl:call-template name="generate-content-for-page-margin-box">
                                <xsl:with-param name="doc" select="$doc"/>
                                <xsl:with-param name="page-name" select="$page-name"/>
                                <xsl:with-param name="page-side" select="$page-side"/>
                                <xsl:with-param name="page-position" select="$page-position"/>
                                <xsl:with-param name="page-blank" select="$page-blank"/>
                                <xsl:with-param name="margin-box-name" select="'bottom-right-corner'"/>
                                <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                            </xsl:call-template>
                        </xsl:when>
                        <xsl:otherwise>
                            <fo:block/>
                        </xsl:otherwise>
                    </xsl:choose>
                </fo:block-container>
            </fo:static-content>
        </xsl:if>
    </xsl:template>
    
    <!-- 
        Generates in the "start" FO region, the static extracted from:
        @top-left-corner, @left-top, @left-middle, @left-bottom, @left-right-corner     
    -->
    <xsl:template name="generate-page-statics-left">
        <xsl:param name="doc" required="yes"/>
        <xsl:param name="page-name" required="yes"/>
        <xsl:param name="page-side" required="yes"/>
        <xsl:param name="page-position" required="yes"/>
        <xsl:param name="page-blank" required="yes"/>
        
        <xsl:param name="page-top-margin" required="yes"/>
        <xsl:param name="page-left-margin" required="yes"/>
        <xsl:param name="page-bottom-margin" required="yes"/>
        <xsl:param name="page-right-margin" required="yes"/>
        <xsl:param name="page-height" required="yes"/>
        
        <xsl:variable name="fullH" select="concat('(', $page-height, ' - ', $page-top-margin, ' - ', $page-bottom-margin, ')')"/>
        <xsl:variable name="thirdH" select="concat($fullH, ' div 3')"/>
        <xsl:variable name="halfH" select="concat($fullH, ' div 2')"/>
        
        <xsl:variable name="nrOfBoxes" select="oxy:count-page-margin-boxes-with-content($doc, $page-name, $page-side, $page-position, $page-blank, 'top-left-corner, left-top, left-middle, left-bottom, bottom-left-corner')"/>
        <xsl:if test="$nrOfBoxes > 0">
            <fo:static-content flow-name="{oxy:generate-name('sc-start', $page-side, $page-position, $page-blank)}" role="artifact">
                <!-- TOP-LEFT corner -->
                <xsl:comment>TOP-LEFT-CORNER</xsl:comment>
                
                <xsl:variable name="margin-box-width" select="oxy:adjust-page-margin-box-width($page-left-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'top-left-corner')"/>
                <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($page-top-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'top-left-corner')"/>
                
                <fo:block-container width="{$margin-box-width}" height="{$margin-box-height}" margin="0">
                    <xsl:choose>
                        <xsl:when test="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'top-left-corner')">
                            <xsl:call-template name="generate-properties-for-margin-box">
                                <xsl:with-param name="doc" select="$doc"/>
                                <xsl:with-param name="page-name" select="$page-name"/>
                                <xsl:with-param name="page-side" select="$page-side"/>
                                <xsl:with-param name="page-position" select="$page-position"/>
                                <xsl:with-param name="page-blank" select="$page-blank"/>
                                <xsl:with-param name="margin-box-name" select="'top-left-corner'"/>
                                <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                
                                <xsl:with-param name="default-text-align" select="'right'"/>
                                <xsl:with-param name="default-display-align" select="'center'"/>
                            </xsl:call-template>
                            <xsl:call-template name="generate-content-for-page-margin-box">
                                <xsl:with-param name="doc" select="$doc"/>
                                <xsl:with-param name="page-name" select="$page-name"/>
                                <xsl:with-param name="page-side" select="$page-side"/>
                                <xsl:with-param name="page-position" select="$page-position"/>
                                <xsl:with-param name="page-blank" select="$page-blank"/>
                                <xsl:with-param name="margin-box-name" select="'top-left-corner'"/>
                                <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                            </xsl:call-template>
                        </xsl:when>
                        <xsl:otherwise>
                            <fo:block/>
                        </xsl:otherwise>
                    </xsl:choose>
                </fo:block-container>
                
                <!-- This container holds the "left-top", "left-middle", "left-bottom" boxes, has the height of the page minus the top and bottom margins. -->
                <fo:block-container width="{$page-left-margin}" height="{$fullH}">
                    <xsl:variable name="nrOfBoxes" select="oxy:count-page-margin-boxes-with-content($doc, $page-name, $page-side, $page-position, $page-blank, 'left-top, left-middle, left-bottom')"/>
                    <xsl:choose>
                        <xsl:when test="$nrOfBoxes > 0">
                            
                            <xsl:variable name="exists-top" select="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'left-top')"/>
                            <xsl:variable name="exists-middle" select="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'left-middle')"/>
                            <xsl:variable name="exists-bottom" select="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'left-bottom')"/>
                            
                            <!-- The boxes span an equal height. -->
                            <xsl:variable name="boxHeight">
                                <xsl:choose>
                                    <xsl:when test="$nrOfBoxes = 1">
                                        <xsl:value-of select="$fullH"/>
                                    </xsl:when>
                                    <xsl:when test="$nrOfBoxes = 2 and not($exists-middle)">
                                        <xsl:value-of select="$halfH"/>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:value-of select="$thirdH"/>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </xsl:variable>
                            
                            <xsl:if test="$exists-top or ($exists-middle and $exists-bottom)">
                                <xsl:comment>LEFT-TOP</xsl:comment>
                                
                                <xsl:variable name="margin-box-width" select="oxy:adjust-page-margin-box-width($page-left-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'left-top')"/>
                                <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($boxHeight, $doc, $page-name, $page-side, $page-position, $page-blank, 'left-top')"/>
                                
                                <fo:block-container width="{$margin-box-width}" height="{$margin-box-height}" margin="0">
                                    <xsl:call-template name="generate-properties-for-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'left-top'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                        
                                        <xsl:with-param name="default-text-align" select="'center'"/>
                                        <xsl:with-param name="default-display-align" select="'before'"/>
                                    </xsl:call-template>
                                    <xsl:call-template name="generate-content-for-page-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'left-top'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                    </xsl:call-template>
                                </fo:block-container>
                            </xsl:if>
                            
                            <xsl:if test="$exists-middle">
                                <xsl:comment>LEFT-MIDDLE</xsl:comment>
                                
                                <xsl:variable name="margin-box-width" select="oxy:adjust-page-margin-box-width($page-left-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'left-middle')"/>
                                <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($boxHeight, $doc, $page-name, $page-side, $page-position, $page-blank, 'left-middle')"/>
                                
                                <fo:block-container width="{$margin-box-width}" height="{$margin-box-height}" margin="0">
                                    <xsl:call-template name="generate-properties-for-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'left-middle'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                        
                                        <xsl:with-param name="default-text-align" select="'center'"/>
                                        <xsl:with-param name="default-display-align" select="'center'"/>
                                    </xsl:call-template>
                                    <xsl:call-template name="generate-content-for-page-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'left-middle'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                    </xsl:call-template>
                                </fo:block-container>
                            </xsl:if>
                            
                            <xsl:if test="$exists-bottom">
                                <xsl:comment>LEFT-BOTTOM</xsl:comment>
                                
                                <xsl:variable name="margin-box-width" select="oxy:adjust-page-margin-box-width($page-left-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'left-bottom')"/>
                                <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($boxHeight, $doc, $page-name, $page-side, $page-position, $page-blank, 'left-bottom')"/>
                                
                                <fo:block-container width="{$margin-box-width}" height="{$margin-box-height}" margin="0">
                                    <xsl:call-template name="generate-properties-for-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'left-bottom'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                        
                                        <xsl:with-param name="default-text-align" select="'center'"/>
                                        <xsl:with-param name="default-display-align" select="'after'"/>
                                    </xsl:call-template>
                                    <xsl:call-template name="generate-content-for-page-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'left-bottom'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                    </xsl:call-template>
                                </fo:block-container>
                            </xsl:if>
                        </xsl:when>
                        <xsl:otherwise>
                            <fo:block/>
                        </xsl:otherwise>
                    </xsl:choose>
                </fo:block-container>
                
                <!-- BOTTOM-LEFT corner -->
                <xsl:comment>BOTTOM-LEFT-CORNER</xsl:comment>
                
                <xsl:variable name="margin-box-width" select="oxy:adjust-page-margin-box-width($page-left-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-left-corner')"/>
                <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($page-bottom-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-left-corner')"/>
                
                <fo:block-container width="{$margin-box-width}" height="{$margin-box-height}" margin="0">
                    <xsl:choose>
                        <xsl:when test="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-left-corner')">
                            <xsl:call-template name="generate-properties-for-margin-box">
                                <xsl:with-param name="doc" select="$doc"/>
                                <xsl:with-param name="page-name" select="$page-name"/>
                                <xsl:with-param name="page-side" select="$page-side"/>
                                <xsl:with-param name="page-position" select="$page-position"/>
                                <xsl:with-param name="page-blank" select="$page-blank"/>
                                <xsl:with-param name="margin-box-name" select="'bottom-left-corner'"/>
                                <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                
                                <xsl:with-param name="default-text-align" select="'right'"/>
                                <xsl:with-param name="default-display-align" select="'center'"/>
                            </xsl:call-template>
                            <xsl:call-template name="generate-content-for-page-margin-box">
                                <xsl:with-param name="doc" select="$doc"/>
                                <xsl:with-param name="page-name" select="$page-name"/>
                                <xsl:with-param name="page-side" select="$page-side"/>
                                <xsl:with-param name="page-position" select="$page-position"/>
                                <xsl:with-param name="page-blank" select="$page-blank"/>
                                <xsl:with-param name="margin-box-name" select="'bottom-left-corner'"/>
                                <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                            </xsl:call-template>
                        </xsl:when>
                        <xsl:otherwise>
                            <fo:block/>
                        </xsl:otherwise>
                    </xsl:choose>
                </fo:block-container>
            </fo:static-content>
        </xsl:if>
    </xsl:template>
    
    <!--
        Generates the background for a page.
        The background is a block container that has a background image, color, size, etc..
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').        
        @param page-width The width of the page
        @param page-height The height of the page
        @param page-left-margin The left margin of the page
    -->
    <xsl:template name="generate-page-background">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="page-width"/>
        <xsl:param name="page-height"/>
        <xsl:param name="page-left-margin"/>
        
        <!-- All properties related to the background -->
        <xsl:variable name="page-background-properties" select="oxy:get-page-background-properties($doc, $page-name, $page-side, $page-position, $page-blank)"/>
        <xsl:if test="$page-background-properties">
            <xsl:variable name="left" select="
                if ($page-left-margin = '0') then
                    '0'
                else
                    concat('-', $page-left-margin)"/>
            <fo:block-container height="{$page-height}" left="{$left}" position="absolute" top="0" width="{$page-width}">
                <xsl:for-each select="$page-background-properties">
                    <xsl:attribute name="{@name}" select="@value"/>
                </xsl:for-each>
                
                <!-- In addition, use the FOX extension attributes to pass to the FOP the size of the background and implement image resoltution. -->
                <xsl:call-template name="generate-background-image-size-attributes">
                    <xsl:with-param name="background-size" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'background-size')"/>
                    <xsl:with-param name="parent-width" select="$page-width"/>
                    <xsl:with-param name="parent-height" select="$page-height"/>
                    <xsl:with-param name="extracted-background-image-width" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'extracted-background-image-width')"/>
                    <xsl:with-param name="extracted-background-image-height" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'extracted-background-image-height')"/>
                    <xsl:with-param name="image-resolution" select="oxy:extract-dpi-value(oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'image-resolution'))"/>
                </xsl:call-template>
                
                <!-- In addition, use the the background-position-vertical and background-position-horizontal to pass to the FOP the position of the background. -->
                <xsl:call-template name="generate-background-image-position-attributes">
                    <xsl:with-param name="background-position" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'background-position')"/>
                </xsl:call-template>
                
                <fo:block/>
            </fo:block-container>
        </xsl:if>
    </xsl:template>
    
    <!--
        Generates the background for a page.
        The background is a block container that has a background image, color, size, etc..
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').        
        @param page-width The width of the page
        @param page-height The height of the page
    -->
    
    <xsl:template name="generate-page-margin-box-background">
        <xsl:param name="doc" required="yes"/>
        <xsl:param name="page-name" required="yes"/>
        <xsl:param name="page-side" required="yes"/>
        <xsl:param name="page-position" required="yes"/>
        <xsl:param name="page-blank" required="yes"/>
        <xsl:param name="margin-box-name" required="yes"/>
        <xsl:param name="margin-box-width" required="no"/>
        <xsl:param name="margin-box-height" required="no"/>
        
        <xsl:if test="$margin-box-height and $margin-box-width">
            <!-- All properties related to the background -->
            <xsl:variable name="page-background-properties" select="oxy:get-page-margin-box-background-properties($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name)"/>
            <xsl:if test="$page-background-properties">
                <!-- In addition, use the FOX extension attributes to pass to the FOP the size of the background and implement image resoltution. -->
                <xsl:call-template name="generate-background-image-size-attributes">
                    <xsl:with-param name="background-size" select="oxy:get-property-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'background-size')"/>
                    <xsl:with-param name="parent-width" select="$margin-box-width"/>
                    <xsl:with-param name="parent-height" select="$margin-box-height"/>
                    <xsl:with-param name="extracted-background-image-width" select="oxy:get-property-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'extracted-background-image-width')"/>
                    <xsl:with-param name="extracted-background-image-height" select="oxy:get-property-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'extracted-background-image-height')"/>
                    <xsl:with-param name="image-resolution" select="oxy:extract-dpi-value(oxy:get-property-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'image-resolution'))"/>
                </xsl:call-template>
                
                <!-- In addition, use the the background-position-vertical and background-position-horizontal to pass to the FOP the position of the background. -->
                <xsl:call-template name="generate-background-image-position-attributes">
                    <xsl:with-param name="background-position" select="oxy:get-property-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'background-position')"/>
                </xsl:call-template>
            </xsl:if>
        </xsl:if>
    </xsl:template>
    
    <!-- 
        Subtracts from the current width the width of the borders and paddings. 
        This way we squize the PMB borders and content into the allocated space. 
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').               
        @param margin-box-name The margin box, for instance 'top-left-corner', 'bottom-center', etc...
        
        @return The new width, as an expression.
    -->
    <xsl:function name="oxy:adjust-page-margin-box-width">
        <xsl:param name="width"/>
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="margin-box-name"/>
        
        <xsl:variable name="insets" select="oxy:get-insets-horizontal($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name)"/>
        <xsl:variable name="width" select="
            if ($insets) then
                concat($width, ' - (', $insets, ')')
            else
                $width"/>
        
        <xsl:value-of select="$width"/>
    </xsl:function>
    
    <!-- 
        Subtracts from the current height the height of the borders and paddings. 
        This way we squize the PMB borders and content into the allocated space. 
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').               
        @param margin-box-name The margin box, for instance 'top-left-corner', 'bottom-center', etc...
        
        @return The new height, as an expression.        
    -->
    <xsl:function name="oxy:adjust-page-margin-box-height">
        <xsl:param name="height"/>
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="margin-box-name"/>
        
        <xsl:variable name="insets" select="oxy:get-insets-vertical($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name)"/>
        <xsl:variable name="height" select="
            if ($insets) then
                concat($height, ' - (', $insets, ')')
            else
                $height"/>
        
        <xsl:value-of select="$height"/>
    </xsl:function>
    
    <!-- 
        Generates in the "after" FO region, the static extracted from:
        @bottom-left, @bottom-center, @bottom-right     
    -->
    <xsl:template name="generate-page-statics-bottom">
        <xsl:param name="doc" required="yes"/>
        <xsl:param name="page-name" required="yes"/>
        <xsl:param name="page-side" required="yes"/>
        <xsl:param name="page-position" required="yes"/>
        <xsl:param name="page-blank" required="yes"/>
        
        <xsl:param name="page-left-margin" required="yes"/>
        <xsl:param name="page-bottom-margin" required="yes"/>
        <xsl:param name="page-right-margin" required="yes"/>
        <xsl:param name="page-width" required="yes"/>
        
        <xsl:variable name="fullW" select="concat('(', $page-width, ' - ', $page-left-margin, ' - ', $page-right-margin, ')')"/>
        <xsl:variable name="thirdW" select="concat($fullW, ' div 3')"/>
        <xsl:variable name="halfW" select="concat($fullW, ' div 2')"/>
        
        <xsl:variable name="nrOfBoxes" select="oxy:count-page-margin-boxes-with-content($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-left, bottom-center, bottom-right')"/>
        <xsl:if test="$nrOfBoxes > 0">
            
            <fo:static-content flow-name="{oxy:generate-name('sc-after', $page-side, $page-position, $page-blank)}" role="artifact">
                
                <xsl:variable name="exists-left" select="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-left')"/>
                <xsl:variable name="exists-center" select="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-center')"/>
                <xsl:variable name="exists-right" select="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-right')"/>
                
                <!-- get the width from user if exists -->
                <xsl:variable name="bottom-left-margin-box" select="oxy:get-page-margin-boxes($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-left')"/>
                <xsl:variable name="bottom-center-margin-box" select="oxy:get-page-margin-boxes($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-center')"/>
                <xsl:variable name="bottom-right-margin-box" select="oxy:get-page-margin-boxes($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-right')"/>
                
                <xsl:variable name="external-width-bottom-left">
                    <xsl:if test="$bottom-left-margin-box//property/@name = 'width'">
                        <xsl:value-of select="oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-left', 'width', true())/@value"/>
                    </xsl:if>
                </xsl:variable>
                
                <xsl:variable name="external-width-bottom-center">
                    <xsl:if test="$bottom-center-margin-box//property/@name = 'width'">
                        <xsl:value-of select="oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-center', 'width', true())/@value"/>
                    </xsl:if>
                </xsl:variable>
                
                <xsl:variable name="external-width-bottom-right">
                    <xsl:if test="$bottom-right-margin-box//property/@name = 'width'">
                        <xsl:value-of select="oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-right', 'width', true())/@value"/>
                    </xsl:if>
                </xsl:variable>
                
                <xsl:variable name="check-bottom-left" select="string-length($external-width-bottom-left) > 0" as="xs:boolean"/>
                <xsl:variable name="check-bottom-center" select="string-length($external-width-bottom-center) > 0" as="xs:boolean"/>
                <xsl:variable name="check-bottom-right" select="string-length($external-width-bottom-right) > 0" as="xs:boolean"/>
                
                <xsl:variable name="all-widths-are-set" select="$check-bottom-left and $check-bottom-center and $check-bottom-right" as="xs:boolean"/>
                <xsl:variable name="some-widths-are-set" select="$check-bottom-left or $check-bottom-center or $check-bottom-right" as="xs:boolean"/>
                
                <xsl:if test="$some-widths-are-set and not($all-widths-are-set)">
                    <xsl:message terminate="no">ERROR All 3 widths must be set for BOTTOM at-rules (bottom-right, bottom-center, and bottom-left). Reverting to default behavior.</xsl:message>
                </xsl:if>
                
                <!-- The boxes span an equal width. -->
                <xsl:variable name="boxWidth">
                    <xsl:choose>
                        <xsl:when test="$nrOfBoxes = 1">
                            <xsl:value-of select="$fullW"/>
                        </xsl:when>
                        <xsl:when test="$nrOfBoxes = 2 and not($exists-center)">
                            <xsl:value-of select="$halfW"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="$thirdW"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:variable>
                
                <xsl:variable name="table-layout" select="
                    if ($exists-left and $exists-right and not($exists-center)) then
                        'auto'
                    else
                        'fixed'"/>
                <fo:table border-collapse="separate" table-layout="{$table-layout}" width="100%">
                    <fo:table-body>
                        <fo:table-row>
                            <!-- BOTTOM-LEFT -->
                            <xsl:if test="$exists-left or ($exists-center and $exists-right)">
                                <xsl:comment>BOTTOM-LEFT</xsl:comment>
                                
                                <xsl:variable name="margin-box-width" select="$boxWidth"/>
                                <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($page-bottom-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-left')"/>
                                
                                <fo:table-cell start-indent="0" end-indent="0" height="{$margin-box-height}">
                                    <xsl:choose>
                                        <xsl:when test="$table-layout = 'fixed' and not($all-widths-are-set)">
                                            <xsl:attribute name="width" select="$margin-box-width"/>
                                        </xsl:when>
                                        <xsl:when test="$table-layout = 'fixed' and $all-widths-are-set">
                                            <xsl:attribute name="width" select="oxy:expand-percent($external-width-bottom-left, $fullW)"/>
                                        </xsl:when>
                                    </xsl:choose>
                                    
                                    <xsl:call-template name="generate-properties-for-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'bottom-left'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                        
                                        <xsl:with-param name="default-text-align" select="'left'"/>
                                        <xsl:with-param name="default-display-align" select="'center'"/>
                                    </xsl:call-template>
                                    <xsl:call-template name="generate-content-for-page-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'bottom-left'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                    </xsl:call-template>
                                </fo:table-cell>
                            </xsl:if>
                            
                            <!-- BOTTOM-CENTER -->
                            <xsl:if test="$exists-center">
                                <xsl:comment>BOTTOM-CENTER</xsl:comment>
                                
                                <xsl:variable name="margin-box-width" select="$boxWidth"/>
                                <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($page-bottom-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-center')"/>
                                
                                <fo:table-cell start-indent="0" end-indent="0" height="{$margin-box-height}">
                                    <xsl:choose>
                                        <xsl:when test="$table-layout = 'fixed' and not($all-widths-are-set)">
                                            <xsl:attribute name="width" select="$margin-box-width"/>
                                        </xsl:when>
                                        <xsl:when test="$table-layout = 'fixed' and $all-widths-are-set">
                                            <xsl:attribute name="width" select="oxy:expand-percent($external-width-bottom-center, $fullW)"/>
                                        </xsl:when>
                                    </xsl:choose>
                                    
                                    <xsl:call-template name="generate-properties-for-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'bottom-center'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                        
                                        <xsl:with-param name="default-text-align" select="'center'"/>
                                        <xsl:with-param name="default-display-align" select="'center'"/>
                                    </xsl:call-template>
                                    <xsl:call-template name="generate-content-for-page-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'bottom-center'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                    </xsl:call-template>
                                </fo:table-cell>
                            </xsl:if>
                            
                            <!-- BOTTOM-RIGHT -->
                            <xsl:if test="$exists-right">
                                <xsl:comment>BOTTOM-RIGHT</xsl:comment>
                                <xsl:variable name="margin-box-width" select="$boxWidth"/>
                                <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($page-bottom-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'bottom-right')"/>
                                
                                <fo:table-cell start-indent="0" end-indent="0" height="{$margin-box-height}">
                                    
                                    <xsl:choose>
                                        <xsl:when test="$table-layout = 'fixed' and not($all-widths-are-set)">
                                            <xsl:attribute name="width" select="$margin-box-width"/>
                                        </xsl:when>
                                        <xsl:when test="$table-layout = 'fixed' and $all-widths-are-set">
                                            <xsl:attribute name="width" select="oxy:expand-percent($external-width-bottom-right, $fullW)"/>
                                        </xsl:when>
                                    </xsl:choose>
                                    
                                    <xsl:call-template name="generate-properties-for-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'bottom-right'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                        
                                        <xsl:with-param name="default-text-align" select="'right'"/>
                                        <xsl:with-param name="default-display-align" select="'center'"/>
                                    </xsl:call-template>
                                    <xsl:call-template name="generate-content-for-page-margin-box">
                                        <xsl:with-param name="doc" select="$doc"/>
                                        <xsl:with-param name="page-name" select="$page-name"/>
                                        <xsl:with-param name="page-side" select="$page-side"/>
                                        <xsl:with-param name="page-position" select="$page-position"/>
                                        <xsl:with-param name="page-blank" select="$page-blank"/>
                                        <xsl:with-param name="margin-box-name" select="'bottom-right'"/>
                                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                    </xsl:call-template>
                                </fo:table-cell>
                            </xsl:if>
                            
                        </fo:table-row>
                    </fo:table-body>
                </fo:table>
                
            </fo:static-content>
        </xsl:if>
    </xsl:template>
    
    <!-- 
        Generates in the "before" FO region, the static extracted from:
        @top-left, @top-center, @top-right     
    -->
    <xsl:template name="generate-page-statics-top">
        <xsl:param name="doc" required="yes"/>
        <xsl:param name="page-name" required="yes"/>
        <xsl:param name="page-side" required="yes"/>
        <xsl:param name="page-position" required="yes"/>
        <xsl:param name="page-blank" required="yes"/>
        
        <xsl:param name="page-top-margin" required="yes"/>
        <xsl:param name="page-left-margin" required="yes"/>
        <xsl:param name="page-right-margin" required="yes"/>
        <xsl:param name="page-width" required="yes"/>
        <xsl:param name="page-height" required="yes"/>
        
        <!-- Because of a limitation of FOP, you cannot set background color and image on the entire page (on fo:simple-page-master). 
            As a workaround, we use an absolute positioned block container put in the top left, having the size of the page. -->
        <xsl:variable name="page-background-properties" select="oxy:get-page-background-properties($doc, $page-name, $page-side, $page-position, $page-blank)"/>
        
        <xsl:variable name="fullW" select="concat('(', $page-width, ' - ', $page-left-margin, ' - ', $page-right-margin, ')')"/>
        <xsl:variable name="thirdW" select="concat($fullW, ' div 3')"/>
        <xsl:variable name="halfW" select="concat($fullW, ' div 2')"/>
        
        <xsl:variable name="nrOfBoxes" select="oxy:count-page-margin-boxes-with-content($doc, $page-name, $page-side, $page-position, $page-blank, 'top-left, top-center, top-right')"/>
        <xsl:if test="$nrOfBoxes > 0 or $page-background-properties">
            
            <fo:static-content flow-name="{oxy:generate-name('sc-before', $page-side, $page-position, $page-blank)}" role="artifact">
                
                <!-- Page background image and background color -->
                <xsl:if test="$page-background-properties">
                    <xsl:call-template name="generate-page-background">
                        <xsl:with-param name="doc" select="$doc"/>
                        <xsl:with-param name="page-name" select="$page-name"/>
                        <xsl:with-param name="page-side" select="$page-side"/>
                        <xsl:with-param name="page-position" select="$page-position"/>
                        <xsl:with-param name="page-blank" select="$page-blank"/>
                        <xsl:with-param name="page-width" select="$page-width"/>
                        <xsl:with-param name="page-height" select="$page-height"/>
                        <xsl:with-param name="page-left-margin" select="$page-left-margin"/>
                    </xsl:call-template>
                </xsl:if>
                
                <xsl:if test="$nrOfBoxes > 0">
                    
                    <xsl:variable name="exists-left" select="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'top-left')"/>
                    <xsl:variable name="exists-center" select="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'top-center')"/>
                    <xsl:variable name="exists-right" select="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, 'top-right')"/>
                    
                    <!-- get the width from user if exists -->
                    <xsl:variable name="top-left-margin-box" select="oxy:get-page-margin-boxes($doc, $page-name, $page-side, $page-position, $page-blank, 'top-left')"/>
                    <xsl:variable name="top-center-margin-box" select="oxy:get-page-margin-boxes($doc, $page-name, $page-side, $page-position, $page-blank, 'top-center')"/>
                    <xsl:variable name="top-right-margin-box" select="oxy:get-page-margin-boxes($doc, $page-name, $page-side, $page-position, $page-blank, 'top-right')"/>
                    
                    <xsl:variable name="external-width-top-left">
                        <xsl:if test="$top-left-margin-box//property/@name = 'width'">
                            <xsl:value-of select="oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, 'top-left', 'width', true())/@value"/>
                        </xsl:if>
                    </xsl:variable>
                    
                    <xsl:variable name="external-width-top-center">
                        <xsl:if test="$top-center-margin-box//property/@name = 'width'">
                            <xsl:value-of select="oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, 'top-center', 'width', true())/@value"/>
                        </xsl:if>
                    </xsl:variable>
                    
                    <xsl:variable name="external-width-top-right">
                        <xsl:if test="$top-right-margin-box//property/@name = 'width'">
                            <xsl:value-of select="oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, 'top-right', 'width', true())/@value"/>
                        </xsl:if>
                    </xsl:variable>
                    
                    <xsl:variable name="check-top-left" select="string-length($external-width-top-left) > 0" as="xs:boolean"/>
                    <xsl:variable name="check-top-center" select="string-length($external-width-top-center) > 0" as="xs:boolean"/>
                    <xsl:variable name="check-top-right" select="string-length($external-width-top-right) > 0" as="xs:boolean"/>
                    
                    <xsl:variable name="all-widths-are-set" select="$check-top-left and $check-top-center and $check-top-right" as="xs:boolean"/>
                    <xsl:variable name="some-widths-are-set" select="$check-top-left or $check-top-center or $check-top-right" as="xs:boolean"/>
                    
                    <xsl:if test="$some-widths-are-set and not($all-widths-are-set)">
                        <xsl:message terminate="no">ERROR All 3 widths must be set for TOP at-rules (top-right, top-center, and top-left). Reverting to default behavior.</xsl:message>
                    </xsl:if>
                    
                    <!-- The boxes span an equal width. -->
                    <xsl:variable name="boxWidth">
                        <xsl:choose>
                            <xsl:when test="$nrOfBoxes = 1">
                                <xsl:value-of select="$fullW"/>
                            </xsl:when>
                            <xsl:when test="$nrOfBoxes = 2 and not($exists-center)">
                                <xsl:value-of select="$halfW"/>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:value-of select="$thirdW"/>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:variable>
                    
                    <xsl:variable name="table-layout" select="
                        if ($exists-left and $exists-right and not($exists-center)) then
                            'auto'
                        else
                            'fixed'"/>
                    <fo:table border-collapse="separate" table-layout="{$table-layout}" width="100%">
                        <fo:table-body>
                            <fo:table-row>
                                <!-- TOP-LEFT -->
                                <xsl:if test="$exists-left or ($exists-center and $exists-right)">
                                    <xsl:comment>TOP-LEFT</xsl:comment>
                                    <xsl:variable name="margin-box-width" select="$boxWidth"/>
                                    <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($page-top-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'top-left')"/>
                                    
                                    <fo:table-cell start-indent="0" end-indent="0" height="{$margin-box-height}">
                                        <xsl:choose>
                                            <xsl:when test="$table-layout = 'fixed' and not($all-widths-are-set)">
                                                <xsl:attribute name="width" select="$margin-box-width"/>
                                            </xsl:when>
                                            <xsl:when test="$table-layout = 'fixed' and $all-widths-are-set">
                                                <xsl:attribute name="width" select="oxy:expand-percent($external-width-top-left, $fullW)"/>
                                            </xsl:when>
                                        </xsl:choose>
                                        
                                        <xsl:call-template name="generate-properties-for-margin-box">
                                            <xsl:with-param name="doc" select="$doc"/>
                                            <xsl:with-param name="page-name" select="$page-name"/>
                                            <xsl:with-param name="page-side" select="$page-side"/>
                                            <xsl:with-param name="page-position" select="$page-position"/>
                                            <xsl:with-param name="page-blank" select="$page-blank"/>
                                            <xsl:with-param name="margin-box-name" select="'top-left'"/>
                                            <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                            <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                            
                                            <xsl:with-param name="default-text-align" select="'left'"/>
                                            <xsl:with-param name="default-display-align" select="'center'"/>
                                        </xsl:call-template>
                                        <xsl:call-template name="generate-content-for-page-margin-box">
                                            <xsl:with-param name="doc" select="$doc"/>
                                            <xsl:with-param name="page-name" select="$page-name"/>
                                            <xsl:with-param name="page-side" select="$page-side"/>
                                            <xsl:with-param name="page-position" select="$page-position"/>
                                            <xsl:with-param name="page-blank" select="$page-blank"/>
                                            <xsl:with-param name="margin-box-name" select="'top-left'"/>
                                            <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                            <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                        </xsl:call-template>
                                    </fo:table-cell>
                                </xsl:if>
                                
                                <!-- TOP-CENTER -->
                                <xsl:if test="$exists-center">
                                    <xsl:comment>TOP-CENTER</xsl:comment>
                                    <xsl:variable name="margin-box-width" select="$boxWidth"/>
                                    <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($page-top-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'top-center')"/>
                                    
                                    <fo:table-cell start-indent="0" end-indent="0" height="{$margin-box-height}">
                                        <xsl:choose>
                                            <xsl:when test="$table-layout = 'fixed' and not($all-widths-are-set)">
                                                <xsl:attribute name="width" select="$margin-box-width"/>
                                            </xsl:when>
                                            <xsl:when test="$table-layout = 'fixed' and $all-widths-are-set">
                                                <xsl:attribute name="width" select="oxy:expand-percent($external-width-top-center, $fullW)"/>
                                            </xsl:when>
                                        </xsl:choose>
                                        
                                        <xsl:call-template name="generate-properties-for-margin-box">
                                            <xsl:with-param name="doc" select="$doc"/>
                                            <xsl:with-param name="page-name" select="$page-name"/>
                                            <xsl:with-param name="page-side" select="$page-side"/>
                                            <xsl:with-param name="page-position" select="$page-position"/>
                                            <xsl:with-param name="page-blank" select="$page-blank"/>
                                            <xsl:with-param name="margin-box-name" select="'top-center'"/>
                                            <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                            <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                            
                                            <xsl:with-param name="default-text-align" select="'center'"/>
                                            <xsl:with-param name="default-display-align" select="'center'"/>
                                        </xsl:call-template>
                                        <xsl:call-template name="generate-content-for-page-margin-box">
                                            <xsl:with-param name="doc" select="$doc"/>
                                            <xsl:with-param name="page-name" select="$page-name"/>
                                            <xsl:with-param name="page-side" select="$page-side"/>
                                            <xsl:with-param name="page-position" select="$page-position"/>
                                            <xsl:with-param name="page-blank" select="$page-blank"/>
                                            <xsl:with-param name="margin-box-name" select="'top-center'"/>
                                            <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                            <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                        </xsl:call-template>
                                    </fo:table-cell>
                                </xsl:if>
                                
                                <!-- TOP-RIGHT -->
                                <xsl:if test="$exists-right">
                                    <xsl:comment>TOP-RIGHT</xsl:comment>
                                    <xsl:variable name="margin-box-width" select="$boxWidth"/>
                                    <xsl:variable name="margin-box-height" select="oxy:adjust-page-margin-box-height($page-top-margin, $doc, $page-name, $page-side, $page-position, $page-blank, 'top-right')"/>
                                    
                                    <fo:table-cell start-indent="0" end-indent="0" height="{$margin-box-height}">
                                        <xsl:choose>
                                            <xsl:when test="$table-layout = 'fixed' and not($all-widths-are-set)">
                                                <xsl:attribute name="width" select="$margin-box-width"/>
                                            </xsl:when>
                                            <xsl:when test="$table-layout = 'fixed' and $all-widths-are-set">
                                                <xsl:attribute name="width" select="oxy:expand-percent($external-width-top-right, $fullW)"/>
                                            </xsl:when>
                                        </xsl:choose>
                                        
                                        <xsl:call-template name="generate-properties-for-margin-box">
                                            <xsl:with-param name="doc" select="$doc"/>
                                            <xsl:with-param name="page-name" select="$page-name"/>
                                            <xsl:with-param name="page-side" select="$page-side"/>
                                            <xsl:with-param name="page-position" select="$page-position"/>
                                            <xsl:with-param name="page-blank" select="$page-blank"/>
                                            <xsl:with-param name="margin-box-name" select="'top-right'"/>
                                            <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                            <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                            
                                            <xsl:with-param name="default-text-align" select="'right'"/>
                                            <xsl:with-param name="default-display-align" select="'center'"/>
                                        </xsl:call-template>
                                        <xsl:call-template name="generate-content-for-page-margin-box">
                                            <xsl:with-param name="doc" select="$doc"/>
                                            <xsl:with-param name="page-name" select="$page-name"/>
                                            <xsl:with-param name="page-side" select="$page-side"/>
                                            <xsl:with-param name="page-position" select="$page-position"/>
                                            <xsl:with-param name="page-blank" select="$page-blank"/>
                                            <xsl:with-param name="margin-box-name" select="'top-right'"/>
                                            <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                                            <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                                        </xsl:call-template>
                                        
                                    </fo:table-cell>
                                </xsl:if>
                            </fo:table-row>
                        </fo:table-body>
                    </fo:table>
                </xsl:if>
                
            </fo:static-content>
        </xsl:if>
    </xsl:template>
    
    <!-- 
        Generates in the "footnote" FO region, with the border top extracted from
        the css:footnote element. 
    -->
    <xsl:template name="generate-page-statics-footnote">
        <xsl:param name="doc" required="yes"/>
        <xsl:param name="page-name" required="yes"/>
        <xsl:param name="page-side" required="yes"/>
        <xsl:param name="page-position" required="yes"/>
        <xsl:param name="page-blank" required="yes"/>
        
        <fo:static-content flow-name="xsl-footnote-separator" role="artifact">
            <xsl:call-template name="generate-properties-for-margin-box">
                <xsl:with-param name="doc" select="$doc"/>
                <xsl:with-param name="page-name" select="$page-name"/>
                <xsl:with-param name="page-side" select="$page-side"/>
                <xsl:with-param name="page-position" select="$page-position"/>
                <xsl:with-param name="page-blank" select="$page-blank"/>
                <xsl:with-param name="margin-box-name" select="'footnote'"/>
                
                <xsl:with-param name="default-text-align" select="'left'"/>
                <xsl:with-param name="default-display-align" select="'auto'"/>
            </xsl:call-template>
            
            <xsl:call-template name="generate-content-for-page-margin-box">
                <xsl:with-param name="doc" select="$doc"/>
                <xsl:with-param name="page-name" select="$page-name"/>
                <xsl:with-param name="page-side" select="$page-side"/>
                <xsl:with-param name="page-position" select="$page-position"/>
                <xsl:with-param name="page-blank" select="$page-blank"/>
                <xsl:with-param name="margin-box-name" select="'footnote'"/>
                <xsl:with-param name="margin-box-width" select="'none'"/>
                <xsl:with-param name="margin-box-height" select="'none'"/>
            </xsl:call-template>
        </fo:static-content>
    </xsl:template>
    
    <!-- 
        Generates the contents of a page margin box. 
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page. 
        @param page-side The side of the page. Can be 'left', 'right', 'any'.
        @param page-position The position of the page in the sequence. Can be 'first' or 'any', etc.
        @param page-black The page blank. Can be 'blank' or 'any-blank-or-not', etc.
        @param margin-box-name The name of the margin box.
    -->
    <xsl:template name="generate-content-for-page-margin-box">
        <xsl:param name="doc" required="yes"/>
        <xsl:param name="page-name" required="yes"/>
        <xsl:param name="page-side" required="yes"/>
        <xsl:param name="page-position" required="yes"/>
        <xsl:param name="page-blank" required="yes"/>
        <xsl:param name="margin-box-name" required="yes"/>
        <xsl:param name="margin-box-width" required="yes"/>
        <xsl:param name="margin-box-height" required="yes"/>
        
        <!-- Box background image and background color, as a block container with position absolute. -->
        
        <xsl:variable name="link" select="oxy:get-property-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'link')"/>
        
        <xsl:choose>
            <xsl:when test="$link">
                <fo:block>
                    <fo:basic-link>
                        <xsl:call-template name="generate-link-attributes-generic">
                            <xsl:with-param name="doc" select="$doc"/>
                            <xsl:with-param name="link-attr-val" select="$link"/>
                        </xsl:call-template>
                        
                        <xsl:call-template name="generate-content-for-page-margin-box-basic">
                            <xsl:with-param name="doc" select="$doc"/>
                            <xsl:with-param name="page-name" select="$page-name"/>
                            <xsl:with-param name="page-side" select="$page-side"/>
                            <xsl:with-param name="page-position" select="$page-position"/>
                            <xsl:with-param name="page-blank" select="$page-blank"/>
                            <xsl:with-param name="margin-box-name" select="$margin-box-name"/>
                            <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                            <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                        </xsl:call-template>
                    </fo:basic-link>
                </fo:block>
            </xsl:when>
            <xsl:otherwise>
                <fo:block>
                    <xsl:call-template name="generate-content-for-page-margin-box-basic">
                        <xsl:with-param name="doc" select="$doc"/>
                        <xsl:with-param name="page-name" select="$page-name"/>
                        <xsl:with-param name="page-side" select="$page-side"/>
                        <xsl:with-param name="page-position" select="$page-position"/>
                        <xsl:with-param name="page-blank" select="$page-blank"/>
                        <xsl:with-param name="margin-box-name" select="$margin-box-name"/>
                        <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
                        <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
                    </xsl:call-template>
                </fo:block>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <xsl:template name="generate-content-for-page-margin-box-basic">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="margin-box-name"/>
        <xsl:param name="margin-box-width"/>
        <xsl:param name="margin-box-height"/>
        <xsl:variable name="static" select="oxy:get-static-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name)"/>
        
        <xsl:variable name="exists-leader">
            <xsl:for-each select="$static">
                <xsl:if test=".//*:leader | self::*:leader">true</xsl:if>
            </xsl:for-each>
        </xsl:variable>
        
        <xsl:if test="string-length(normalize-space($exists-leader)) > 0">
            <!-- Leaders need the parent block to have text justified, this way they can span the available width.-->
            <xsl:attribute name="text-align-last" select="'justify'"/>
        </xsl:if>
        
        <xsl:apply-templates select="$static">
            <!-- Static images need to access the image-resolution property, that is why we tunnel the parameters.-->
            <xsl:with-param name="doc" select="$doc" tunnel="yes"/>
            <xsl:with-param name="page-name" select="$page-name" tunnel="yes"/>
            <xsl:with-param name="page-side" select="$page-side" tunnel="yes"/>
            <xsl:with-param name="page-position" select="$page-position" tunnel="yes"/>
            <xsl:with-param name="page-blank" select="$page-blank" tunnel="yes"/>
            <xsl:with-param name="margin-box-name" select="$margin-box-name" tunnel="yes"/>
            <xsl:with-param name="margin-box-width" select="$margin-box-width" tunnel="yes"/>
            <xsl:with-param name="margin-box-height" select="$margin-box-height" tunnel="yes"/>
        </xsl:apply-templates>
    </xsl:template>
    
    <!-- 
        Gets the a CSS property for a page, providing a default value. The page is defined by the triple (name, side, part).
        For instance one can use this method to get the background color of the named page "intro", the first one from the left:
        
        get-property-for-page('intro', 'first', 'left', 'background-color', 'white');
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page. 
        @param page-side The side of the page. Can be 'left', 'right', 'any'.
        @param page-position The position of the page in the sequence. Can be 'first' or 'any', etc.
        @param page-black The page blank. Can be 'blank' or 'any-blank-or-not', etc.
        @param default-value The default value to be used.
        
        @return the value as string      
    -->
    <xsl:function name="oxy:get-property-for-page" as="xs:string">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="property"/>
        <xsl:param name="default-value"/>
        
        <xsl:variable name="value" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, $property)"/>
        <xsl:value-of select="
            if (string-length($value) > 0) then
                $value
            else
                $default-value"/>
    </xsl:function>
    
    <!-- 
        Gets the a CSS property for a page. The page is defined by the triple (name, side, part).
        For instance one can use this method to get the background color of the named page "intro", the first one from the left:
        
        get-property-for-page('intro', 'first', 'left', 'background-color');
        
        It tries to find the most specifc property definition, starting with the most specific page definitions.
        
        @param doc The context document. 
        @param page-name The name of the page.
        @param page-side The side of the page. Can be 'left', 'right', 'any'.
        @param page-position The position of the page in the sequence. Can be 'first' or 'any', etc.
        @param page-blanck The page blank. Can be 'blank' or 'any', etc.
        @return the value as string  
    -->
    <xsl:function name="oxy:get-property-for-page" as="xs:string">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="property"/>
        
        <xsl:value-of select="oxy:get-property-element-for-page($doc, $page-name, $page-side, $page-position, $page-blank, $property)/@value"/>
    </xsl:function>
    
    <!-- 
        Gets the a CSS property element for a page. The page is defined by the triple (name, side, part).
        For instance one can use this method to get the background color of the named page "intro", the first one from the left:
        
        get-property-element-for-page('intro', 'first', 'left', 'background-color');
        
        It tries to find the most specifc property definition, starting with the most specific page definitions.
        
        @param doc The context document. 
        @param page-name The name of the page.
        @param page-side The side of the page. Can be 'left', 'right', 'any'.
        @param page-position The position of the page in the sequence. Can be 'first' or 'any', etc.
        @param page-blanck The page blank. Can be 'blank' or 'any', etc.
        @return the value as string  
    -->
    <xsl:function name="oxy:get-original-property-element-for-page" as="item()*">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="property"/>
        
        <xsl:variable name="matching-pages" select="oxy:get-matching-pages($doc, $page-name, $page-side, $page-position, $page-blank)"/>
        
        <!-- 
            Cannot use $matching-pages/property... because it 
            would result in nodes sorted in document order, not in the matching 
            sequence order.
        -->
        <xsl:copy-of select="oxy:get-property-from-list($property, $matching-pages)"/>
    </xsl:function>
    
    <!-- 
        Gets the a CSS property element for a page. The page is defined by the triple (name, side, part).
        For instance one can use this method to get the background color of the named page "intro", the first one from the left:
        
        get-property-element-for-page('intro', 'first', 'left', 'background-color');
        
        It tries to find the most specifc property definition, starting with the most specific page definitions.
        
        @param doc The context document. 
        @param page-name The name of the page.
        @param page-side The side of the page. Can be 'left', 'right', 'any'.
        @param page-position The position of the page in the sequence. Can be 'first' or 'any', etc.
        @param page-blanck The page blank. Can be 'blank' or 'any', etc.
        @return the value as string  
    -->
    <xsl:function name="oxy:get-property-element-for-page" as="item()*">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="property"/>
        
        <xsl:variable name="matching-pages" select="oxy:get-matching-pages($doc, $page-name, $page-side, $page-position, $page-blank)"/>
        <!-- 
            Cannot use $matching-pages/property... because it 
            would result in nodes sorted in document order, not in the matching 
            sequence order.
        -->
        <xsl:variable name="elem" select="oxy:get-property-from-list($property, $matching-pages)"/>
        
        <!-- Switches the width and height depending on the page orientation, imposes other defaults. -->
        <xsl:variable name="elem" as="item()*">
            <xsl:choose>
                
                <xsl:when test="starts-with($property, 'margin')">
                    <xsl:choose>
                        <xsl:when test="$elem">
                            <xsl:copy-of select="$elem"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <css:property name="{$property}" value="{$const-def-margin}"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:when>
                
                <xsl:when test="$property = 'width'">
                    <xsl:choose>
                        <xsl:when test="'landscape' = oxy:get-original-property-element-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'orientation')/@value">
                            <xsl:variable name="elem" select="oxy:get-original-property-element-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'height')"/>
                            <xsl:choose>
                                <xsl:when test="$elem">
                                    <xsl:copy-of select="$elem"/>
                                </xsl:when>
                                <xsl:otherwise>
                                    <css:property name="width" value="{$const-def-page-height}"/>
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:choose>
                                <xsl:when test="$elem">
                                    <xsl:copy-of select="$elem"/>
                                </xsl:when>
                                <xsl:otherwise>
                                    <css:property name="width" value="{$const-def-page-width}"/>
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:when>
                <xsl:when test="$property = 'height'">
                    <xsl:choose>
                        <xsl:when test="'landscape' = oxy:get-original-property-element-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'orientation')/@value">
                            <xsl:variable name="elem" select="oxy:get-original-property-element-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'width')"/>
                            <xsl:choose>
                                <xsl:when test="$elem">
                                    <xsl:copy-of select="$elem"/>
                                </xsl:when>
                                <xsl:otherwise>
                                    <css:property name="height" value="{$const-def-page-width}"/>
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:choose>
                                <xsl:when test="$elem">
                                    <xsl:copy-of select="$elem"/>
                                </xsl:when>
                                <xsl:otherwise>
                                    <css:property name="height" value="{$const-def-page-height}"/>
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:copy-of select="$elem"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:copy-of select="$elem"/>
    </xsl:function>
    
    <xsl:template match="/">
        <fo:root>
            <xsl:variable name="xml-lang" select="oxy:get-xml-lang-from-root(/)"/>
            
            <xsl:if test="$xml-lang">
                <xsl:attribute name="xml:lang" select="$xml-lang"/>
            </xsl:if>
            
            <xsl:variable name="writing-mode">
                <xsl:variable name="lang" select="
                    if (contains($xml-lang, '-')) then
                        substring-before($xml-lang, '-')
                    else
                        $xml-lang"/>
                <xsl:choose>
                    <xsl:when test="
                        some $l in ('ar', 'fa', 'he', 'ps', 'ur')
                        satisfies $l eq $lang">rl</xsl:when>
                    <xsl:otherwise>lr</xsl:otherwise>
                </xsl:choose>
            </xsl:variable>
            
            <xsl:if test="$writing-mode">
                <xsl:attribute name="writing-mode" select="$writing-mode"/>
            </xsl:if>
            
            <!-- This section creates the skeleton of the FO regions. 
                The static content is added later, by each of the page-sequences. -->
            <fo:layout-master-set>
                <xsl:choose>
                    <xsl:when test="/root/pages/page">
                        <xsl:call-template name="generate-page-masters"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <!-- No page definition. Since XSL FO requires at least 
                            one page master defintion, we add one, named 'css2fo-default' -->
                        <fo:simple-page-master master-name="css2fo-default" page-width="{$const-def-page-width}" page-height="{$const-def-page-height}">
                            <fo:region-body margin="{$const-def-margin}"/>
                        </fo:simple-page-master>
                    </xsl:otherwise>
                </xsl:choose>
            </fo:layout-master-set>
            
            <!-- The PDF Metainformation -->
            <xsl:call-template name="generate-metadata"/>
            
            <!-- The PDF Bookmarks -->
            <xsl:call-template name="generate-bookmark-tree"/>
            
            <!-- A page sequence for all 'content' elements -->
            <xsl:for-each select="/root/page-sequence">
                
                <xsl:variable name="page-name" select="oxy:get-page-sequence-name(.)"/>
                <xsl:variable name="next-page-name" select="oxy:get-page-sequence-name(following-sibling::page-sequence[1])"/>
                
                <!-- These control the double side pagination -->
                <xsl:variable name="pagedef-force-page-count" select="oxy:get-property-for-page(/, $page-name, 'any', 'any', 'any', 'oxy-force-page-count')"/>
                <xsl:variable name="pagedef-initial-page-number" select="oxy:get-property-for-page(/, $page-name, 'any', 'any', 'any', 'oxy-initial-page-number')"/>
                <xsl:variable name="pagedef-next-initial-page-number" select="oxy:get-property-for-page(/, $next-page-name, 'any', 'any', 'any', 'oxy-initial-page-number')"/>
                
                <fo:page-sequence master-reference="{$page-name}">
                    <!-- Attributes -->
                    <xsl:choose>
                        
                        <xsl:when test="@chfox:rest-of-sequence = 'rest'">
                            <!-- This will go into fop and will inhibit the :first page selector -->
                            <xsl:attribute name="chfox:rest-of-sequence" select="'rest'"/>
                        </xsl:when>
                        
                        <xsl:otherwise>
                            <!-- Force page count.-->
                            <xsl:choose>
                                <xsl:when test="$pagedef-next-initial-page-number">
                                    <!-- The next sequence has a required placement, make 
                                        the current page-sequence stretch by adding blank pages if necessary. -->
                                    <xsl:attribute name="force-page-count" select="'auto'"/>
                                </xsl:when>
                                <xsl:when test="$pagedef-force-page-count">
                                    <!-- The current sequence has a required count. -->
                                    <xsl:attribute name="force-page-count" select="$pagedef-force-page-count"/>
                                </xsl:when>
                                <xsl:otherwise>
                                    <!-- Use no-force to behave as in CSS, where a counter-reset to a bigger value on a page
                                        sequence does not create blank pages before. -->
                                    <xsl:attribute name="force-page-count" select="'no-force'"/>
                                </xsl:otherwise>
                            </xsl:choose>
                            
                            <!-- Initial page number -->
                            <xsl:choose>
                                <!-- It may come from counter-reset set on the DOM elements. -->
                                <xsl:when test="@initial-page-number">
                                    <xsl:attribute name="initial-page-number" select="@initial-page-number"/>
                                </xsl:when>
                                <!-- It may come from  -oxy-initial-page-number set on the page definition. -->
                                <xsl:when test="$pagedef-initial-page-number">
                                    <xsl:attribute name="initial-page-number" select="$pagedef-initial-page-number"/>
                                </xsl:when>
                                <xsl:otherwise>
                                    <!-- Or from a page definition, as a counter reset. it may have numeric values. -->
                                    <xsl:variable name="page-counter-reset" select="oxy:get-counter-reset-from-page-def(/, $page-name, 'page')"/>
                                    <xsl:if test="$page-counter-reset >= 0">
                                        <xsl:attribute name="initial-page-number" select="$page-counter-reset"/>
                                    </xsl:if>
                                </xsl:otherwise>
                            </xsl:choose>
                            
                        </xsl:otherwise>
                    </xsl:choose>
                    
                    <!-- Format of the page number -->
                    <xsl:variable name="page-number-format" select="oxy:get-page-number-format(/, $page-name)"/>
                    <xsl:if test="not($page-number-format = '1')">
                        <xsl:attribute name="format" select="$page-number-format"/>
                    </xsl:if>
                    
                    <!-- We need this for counter(pages). It uses a page-number-citation-last pointing to this sequence. -->
                    <xsl:if test="position() = last()">
                        <xsl:attribute name="id" select="'last-page-sequence'"/>
                    </xsl:if>
                    
                    <!-- We need this for the footnote and other superscripts/subscripts to keep them from increasing the line height. -->
                    <xsl:attribute name="line-height-shift-adjustment" select="'disregard-shifts'"/>
                    
                    <!-- Content -->
                    
                    <!-- Defines the static content for the page, pushing it in the 
                        regions defined in the page masters -->
                    <xsl:call-template name="generate-page-static">
                        <xsl:with-param name="doc" select="/"/>
                        <xsl:with-param name="page-name" select="$page-name"/>
                    </xsl:call-template>
                    
                    <fo:flow flow-name="xsl-region-body">
                        <xsl:choose>
                            <xsl:when test="*[@css:display = 'block']">
                                <xsl:apply-templates/>
                            </xsl:when>
                            <xsl:otherwise>
                                <!-- We need at least a fo: element in the fo:flow, we cannot copy the text directly here. -->
                                <fo:block>
                                    <xsl:apply-templates/>
                                </fo:block>
                            </xsl:otherwise>
                        </xsl:choose>
                    </fo:flow>
                    
                </fo:page-sequence>
            </xsl:for-each>
        </fo:root>
    </xsl:template>
    
    <!-- 
        Gets the name of the page sequence. 
        
        @param page-sequence-element The page sequence element from the input document.
    -->
    <xsl:function name="oxy:get-page-sequence-name">
        <xsl:param name="page-sequence-element"/>
        <xsl:value-of select="
            if ($page-sequence-element/@page) then
                $page-sequence-element/@page
            else
                'css2fo-default'"/>
    </xsl:function>
    
    <!--
        Gets the format for the page number counters used in a page sequence. Unlike the CSS 'page' counter that may be 
        formatted differently in each of its uses, the XSL-FO restricts to one format in a page sequence.
        
        @return The first @format attribute used on a counter from the matching pages static content.
    -->
    <xsl:function name="oxy:get-page-number-format" as="xs:string">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        
        <xsl:variable name="css-format" select="
            (for $f in
            ($doc/css:root/css:pages/css:page[not(@name)]/*/css:property[@name = 'content']/css:counter[@name = 'page']/@format |
            $doc/css:root/css:pages/css:page[@name = $page-name]/*/css:property[@name = 'content']/css:counter[@name = 'page']/@format)
            return
                $f)[last()]"/>
        
        <xsl:choose>
            <xsl:when test="$css-format">
                <xsl:variable name="fo-format" select="oxy:get-number-format($css-format)"/>
                <xsl:value-of select="normalize-space($fo-format)"/>
            </xsl:when>
            <xsl:otherwise>
                <!-- decimal -->
                <xsl:value-of select="'1'"/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:function>
    
    <!--
        Copies all properties of the page or page margin box that 
        are equal to the specified names, as attributes in 
        the FO result.
        
        @param context  If missing, the dot should be either a page or a page margin box 
        element. (page, top-left-corner, etc...) 
        
        @param props    The list of names.
        
        @param as       The names of the FO attributes that will receive the extracted properties. 
        If this is not specified, use the same name as the CSS properties        
    -->
    <xsl:template name="copy-properties-as">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="props"/>
        <xsl:param name="as" select="$props"/>
        
        <xsl:variable name="foAttributesNames" select="tokenize($as, ', ')"/>
        
        <xsl:for-each select="tokenize($props, ', ')">
            <xsl:variable name="propertyName" select="."/>
            <xsl:variable name="propertyIndex" select="position()"/>
            
            <xsl:variable name="value" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, $propertyName)"/>
            <xsl:if test="string-length($value) > 0">
                <xsl:attribute name="{$foAttributesNames[position() = $propertyIndex]}" select="$value"/>
            </xsl:if>
        </xsl:for-each>
    </xsl:template>
    
    <!--
        Copies all properties of the page or page margin box that 
        start with one of the specified prefixes, as attributes in 
        the FO result. If the property is not found in the most specific 
        page (that matches all criteria), the property is then searched in other pages that match, but are more generic. 
        
        @param doc  The document. 
        @param page-name The name of the page. Can be empty or 'css2fo-default' for the default page.
        @param page-side Can be 'left' or 'right'   
        @param page-position Can be 'first' or 'any'   
        @param props The list of property name prefixes.
    -->
    <xsl:template name="copy-properties-by-prefix">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="props"/>
        
        <xsl:for-each select="tokenize($props, ', ')">
            <xsl:variable name="propertyPrefix" select="."/>
            
            <!-- We try all the properties defined in the entire document since it is har -->
            <xsl:for-each select="distinct-values($doc//property[starts-with(@name, $propertyPrefix)]/@name)">
                <xsl:variable name="name" select="string(.)"/>
                <xsl:variable name="value" select="oxy:get-property-for-page($doc, $page-name, $page-side, $page-position, $page-blank, $name)"/>
                <xsl:if test="string-length($value) > 0">
                    <xsl:attribute name="{$name}" select="$value"/>
                </xsl:if>
                
            </xsl:for-each>
        </xsl:for-each>
    </xsl:template>
    
    <!--
        Gets all the page margin boxes, such as "top-left-corner", "left-middle", etc.. that 
        match a page name. The default page margin boxes are placed at the beginning of the list.
        
        Let's take for example:
        
        @page {
            @top-righ-corner { (1)
                color: blue;
            }
        }
        
        @page {
            @top-righ-corner { (2)
                color: yellow;
            }
        }
        
        @page intro {
            @top-righ-corner { (3)
                color: red;
            }
        }
        
        @page intro:first { (4)
            @top-righ-corner {
                color: green;
            }
        }
        
        When searching for the "top-right-corner" for the page "intro', this function will 
        return a sequence of nodes corresponding to: ( 1, 2, 3). When using these nodes
        to search for a property, the last property with a specific name is the most important.
        
        Important: In FO is not possible to change the page margins static content depending 
        on the :left, :right, :first selectors. This is why we ignore all "top-left-corner", 
        "left-middle", etc defined by a pseudo page. In our example the (4) definition.
        
        @param doc The document with the page definitions 
        @param page-name The name of the page. The default page is always matched. 
        @param margin-box-name The name of the margin box. Can be "top-right-corner" for instance.
        
        @return a sequence of nodes. All the nodes can contribute properties for a margin box, the most important last.
    -->
    <xsl:function name="oxy:get-page-margin-boxes" as="item()*">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        
        <xsl:param name="margin-box-name"/>
        
        <xsl:variable name="matching-pages" select="oxy:get-matching-pages($doc, $page-name, $page-side, $page-position, $page-blank)"/>
        
        <!-- 
            Cannot use $matching-pages//*[local-name() = $margin-box-name] because it 
            would result in nodes sorted in document order, not in the matching 
            sequence order.
        -->
        <xsl:variable name="ret" select="
            for $pg in $matching-pages
            return
                $pg//*[local-name() = $margin-box-name]"/>
        
        <xsl:sequence select="$ret"/>
    </xsl:function>
    
    <!-- 
        Checks if a page margin box exists for a specified page. It takes into 
        account the margin boxes defined in the default (not named) pages.
        
        @param doc The document with the page definitions 
        @param page-name The name of the page. The default page is always matched. 
        @param margin-box-name The name of the margin box. Can be "top-right-corner" for instance.
        
        @return true() if there is a margin box for that page, even if inherited from the default page, or false()
    -->
    <xsl:function name="oxy:exist-page-margin-box" as="xs:boolean">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        
        <xsl:param name="margin-box-name"/>
        
        <xsl:variable name="ret" select="
            if (count(
            oxy:get-page-margin-boxes(
            $doc,
            $page-name,
            $page-side,
            $page-position,
            $page-blank, $margin-box-name)) > 0) then
                true()
            else
                false()" as="xs:boolean"/>
        
        <xsl:value-of select="$ret"/>
    </xsl:function>
    
    <!-- 
        Checks if a page margin box generates content for a specified page. 
        
        @param doc The document with the page definitions 
        @param page-name The name of the page. The default page is always matched. 
        @param margin-box-name The name of the margin box. Can be "top-right-corner" for instance.
        
        @return true() if there is a margin box for that page, even if inherited from the default page, or false()
    -->
    <xsl:function name="oxy:exist-page-margin-box-content" as="xs:boolean">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        
        <xsl:param name="margin-box-name"/>
        
        <xsl:variable name="ret" select="
            if (count(
            (for $pmb
            in oxy:get-page-margin-boxes(
            $doc,
            $page-name,
            $page-side,
            $page-position,
            $page-blank, $margin-box-name)
            return
                $pmb/property[@name = 'content']
            )[last()][count(*) > 0]) > 0) then
                true()
            else
                false()" as="xs:boolean"/>
        
        <xsl:value-of select="$ret"/>
    </xsl:function>
    
    <!-- 
        Computes how many page margin boxes are available for the specified page.
        
        For example:
        
        @page {
            @top-left {}
            @top-right {}
        }
        @page intro{
            @top-center {}
            @top-right {}
        }
        
        When calling oxy:count-page-margin-boxes(intro, 'top-left, top-center, top-right'), it will return 3, since
        the page intro defines a @top-center and a @top-right, and inherits a @top-left.
        
        When calling oxy:count-page-margin-boxes(intro, 'top-right'), it will return 1, since
        the page intro defines a @top-right that overrides the one from the base.
        
        @param doc The document with the page definitions 
        @param page-name The name of the page. The default page is always matched. 
        @param margin-box-names The list of comma separated names of the margin box. 
        
        @return The number of margin boxes, from the ones specified by the margin-box-names parameter, that can be used on the page.
    -->
    <xsl:function name="oxy:count-page-margin-boxes-with-content">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        
        <xsl:param name="margin-box-names"/>
        
        <xsl:variable name="found">
            <xsl:for-each select="tokenize($margin-box-names, ', ')">
                <xsl:if test="oxy:exist-page-margin-box-content($doc, $page-name, $page-side, $page-position, $page-blank, .)" xml:space="preserve">1</xsl:if>
            </xsl:for-each>
        </xsl:variable>
        
        <xsl:value-of select="string-length($found)"/>
    </xsl:function>
    
    <!-- 
        Copies all the CSS properties as attributes, except some of them matching some prefixes.
        
        @param doc The document with the page definitions 
        @param page-name The name of the page. The default page is always matched. 
        @param margin-box-name The name of the margin box. Can be "top-right-corner" for instance.
        
        @param props      The list of comma separated property prefixes that are to be excluded.
        @param default-display-align   The default value for the vertical text align.    
    -->
    <xsl:template name="copy-properties-excluding-prefix">
        <xsl:param name="doc" required="yes"/>
        <xsl:param name="page-name" required="yes"/>
        <xsl:param name="page-side" required="yes"/>
        <xsl:param name="page-position" required="yes"/>
        <xsl:param name="page-blank" required="yes"/>
        <xsl:param name="margin-box-name" required="yes"/>
        
        <xsl:param name="props" select="''"/>
        <xsl:variable name="excluded-tokens" select="tokenize($props, ', ')"/>
        
        <!-- Put the pages in a sequence. The last has more priority. -->
        <xsl:variable name="pages" select="oxy:get-matching-pages($doc, $page-name, $page-side, $page-position, $page-blank)"/>
        
        <!-- Put the matching margin page boxes in a sequence. The last has more priority. -->
        <xsl:variable name="matching-box-margins" select="oxy:get-page-margin-boxes($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name)"/>
        
        <!-- For all the distinct property names from the matching margin boxes -->
        <xsl:for-each select="distinct-values(($pages/property/@name[oxy:is-inheritable(.)], $matching-box-margins//property/@name))">
            <xsl:variable name="property" select="."/>
            
            <!-- Check if they are matching the prefix exclusion criteria -->
            <xsl:variable name="excluded">
                <xsl:for-each select="$excluded-tokens">
                    <xsl:if test="starts-with(., $property)">true</xsl:if>
                </xsl:for-each>
            </xsl:variable>
            
            <!-- If they do, then the value of the last property having that name, if is the most important. -->
            <xsl:if test="string-length($excluded) = 0">
                <xsl:attribute name="{$property}" select="oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, $property, true())/@value"/>
            </xsl:if>
        </xsl:for-each>
        
    </xsl:template>
    
    <!-- 
        Checks if a property is inheritable. 
        
        @param property The name of the property to check.
        @return true() if the property is inheritable.
    -->
    <xsl:function name="oxy:is-inheritable">
        <xsl:param name="property" as="xs:string"/>
        <xsl:copy-of select="
            if ($property = (
            'border-collapse',
            'border-spacing',
            'color',
            'direction',
            'empty-cells',
            'font-family',
            'font-size',
            'font-style',
            'font-variant',
            'font-weight',
            'font',
            'letter-spacing',
            'line-height',
            'list-style-image',
            'list-style-position',
            'list-style-type',
            'list-style',
            'orphans',
            'text-align',
            'text-indent',
            'text-transform',
            'widows')) then
                true()
            else
                false()"/>
    </xsl:function>
    
    <!--     
        Propagates all the CSS properties as FO attributes from the defined margin boxes.
        Some of the properties are excepted, for instance the 'width' and 'height' are taken from the page margins.
        
        @param ctxt                    The page box, example: top-left-corner element, etc..
        @param default-text-align      The default value for the horizontal text align.
        @param default-display-align   The default value for the vertical text align.
    -->
    <xsl:template name="generate-properties-for-margin-box">
        <xsl:param name="doc" required="yes"/>
        <xsl:param name="page-name" required="yes"/>
        <xsl:param name="page-side" required="yes"/>
        <xsl:param name="page-position" required="yes"/>
        <xsl:param name="page-blank" required="yes"/>
        <xsl:param name="margin-box-name" required="yes"/>
        <xsl:param name="margin-box-width" required="no"/>
        <xsl:param name="margin-box-height" required="no"/>
        
        <xsl:param name="default-text-align"/>
        <xsl:param name="default-display-align"/>
        
        <!-- Attributes for background image and background color -->
        <xsl:call-template name="generate-page-margin-box-background">
            <xsl:with-param name="doc" select="$doc"/>
            <xsl:with-param name="page-name" select="$page-name"/>
            <xsl:with-param name="page-side" select="$page-side"/>
            <xsl:with-param name="page-position" select="$page-position"/>
            <xsl:with-param name="page-blank" select="$page-blank"/>
            <xsl:with-param name="margin-box-name" select="$margin-box-name"/>
            <xsl:with-param name="margin-box-width" select="$margin-box-width"/>
            <xsl:with-param name="margin-box-height" select="$margin-box-height"/>
        </xsl:call-template>
        
        <!-- Vertical and horizontal alignment  -->
        <xsl:call-template name="copy-align">
            <xsl:with-param name="doc" select="$doc"/>
            <xsl:with-param name="page-name" select="$page-name"/>
            <xsl:with-param name="page-side" select="$page-side"/>
            <xsl:with-param name="page-position" select="$page-position"/>
            <xsl:with-param name="page-blank" select="$page-blank"/>
            <xsl:with-param name="margin-box-name" select="$margin-box-name"/>
            
            <xsl:with-param name="default-text-align" select="$default-text-align"/>
            <xsl:with-param name="default-display-align" select="$default-display-align"/>
        </xsl:call-template>
        
        <!-- Rest of properties -->
        <xsl:call-template name="copy-properties-excluding-prefix">
            <xsl:with-param name="doc" select="$doc"/>
            <xsl:with-param name="page-name" select="$page-name"/>
            <xsl:with-param name="page-side" select="$page-side"/>
            <xsl:with-param name="page-position" select="$page-position"/>
            <xsl:with-param name="page-blank" select="$page-blank"/>
            <xsl:with-param name="margin-box-name" select="$margin-box-name"/>
            <xsl:with-param name="props" select="'text-align, vertical-align, content, margin, width, height, image-resolution, background-size, extracted-background-image-width, extracted-background-image-height, link'"/>
        </xsl:call-template>
        
    </xsl:template>
    
    <!-- 
        Propagates the CSS text-align and vertical-align (translated to display-align) 
        CSS properties to the FO element. If there is no property defined, the defaults are used.  
        
        @param ctxt                    The page box, example: top-left-corner element, etc..    
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').                
        @param margin-box-name The margin box, for instance 'top-left-corner', 'bottom-center', etc...        
        @param default-text-align      The default value for the horizontal text align.
        @param default-display-align   The default value for the vertical text align.
    -->
    <xsl:template name="copy-align">
        <xsl:param name="doc" required="yes"/>
        <xsl:param name="page-name" required="yes"/>
        <xsl:param name="page-side" required="yes"/>
        <xsl:param name="page-position" required="yes"/>
        <xsl:param name="page-blank" required="yes"/>
        <xsl:param name="margin-box-name" required="yes"/>
        
        <xsl:param name="default-text-align"/>
        <xsl:param name="default-display-align"/>
        
        <xsl:variable name="text-align" select="oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'text-align', false())"/>
        <xsl:variable name="vertical-align" select="oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'vertical-align', false())"/>
        
        <xsl:variable name="ta">
            <xsl:choose>
                <xsl:when test="$text-align">
                    <xsl:value-of select="$text-align/@value"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="$default-text-align"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:attribute name="text-align" select="$ta"/>
        
        <xsl:variable name="va">
            <xsl:choose>
                <xsl:when test="$vertical-align">
                    <xsl:variable name="vaa" select="$vertical-align/@value"/>
                    <xsl:choose>
                        <xsl:when test="$vaa = 'top'">before</xsl:when>
                        <xsl:when test="$vaa = 'bottom'">after</xsl:when>
                        <xsl:otherwise>center</xsl:otherwise>
                    </xsl:choose>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="$default-display-align"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:attribute name="display-align" select="$va"/>
        
    </xsl:template>
    
    <!-- 
        Gets the static content nodes for a page margin box. The page is defined by the name.
        You can use this to get the content nodes for the "top-left-corner"
        
        get-static-for-page-margin-box('intro', 'top-left-corner');
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').                
        @param margin-box-name The margin box, for instance 'top-left-corner', 'bottom-center', etc...
        @return a sequence of elements.  
    -->
    <xsl:function name="oxy:get-static-for-page-margin-box" as="item()*">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="margin-box-name"/>
        
        <!-- Put the matching margin page boxes in a sequence. The last has more priority. -->
        <xsl:variable name="prop" select="oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'content', false())"/>
        <!-- Uses the last declared property matching the provided name. -->
        <xsl:copy-of select="$prop/*"/>
    </xsl:function>
    
    <!-- 
        Gets a property element for a page margin box, takes into account inheritence from the parent page element, and also overriding.
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').               
        @param margin-box-name The margin box, for instance 'top-left-corner', 'bottom-center', etc...
        @param can-be-inherited-from-page True if the property is inheritable from the parent page.
        @param property The name of the property to extract.
        @return The property element.
    -->
    <xsl:function name="oxy:get-property-element-for-page-margin-box" as="item()*">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="margin-box-name"/>
        <xsl:param name="property"/>
        <xsl:param name="can-be-inherited-from-page"/>
        
        <!-- Put the pages in a sequence. The last has more priority. -->
        <xsl:variable name="pages" select="
            if ($can-be-inherited-from-page) then
                oxy:get-matching-pages($doc, $page-name, $page-side, $page-position, $page-blank)
            else
                ()"/>
        
        <!-- Put the matching margin page boxes in a sequence. The last has more priority. -->
        <xsl:variable name="margin-boxes" select="oxy:get-page-margin-boxes($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name)"/>
        
        <!-- Uses the last declared property matching the provided name. -->
        <!-- 
            Cannot use $margin-boxes//property.. because it 
            would result in nodes sorted in document order, not in the matching 
            sequence order.
        -->
        <xsl:copy-of select="oxy:get-property-from-list($property, ($pages, $margin-boxes))"/>
    </xsl:function>
    
    <!-- 
        Gets a property specified by its name from a list of page margin boxes, and pages. 
        First it looks in normal properties, than in the important properties having that name.
        The last occurence is the most important.
        The list parameter contains the pages or page margin boxes already sorted by theyr specificity.
        @param property The name of the property
        @param list The list of elements that have other "property" elements as theyr child.      
    -->
    <xsl:function name="oxy:get-property-from-list">
        <xsl:param name="property"/>
        <xsl:param name="list"/>
        
        <xsl:variable name="important-props" select="
            for $p in $list
            return
                $p/property[@name = $property][@important = 'true']"/>
        <xsl:variable name="normal-props" select="
            for $p in $list
            return
                $p/property[@name = $property][not(@important) or @important != 'true']"/>
        
        <xsl:copy-of select="
            (for $p in ($normal-props, $important-props)
            return
                $p)[last()]"/>
    </xsl:function>
    
    <!-- 
        Gets a property for a page margin box. Does not search for the property in the matching @pages.
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').               
        @param margin-box-name The margin box, for instance 'top-left-corner', 'bottom-center', etc...
        @param property The name of the property to extract.
        @return The property value, as string.
        
    -->
    <xsl:function name="oxy:get-property-for-page-margin-box" as="xs:string">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="margin-box-name"/>
        <xsl:param name="property"/>
        
        <xsl:value-of select="oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, $property, false())/@value"/>
    </xsl:function>
    
    <!-- 
        Gets a property for a page margin box. Does search for the property in the matching @pages, but with lower priority reported to the page margin boxes.
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').               
        @param margin-box-name The margin box, for instance 'top-left-corner', 'bottom-center', etc...
        @param property The name of the property to extract.
        @return The property value, as string.
    -->
    <xsl:function name="oxy:get-property-for-page-margin-box-and-inherited-from-page" as="xs:string">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="margin-box-name"/>
        <xsl:param name="property"/>
        
        <xsl:value-of select="oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, $property, true())/@value"/>
    </xsl:function>
    
    <!--
        Tests if the argument is a page side.
    -->
    <xsl:function name="oxy:no-page-side" as="xs:boolean">
        <xsl:param name="pseudo"/>
        <xsl:sequence select="
            if ($pseudo) then
                not(contains($pseudo, 'left')) and not(contains($pseudo, 'right'))
            else
                true()"/>
    </xsl:function>
    
    <!--
        Tests if the argument is a page position.
    -->
    <xsl:function name="oxy:no-page-position" as="xs:boolean">
        <xsl:param name="pseudo"/>
        <xsl:sequence select="
            if ($pseudo) then
                not(contains($pseudo, 'first')) and not(contains($pseudo, 'last'))
            else
                true()"/>
    </xsl:function>
    
    <!--
        Tests if the argument is a page blank.
    -->
    <xsl:function name="oxy:no-page-blank" as="xs:boolean">
        <xsl:param name="pseudo"/>
        <xsl:sequence select="
            if ($pseudo) then
                not(contains($pseudo, 'blank'))
            else
                true()"/>
    </xsl:function>
    
    <!-- 
        Gets the matching page definitions, putting them in a sequence  
        that starts with the least specific pages, and ending 
        with the most specific. In this way we solve also page definitions 
        of the same specificity. The properties from last one are more important. 
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').
        
        @return a sequence of elements.  
    -->
    <xsl:function name="oxy:get-matching-pages" as="item()*">
        <xsl:param name="doc"/>
        <xsl:param name="page-name" as="xs:string"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        
        <xsl:variable name="val-def" xml:space="preserve" select="
            $doc/root/pages/page[not(@name) or @name eq 'css2fo-default'][oxy:no-page-side(@pseudo)][oxy:no-page-position(@pseudo)][oxy:no-page-blank(@pseudo)],
            $doc/root/pages/page[not(@name) or @name eq 'css2fo-default'][contains(@pseudo, $page-side)][oxy:no-page-position(@pseudo)][oxy:no-page-blank(@pseudo)],
            $doc/root/pages/page[not(@name) or @name eq 'css2fo-default'][oxy:no-page-side(@pseudo)][contains(@pseudo, $page-position)][oxy:no-page-blank(@pseudo)],
            $doc/root/pages/page[not(@name) or @name eq 'css2fo-default'][contains(@pseudo, $page-side)][contains(@pseudo, $page-position)][oxy:no-page-blank(@pseudo)],
            
            $doc/root/pages/page[@name = $page-name and @name ne 'css2fo-default'][oxy:no-page-side(@pseudo)][oxy:no-page-position(@pseudo)][oxy:no-page-blank(@pseudo)],
            $doc/root/pages/page[@name = $page-name and @name ne 'css2fo-default'][contains(@pseudo, $page-side)][oxy:no-page-position(@pseudo)][oxy:no-page-blank(@pseudo)],
            $doc/root/pages/page[@name = $page-name and @name ne 'css2fo-default'][oxy:no-page-side(@pseudo)][contains(@pseudo, $page-position)][oxy:no-page-blank(@pseudo)],
            $doc/root/pages/page[@name = $page-name and @name ne 'css2fo-default'][contains(@pseudo, $page-side)][contains(@pseudo, $page-position)][oxy:no-page-blank(@pseudo)],
            
            $doc/root/pages/page[not(@name) or @name eq 'css2fo-default'][oxy:no-page-side(@pseudo)][oxy:no-page-position(@pseudo)][contains(@pseudo, $page-blank)],
            $doc/root/pages/page[not(@name) or @name eq 'css2fo-default'][contains(@pseudo, $page-side)][oxy:no-page-position(@pseudo)][contains(@pseudo, $page-blank)],
            $doc/root/pages/page[not(@name) or @name eq 'css2fo-default'][oxy:no-page-side(@pseudo)][contains(@pseudo, $page-position)][contains(@pseudo, $page-blank)],
            $doc/root/pages/page[not(@name) or @name eq 'css2fo-default'][contains(@pseudo, $page-side)][contains(@pseudo, $page-position)][contains(@pseudo, $page-blank)],
            
            $doc/root/pages/page[@name = $page-name and @name ne 'css2fo-default'][oxy:no-page-side(@pseudo)][oxy:no-page-position(@pseudo)][contains(@pseudo, $page-blank)],
            $doc/root/pages/page[@name = $page-name and @name ne 'css2fo-default'][contains(@pseudo, $page-side)][oxy:no-page-position(@pseudo)][contains(@pseudo, $page-blank)],
            $doc/root/pages/page[@name = $page-name and @name ne 'css2fo-default'][oxy:no-page-side(@pseudo)][contains(@pseudo, $page-position)][contains(@pseudo, $page-blank)],
            $doc/root/pages/page[@name = $page-name and @name ne 'css2fo-default'][contains(@pseudo, $page-side)][contains(@pseudo, $page-position)][contains(@pseudo, $page-blank)]
            "/>
        
        <xsl:if test="$debug">
            <xsl:message>Value for <xsl:value-of select="$page-name"/> - <xsl:value-of select="$page-side"/> - <xsl:value-of select="$page-position"/> - <xsl:value-of select="$page-blank"/> = </xsl:message>
            
            <xsl:for-each select="$val-def">
                <xsl:message> -- <xsl:value-of select="@name"/>, <xsl:value-of select="@pseudo"/>
                </xsl:message>
            </xsl:for-each>
        </xsl:if>
        <xsl:sequence select="$val-def"/>
    </xsl:function>
    
    <!-- 
        Gets the sum of horizontal padding and borders width. 
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').               
        @param margin-box-name The margin box, for instance 'top-left-corner', 'bottom-center', etc...
        
        @return a string expression with the all the horizontal dimensions, 
        aggregated using the ' + ' operator. May return null, if there are no insets on the element.
    -->
    <xsl:function as="xs:string" name="oxy:get-insets-horizontal">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="margin-box-name"/>
        
        <xsl:variable name="insets-left" select="oxy:get-insets-left($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name)"/>
        <xsl:variable name="insets-right" select="oxy:get-insets-right($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name)"/>
        
        <xsl:variable name="insets" select="$insets-left"/>
        <xsl:variable name="insets" select="
            if ($insets-right) then
                if (not($insets)) then
                    $insets-right
                else
                    concat($insets, ' + ', $insets-right)
            else
                $insets"/>
        
        <xsl:value-of select="$insets"/>
    </xsl:function>
    
    <!-- 
        Gets the sum of horizontal left padding and borders width. 
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').               
        @param margin-box-name The margin box, for instance 'top-left-corner', 'bottom-center', etc...
        
        @return a string expression with the all the horizontal dimensions, 
        aggregated using the ' + ' operator. May return null, if there are no insets on the element.
    -->
    <xsl:function as="xs:string" name="oxy:get-insets-left">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="margin-box-name"/>
        
        <xsl:variable name="insets" select="oxy:get-property-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'padding-left')"/>
        <xsl:variable name="border" select="oxy:get-property-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'border-left-width')"/>
        <xsl:variable name="insets" select="
            if ($border) then
                if (not($insets)) then
                    $border
                else
                    concat($insets, ' + ', $border)
            else
                $insets"/>
        
        <xsl:value-of select="$insets"/>
    </xsl:function>
    
    <!-- 
        Gets the sum of right padding and borders width. 
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').               
        @param margin-box-name The margin box, for instance 'top-left-corner', 'bottom-center', etc...
        
        @return a string expression with the all the horizontal dimensions, 
        aggregated using the ' + ' operator. May return null, if there are no insets on the element.
    -->
    <xsl:function as="xs:string" name="oxy:get-insets-right">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="margin-box-name"/>
        
        <xsl:variable name="insets" select="oxy:get-property-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'padding-right')"/>
        <xsl:variable name="border" select="oxy:get-property-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'border-right-width')"/>
        <xsl:variable name="insets" select="
            if ($border) then
                if (not($insets)) then
                    $border
                else
                    concat($insets, ' + ', $border)
            else
                $insets"/>
        
        <xsl:value-of select="$insets"/>
    </xsl:function>
    
    <!-- 
        Gets the sum of vertical padding and borders width. 
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').               
        @param margin-box-name The margin box, for instance 'top-left-corner', 'bottom-center', etc...
        
        @return a string expression with the all the vertical dimensions, 
        aggregated using the ' + ' operator. May return null, if there are no insets on the element.
    -->
    <xsl:function as="xs:string" name="oxy:get-insets-vertical">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="margin-box-name"/>
        
        <xsl:variable name="insets-top" select="oxy:get-insets-top($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name)"/>
        <xsl:variable name="insets-bottom" select="oxy:get-insets-bottom($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name)"/>
        
        
        <xsl:variable name="insets" select="$insets-top"/>
        <xsl:variable name="insets" select="
            if ($insets-bottom) then
                if (not($insets)) then
                    $insets-bottom
                else
                    concat($insets, ' + ', $insets-bottom)
            else
                $insets"/>
        <xsl:value-of select="$insets"/>
    </xsl:function>
    
    <!-- 
        Gets the sum of bottom padding and borders width. 
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').               
        @param margin-box-name The margin box, for instance 'top-left-corner', 'bottom-center', etc...
        
        @return a string expression with the all the vertical dimensions, 
        aggregated using the ' + ' operator. May return null, if there are no insets on the element.
    -->
    <xsl:function as="xs:string" name="oxy:get-insets-bottom">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="margin-box-name"/>
        
        <xsl:variable name="insets" select="oxy:get-property-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'padding-bottom')"/>
        <xsl:variable name="border" select="oxy:get-property-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'border-bottom-width')"/>
        
        <xsl:variable name="insets" select="
            if ($border) then
                if (not($insets)) then
                    $border
                else
                    concat($insets, ' + ', $border)
            else
                $insets"/>
        
        <xsl:value-of select="$insets"/>
    </xsl:function>
    
    <!-- 
        Gets the sum of vertical top padding and borders width. 
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').               
        @param margin-box-name The margin box, for instance 'top-left-corner', 'bottom-center', etc...
        
        @return a string expression with the all the vertical dimensions, 
        aggregated using the ' + ' operator. May return null, if there are no insets on the element.
    -->
    <xsl:function as="xs:string" name="oxy:get-insets-top">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="margin-box-name"/>
        
        <xsl:variable name="insets" select="oxy:get-property-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'padding-top')"/>
        <xsl:variable name="border" select="oxy:get-property-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'border-top-width')"/>
        
        <xsl:variable name="insets" select="
            if ($border) then
                if (not($insets)) then
                    $border
                else
                    concat($insets, ' + ', $border)
            else
                $insets"/>
        
        <xsl:value-of select="$insets"/>
    </xsl:function>
    
    <!-- 
        Gets all the background properties associated to a page. Takes into account overriding.  
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').               
        
        @return a list of property elements.
    -->
    <xsl:function name="oxy:get-page-background-properties">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        
        <xsl:sequence select="
            (
            oxy:get-property-element-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'background'),
            oxy:get-property-element-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'background-image'),
            oxy:get-property-element-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'background-color'),
            oxy:get-property-element-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'background-repeat'),
            oxy:get-property-element-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'background-position')
            )
            "/>
    </xsl:function>
    
    <!-- 
        Gets all the foreground properties associated to a page. Takes into account overriding.  
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').               
        
        @return a list of property elements.
    -->
    <xsl:function name="oxy:get-page-foreground-properties">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        
        <xsl:sequence select="
            (
            oxy:get-property-element-for-page($doc, $page-name, $page-side, $page-position, $page-blank, 'oxy-foreground-image')
            )
            "/>
    </xsl:function>
    
    <!-- 
        Gets all the background properties associated to a page margin box. Takes into account overriding.  
        
        @param doc The context document. We need it to search the property in it
        @param page-name The name of the page.
        @param page-side The side of the page ('left', 'right', 'any').
        @param page-position The position of the page ('first', 'last', 'any').
        @param page-blank The side of the page ('blank', 'any').         
        @param margin-box-name The name of the margin box.
        
        @return a list of property elements.
    -->
    <xsl:function name="oxy:get-page-margin-box-background-properties">
        <xsl:param name="doc"/>
        <xsl:param name="page-name"/>
        <xsl:param name="page-side"/>
        <xsl:param name="page-position"/>
        <xsl:param name="page-blank"/>
        <xsl:param name="margin-box-name"/>
        
        <xsl:sequence select="
            (
            oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'background', false()),
            oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'background-image', false()),
            oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'background-color', false()),
            oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'background-repeat', false()),
            oxy:get-property-element-for-page-margin-box($doc, $page-name, $page-side, $page-position, $page-blank, $margin-box-name, 'background-position', false())
            )
            "/>
    </xsl:function>
    
</xsl:stylesheet>
