<!-- 
	Generates the metadata for a PDF document. 
-->
<xsl:stylesheet 
    xmlns:xsl="http://www.w3.org/1999/XSL/Transform" 
    xmlns:xs="http://www.w3.org/2001/XMLSchema" 
    xmlns:fo="http://www.w3.org/1999/XSL/Format" 
    xmlns:oxy="http://www.oxygenxml.com/css2fo" 
    xmlns:css="http://www.w3.org/1998/CSS"
    xmlns:dc="http://purl.org/dc/elements/1.1/"
    xmlns:xmp="http://ns.adobe.com/xap/1.0/"
    xmlns:x="adobe:ns:meta/"
    xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#"
    xmlns:pdf="http://ns.adobe.com/pdf/1.3/"
    xmlns:fop-pdf="http://xmlgraphics.apache.org/fop/extensions/pdf"
    xmlns:nd="http://www.oxygenxml.com/css2fo/named-destinations"
    xmlns:xmpRights="http://ns.adobe.com/xap/1.0/rights/"
    
    exclude-result-prefixes="xs oxy dc xmp x pdf fop-pdf rdf nd xmpRights" version="2.0"
    xpath-default-namespace="http://www.w3.org/1998/CSS">
    
    <!--
        Collects the relevant properties that are set across the document, prefering the first 
        ocurrence of each one and sets them in a XMP structure.

		Context: The root element.
      -->    
  <xsl:template name="generate-metadata">
    
    <!-- Metadata for the PDF viewer -->
    <xsl:variable name="viewer-hide-toolbar" select="(//*/@css:oxy-pdf-viewer-hide-toolbar)[1]"/>
    <xsl:variable name="viewer-hide-menubar" select="(//*/@css:oxy-pdf-viewer-hide-menubar)[1]"/>
    <xsl:variable name="viewer-fit-window" select="(//*/@css:oxy-pdf-viewer-fit-window)[1]"/>
    <xsl:variable name="viewer-page-layout" select="(//*/@css:oxy-pdf-viewer-page-layout)[1]"/>
    <xsl:variable name="viewer-page-mode" select="(//*/@css:oxy-pdf-viewer-page-mode)[1]"/>
    <xsl:variable name="viewer-display-filename" select="(//*/@css:oxy-pdf-viewer-display-filename)[1]"/>
    
    <!-- Metadata that appears in the Acrobat Reader "Document Properties"/"Description" page -->
    <xsl:variable name="title" select="(//css:oxy-pdf-meta-title)[1]"/>        
    <xsl:variable name="author" select="(//css:oxy-pdf-meta-author)[1]"/>
    <xsl:variable name="copyright" select="(//css:oxy-pdf-meta-copyright)[1]"/>
    <xsl:variable name="copyrighted" select="(//css:oxy-pdf-meta-copyrighted)[1]"/>
    <xsl:variable name="copyright-url" select="(//css:oxy-pdf-meta-copyright-url)[1]"/>
    <xsl:variable name="description" select="(//css:oxy-pdf-meta-description)[1]"/>
    <xsl:variable name="keywords" select="(//css:oxy-pdf-meta-keywords)[1]"/>
    <xsl:variable name="keyword-set" select="//css:oxy-pdf-meta-keyword"/>
    
    <fo:declarations>
      <x:xmpmeta>
        <rdf:RDF>
          <rdf:Description>
            
            <xsl:if test="$title">
              <dc:title>
                <xsl:value-of select="oxy:get-all-static-as-text($title)"/>
              </dc:title>
            </xsl:if>
            
            <xsl:if test="$author">
              <dc:creator>
                <xsl:value-of select="oxy:get-all-static-as-text($author)"/>
              </dc:creator>
            </xsl:if>
            
            <xsl:if test="$description">
              <dc:description>
                <xsl:value-of select="oxy:get-all-static-as-text($description)"/>
              </dc:description>
            </xsl:if>
            
            <xsl:if test="$copyrighted">
              <xsl:choose>
                <xsl:when test="$copyrighted='copyrighted'">
                  <xmpRights:Marked>True</xmpRights:Marked>
                </xsl:when>
                <xsl:when test="$copyrighted='public-domain'">
                  <xmpRights:Marked>False</xmpRights:Marked>
                </xsl:when>
              </xsl:choose>
            </xsl:if>
            
            <xsl:if test="$copyright">
              <dc:rights>
                <rdf:Alt>
                  <rdf:li xml:lang="x-default"><xsl:value-of select="oxy:get-all-static-as-text($copyright)"/></rdf:li>
                </rdf:Alt>
              </dc:rights>
            </xsl:if>
            
            <xsl:if test="$copyright-url">
              <xmpRights:WebStatement>
                <xsl:value-of select="oxy:get-all-static-as-text($copyright-url)"/>
              </xmpRights:WebStatement>
            </xsl:if>
            
            <xsl:choose>
              <xsl:when test="$keywords">
                <pdf:Keywords>
                  <xsl:value-of select="oxy:get-all-static-as-text($keywords)"/>
                </pdf:Keywords>
              </xsl:when>
              <xsl:when test="$keyword-set">
                <pdf:Keywords>
                  <xsl:for-each select="$keyword-set">
                    <xsl:value-of select="oxy:get-all-static-as-text(.)"/>
                    <xsl:if test="position() != last()">, </xsl:if>
                  </xsl:for-each>
                </pdf:Keywords>
              </xsl:when>
            </xsl:choose>
            
            <xmp:CreatorTool>
              <xsl:variable name="creator">
                <xsl:value-of select="$application-name"/>
                
                <xsl:if test="$application-version">
                  Version <xsl:value-of select="$application-version"/>                            
                </xsl:if>
                
                <xsl:if test="$application-build-timestamp">
                  Build <xsl:value-of select="$application-build-timestamp"/>                            
                </xsl:if>
              </xsl:variable>
              
              <xsl:value-of select="normalize-space($creator)"/>
            </xmp:CreatorTool>
          </rdf:Description>
        </rdf:RDF>
      </x:xmpmeta>
      
      <!-- Metadata that appears in the Acrobat Reader "Document Properties"/"Custom" page -->
      <xsl:variable name="custom-meta-names" select="distinct-values(//oxy-pdf-meta-custom/@name)"/>
      <xsl:if test="exists($custom-meta-names)">
        <fop-pdf:info>
          <xsl:variable name="doc" select="/"/>
          <xsl:for-each select="$custom-meta-names">
            <xsl:variable name="value" select="oxy:get-all-static-as-text(($doc//oxy-pdf-meta-custom[@name = current()])[1])"/>
            <xsl:if test="$value">
              <fop-pdf:name key="{.}"><xsl:value-of select="$value"/></fop-pdf:name>
            </xsl:if>
          </xsl:for-each>
        </fop-pdf:info>
      </xsl:if>
      
      <fop-pdf:catalog>
        <xsl:if test="$viewer-page-layout">
          <fop-pdf:name key="PageLayout">
            <xsl:choose>                          
              <xsl:when test="$viewer-page-layout = 'single-page'">SinglePage</xsl:when>
              <xsl:when test="$viewer-page-layout = 'one-column'">OneColumn</xsl:when>
              <xsl:when test="$viewer-page-layout = 'two-column-left'">TwoColumnLeft</xsl:when>
              <xsl:when test="$viewer-page-layout = 'two-column-right'">TwoColumnRight</xsl:when>
              <xsl:when test="$viewer-page-layout = 'two-page-left'">TwoPageLeft</xsl:when>
              <xsl:when test="$viewer-page-layout = 'two-page-right'">TwoPageRight</xsl:when>
            </xsl:choose>
          </fop-pdf:name>               
        </xsl:if>
        
        <xsl:if test="$viewer-page-mode">
          <fop-pdf:name key="PageMode">
            <xsl:choose>                          
              <xsl:when test="$viewer-page-mode = 'use-none'">UseNone</xsl:when>
              <xsl:when test="$viewer-page-mode = 'use-outlines'">UseOutlines</xsl:when>
              <xsl:when test="$viewer-page-mode = 'use-thumbs'">UseThumbs</xsl:when>
              <xsl:when test="$viewer-page-mode = 'full-screen'">FullScreen</xsl:when>
            </xsl:choose>
          </fop-pdf:name>
        </xsl:if>
        
        <fop-pdf:dictionary key="ViewerPreferences">
          <xsl:if test="$title and not($viewer-display-filename)">
            <fop-pdf:boolean key="DisplayDocTitle">true</fop-pdf:boolean>
          </xsl:if>
          <xsl:if test="$viewer-hide-toolbar">
            <fop-pdf:boolean key="HideToolbar"><xsl:value-of select="$viewer-hide-toolbar"/></fop-pdf:boolean>                       
          </xsl:if>
          <xsl:if test="$viewer-hide-menubar">
            <fop-pdf:boolean key="HideMenubar"><xsl:value-of select="$viewer-hide-menubar"/></fop-pdf:boolean>                       
          </xsl:if>
          <xsl:if test="$viewer-fit-window">
            <fop-pdf:boolean key="FitWindow"><xsl:value-of select="$viewer-fit-window"/></fop-pdf:boolean>                       
          </xsl:if>
        </fop-pdf:dictionary>
      </fop-pdf:catalog>
    </fo:declarations>
    
    <xsl:call-template name="generate-named-destination-list"/>
    
  </xsl:template>
  
  <!-- Do not propagate the meta attributes to the FO, are useless there. -->
  <xsl:template match="css:oxy-pdf-meta-title" priority="2"/>
  <xsl:template match="css:oxy-pdf-meta-author" priority="2"/>
  <xsl:template match="css:oxy-pdf-meta-description" priority="2"/>
  <xsl:template match="css:oxy-pdf-meta-keywords" priority="2"/>
  <xsl:template match="css:oxy-pdf-meta-keyword" priority="2"/>
  <xsl:template match="css:oxy-pdf-meta-copyright" priority="2"/>
  <xsl:template match="css:oxy-pdf-meta-copyrighted" priority="2"/>
  <xsl:template match="css:oxy-pdf-meta-copyright-url" priority="2"/>
  <xsl:template match="css:oxy-pdf-meta-custom" priority="2"/>
  
  <xsl:template match="@css:oxy-pdf-viewer-fit-window" priority="2"/>
  <xsl:template match="@css:oxy-pdf-viewer-hide-menubar" priority="2"/>
  <xsl:template match="@css:oxy-pdf-viewer-hide-toolbar" priority="2"/>
  <xsl:template match="@css:oxy-pdf-viewer-page-layout" priority="2"/>
  <xsl:template match="@css:oxy-pdf-viewer-page-mode" priority="2"/>
  <xsl:template match="@css:oxy-pdf-viewer-display-filename" priority="2"/>
    
</xsl:stylesheet>