<?xml version="1.0" encoding="UTF-8"?>
<!--
    This stylesheet deals with the differences between the margins treatment in CSS versus XSL-FO.
    
    In XSL-FO, an element without a specified margin (any of the margins, or with a value of 
    "auto" ) is bleeding its left and right padding out of the parent. 
    
    Fortunately, setting margin="0" fixes this, resulting in a CSS compatible layout.
-->
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema" xmlns:css="http://www.w3.org/1998/CSS"
    xmlns:fo="http://www.w3.org/1999/XSL/Format" xmlns:oxy="http://www.oxygenxml.com/css2fo"
    xmlns:fox="http://xmlgraphics.apache.org/fop/extensions"
    exclude-result-prefixes="xs oxy fox" version="2.0">


    <!-- 
        'Yes' if the top and bottom margins associated to a block element should be 
        discarded when the block is at the top or bottom of the page. This is the default and is
        implemented by using space-before and space-after instead margin-top si margin-bottom.         
    -->
    <xsl:param name="drop-block-margins-at-page-boundary" select="'yes'"/>


    <!-- 
        Leave the margins as properties if they are negative, or is 
        an element that spans all columns or is a table caption that repeats.
        
        For repeated captions we have a problem because the we remove the caption Knuth
        elements from the list and process them separately, out of context. Unfortunately 
        this makes the space-before  and space-after set on them be discarded, so
        we must use margins. The disadvantage is that the margins will not collapse with 
        other neighboring elements margins.
     -->
    <xsl:function name="oxy:is-margin-convertible-to-space" as="xs:boolean">
      <xsl:param name="margin-attr"/>
      <xsl:sequence select="$drop-block-margins-at-page-boundary = 'yes' 
        and not(starts-with($margin-attr, '-')) 
        and not($margin-attr/ancestor::*/@css:column-span='all')
        and not(
          $margin-attr/ancestor::*/@css:display='table-caption' and 
          $margin-attr/ancestor::*/@css:oxy-caption-repeat-on-next-pages='yes')"/>      
    </xsl:function>

    <!-- 
        The space-before is smarter than the margin top. If an element with a space-before
        is placed at the start of a page, this space is discarded.
        
    -->
    <xsl:template match="@css:margin-top[oxy:is-margin-convertible-to-space(.)]">
      <xsl:attribute name="space-before" select="."/>
    </xsl:template>
	<xsl:template match="@css:margin-bottom[oxy:is-margin-convertible-to-space(.)]">
      <xsl:attribute name="space-after" select="."/>
    </xsl:template>
    
    <!-- 
        
        Auto Margins.
        
        These are not supported in FO, so we need to construct a table with proportional columns that wrap the block with auto margins.
    
    	Cases:
	        1. Auto margins on both sides. This is used to center the content.
		        
		    	margin-left:auto;
		    	margin-right:auto;
		    	
		    	In this case we build a table with three columns, the middle wrapping the centered content, the outher are propoprtional and span the rest of the space.
	    	2. Auto margin on right side. This is used to align left the content.
	        
		    	margin-right:auto;
		    	
		    	In this case we build a table with two columns, the left one wrapping the content while the right one is propoprtional and span the rest of the space.  
	        3. Auto margin on left side. This is used to align right the content.
	        
		    	margin-left:auto;
		    	
		    	In this case we build a table with two columns, the right one wrapping the content while the left one is propoprtional and span the rest of the space.  
    -->

    <xsl:template
        match="*[@css:display = 'block'][@css:width][@css:margin-left = 'auto' or @css:margin-right = 'auto']"
        priority="4">


        <!-- Computes the width of the columns that implement the dynamic/auto width. -->
        <xsl:variable name="auto-width" select="oxy:get-auto-margin-width(.)"/>

        <xsl:variable name="horizontal-insets" select="oxy:get-insets-horizontal(.)"/>
        <xsl:variable name="left-insets" select="oxy:get-insets-left(.)"/>
        <xsl:variable name="right-insets" select="oxy:get-insets-right(.)"/>
        
        <fo:table width="100%" table-layout="fixed">

            <!-- Migrate the top and bottom borders on the table -->
            <xsl:apply-templates select="@css:margin-top | @css:margin-bottom"/>

            <xsl:choose>
                <xsl:when test="@css:margin-left = 'auto'">
                    <fo:table-column column-width="{$auto-width}"/>
                </xsl:when>
                <xsl:when test="@css:margin-left">
                    <fo:table-column column-width="{@css:margin-left}"/>
                </xsl:when>
            </xsl:choose>

            <xsl:choose>
                <xsl:when test="$horizontal-insets">
                    <fo:table-column column-width="{@css:width} + {$horizontal-insets}"/>
                </xsl:when>
                <xsl:otherwise>
                    <fo:table-column column-width="{@css:width}"/>
                </xsl:otherwise>
            </xsl:choose>

            <xsl:choose>
                <xsl:when test="@css:margin-right = 'auto'">
                    <fo:table-column column-width="{$auto-width}"/>
                </xsl:when>
                <xsl:when test="@css:margin-right">
                    <fo:table-column column-width="{@css:margin-right}"/>
                </xsl:when>
            </xsl:choose>

            <fo:table-body>
                <fo:table-row>
                    <xsl:if test="@css:margin-left">
                        <fo:table-cell start-indent="0" end-indent="0">
                            <fo:block/>
                        </fo:table-cell>
                    </xsl:if>
                    <fo:table-cell start-indent="0" end-indent="0">
                        <xsl:if test="$left-insets">
                            <xsl:attribute name="start-indent" select="$left-insets"/>
                        </xsl:if>
                        <xsl:if test="$right-insets">
                            <xsl:attribute name="end-indent" select="$right-insets"/>
                        </xsl:if>
                        <xsl:next-match>
                            <xsl:with-param name="inhibit-margins-and-width" select="true()" tunnel="yes"/>
                        </xsl:next-match>
                    </fo:table-cell>
                    <xsl:if test="@css:margin-right">
                        <fo:table-cell start-indent="0" end-indent="0">
                            <fo:block/>
                        </fo:table-cell>
                    </xsl:if>
                </fo:table-row>
            </fo:table-body>
        </fo:table>
    </xsl:template>

    <!-- These margins were migrated on the structure of the fo:table that implements the auto margins. -->
    <xsl:template
        match="
        	*[@css:display = 'block'][@css:width][@css:margin-left = 'auto' or @css:margin-right = 'auto']/@css:margin-left |
        	*[@css:display = 'block'][@css:width][@css:margin-left = 'auto' or @css:margin-right = 'auto']/@css:margin-right |
        	*[@css:display = 'block'][@css:width][@css:margin-left = 'auto' or @css:margin-right = 'auto']/@css:margin-top |
            *[@css:display = 'block'][@css:width][@css:margin-left = 'auto' or @css:margin-right = 'auto']/@css:margin-bottom |
            *[@css:display = 'block'][@css:width][@css:margin-left = 'auto' or @css:margin-right = 'auto']/@css:width"
        priority="4">

        <xsl:param name="inhibit-margins-and-width" select="false()" tunnel="yes"/>
        <xsl:if test="not($inhibit-margins-and-width)">
            <xsl:next-match/>
        </xsl:if>
    </xsl:template>

    <!-- 
    
        All other auto margins are converted to zero.
        
    -->
    <xsl:template
        match="
        @css:margin-right[. = 'auto'] |
        @css:margin-left[. = 'auto']
        ">
        <xsl:attribute name="{local-name()}" select="'0'"/>
    </xsl:template>
    

    <!-- 
    
        Remove margins from the table cell elements, if they were set from the CSS. 
    
    -->
    <xsl:template
        match="
        @css:margin-top[../@css:display = 'table-cell'] |
        @css:margin-right[../@css:display = 'table-cell'] |
        @css:margin-bottom[../@css:display = 'table-cell'] |
        @css:margin-left[../@css:display = 'table-cell']" priority="2"/>
    
    
    <!-- 
        
        Adds margin zero attributes for all border attributes that do not have a margin correspondent. 
        
    -->
    <xsl:template match="@css:border-left-width[not(../@css:margin-left)][oxy:margin-fixable(..)]">
        <xsl:attribute name="{local-name()}" select="."/>
        <xsl:attribute name="margin-left" select="'0'"/>
    </xsl:template>
    <xsl:template match="@css:border-right-width[not(../@css:margin-right)][oxy:margin-fixable(..)]">
        <xsl:attribute name="{local-name()}" select="."/>
        <xsl:attribute name="margin-right" select="'0'"/>
    </xsl:template>

    <!-- 
        
        Adds margin zero attributes for all padding attributes that do not have a margin correspondent. 
    
    -->
    <xsl:template
        match="@css:padding-right[not(../@css:border-right-width)][not(../@css:margin-right)][oxy:margin-fixable(..)]">
        <xsl:attribute name="{local-name()}" select="."/>
        <xsl:attribute name="margin-right" select="'0'"/>
    </xsl:template>
    <xsl:template
        match="@css:padding-left[not(../@css:border-left-width)][not(../@css:margin-left)][oxy:margin-fixable(..)]">
        <xsl:attribute name="{local-name()}" select="."/>
        <xsl:attribute name="margin-left" select="'0'"/>
    </xsl:template>

    <!-- 
    
        Checks if the margins should be fixed for an element. (i.e. to add margins='0' when having padding or borders)
        
        @param element The element to be tested.
    
        @return true if the element is not an inline or a table cell, table column.
    -->
    <xsl:function name="oxy:margin-fixable" as="xs:boolean">
        <xsl:param name="element"/>
        <xsl:variable name="display" select="$element/@css:display"/>
        <xsl:value-of
            select="
                not($display = 'inline') and
                not($display = 'table-cell') and
                not($display = 'table-column')"
        />
    </xsl:function>



    <!--
    	Computes the with of the auto margins that can be used as column widths in a FO table. This is needed to implement CSS margin-left:auto, margin-right:auto. 
    -->
    <xsl:function name="oxy:get-auto-margin-width">
        <xsl:param name="elem"/>

        <!-- For an exact positioning, we need to take into account that the width 
                property applies to the content: the borders and paddings come on the outside. -->
        <xsl:variable name="horizontal-insets" select="oxy:get-insets-horizontal($elem)"/>
        <xsl:variable name="horizontal-margins-not-auto"
            select="oxy:get-non-auto-margins-horizontal($elem)"/>

        <xsl:variable name="nr-of-autos"
            select="
                if ($elem/@css:margin-left = 'auto' and $elem/@css:margin-right = 'auto') then
                    2
                else
                    1"/>

        <xsl:variable name="expr" select="concat('(100% - ', $elem/@css:width)"/>
        <xsl:variable name="expr"
            select="
                if ($horizontal-insets and $horizontal-insets != '0') then
                    concat($expr, ' - (', $horizontal-insets, ')')
                else
                    $expr"/>
        <xsl:variable name="expr"
            select="
                if ($horizontal-margins-not-auto and $horizontal-margins-not-auto != '0') then
                    concat($expr, ' - (', $horizontal-margins-not-auto, ')')
                else
                    $expr"/>
        <xsl:variable name="expr" select="concat($expr, ')')"/>

        <xsl:value-of select="concat($expr, ' div ', $nr-of-autos)"/>
    </xsl:function>



    <!--
        
        Gets the horizontal insets for an element. This is the sum of left and right paddings and borders.
        
        @param elem The element
        @return the insets expression, or nothing.
    -->
    <xsl:function name="oxy:get-insets-horizontal" as="xs:string">
        <xsl:param name="elem"/>

        <xsl:variable name="left-insets" select="oxy:get-insets-left($elem)"/>
        <xsl:variable name="right-insets" select="oxy:get-insets-right($elem)"/>
        <xsl:variable name="insets" select="$left-insets"/>
        <xsl:variable name="insets"
            select="
                if ($right-insets) then
                    if ($insets) then
                        concat($insets, ' + ', $right-insets)
                    else
                        $right-insets
                else
                    $insets"/>

        <xsl:value-of select="$insets"/>
    </xsl:function>

    <!--
        
        Gets the horizontal right insets for an element. This is the sum of right paddings and borders.
        
        @param elem The element
        @return the insets expression, or nothing.
    -->
    <xsl:function name="oxy:get-insets-right" as="xs:string">
        <xsl:param name="elem"/>

        <xsl:variable name="insets" select="$elem/@css:padding-right"/>
        <xsl:variable name="insets"
            select="
                if ($elem/@css:border-right-width) then
                    if ($insets) then
                        concat($insets, ' + ', $elem/@css:border-right-width)
                    else
                        $elem/@css:border-right-width
                else
                    $insets"/>

        <xsl:value-of select="$insets"/>
    </xsl:function>


    <!--
        
        Gets the horizontal left insets for an element. This is the sum of left paddings and borders.
        
        @param elem The element
        @return the insets expression, or nothing.
    -->
    <xsl:function name="oxy:get-insets-left" as="xs:string">
        <xsl:param name="elem"/>

        <xsl:variable name="insets" select="$elem/@css:padding-left"/>
        <xsl:variable name="insets"
            select="
                if ($elem/@css:border-left-width) then
                    if ($insets) then
                        concat($insets, ' + ', $elem/@css:border-left-width)
                    else
                        $elem/@css:border-left-width
                else
                    $insets"/>

        <xsl:value-of select="$insets"/>
    </xsl:function>

    <!--
        
        Gets the horizontal non-auto margins for an element. This is the sum of left and right margins.
        
        @param elem The element
        @return the margins expression, or nothing.
    -->
    <xsl:function name="oxy:get-non-auto-margins-horizontal" as="xs:string">
        <xsl:param name="elem"/>

        <xsl:variable name="margins" select="$elem/@css:margin-left[. != 'auto']"/>
        <xsl:variable name="margins"
            select="
                if ($elem/@css:margin-right[. != 'auto']) then
                    if ($margins) then
                        concat('( ', $margins, ' + ', $elem/@css:margin-right, ' )')
                    else
                        $elem/@css:margin-right
                else
                    $margins"/>
        <xsl:value-of select="$margins"/>
    </xsl:function>


    <!-- 
        
        Adds color to the border by taking it from the foreground, if it is missing.  
    
    -->
    <xsl:template match="
                    @css:border-style[not(../@css:border-color)]|
                    @css:border-bottom-style[not(../@css:border-bottom-color)]|
                    @css:border-top-style[not(../@css:border-top-color)]|
                    @css:border-left-style[not(../@css:border-left-color)]|
                    @css:border-right-style[not(../@css:border-right-color)]"
        priority="3">
        
        <xsl:variable name="color" select="((ancestor-or-self::*[@css:color])/@css:color)[last()]"/>        
        <xsl:if test="$color">
            <xsl:attribute name="{substring-before(local-name(),'-style')}-color" select="$color"/>
        </xsl:if>
        
        <xsl:next-match/>
    </xsl:template>

    <xsl:variable name="border-width-thick" select="'3pt'"/>
    <xsl:variable name="border-width-medium" select="'2pt'"/>
    <xsl:variable name="border-width-thin" select="'1pt'"/>
    
    <!-- 
        
        Remaps the thick, medium, thin to some values similar to the web browsers. We could leave 
        them as they are, but FOP uses thinner values.
        
    -->
    <xsl:template match="
        @css:border-width[.='thin']|
        @css:border-bottom-width[.='thin']|
        @css:border-top-width[.='thin']|
        @css:border-left-width[.='thin']|
        @css:border-right-width[.='thin']"
        priority="2">        
        <xsl:attribute name="{local-name()}" select="$border-width-thin"/>            
    </xsl:template>
    
    <xsl:template match="
        @css:border-width[.='medium']|
        @css:border-bottom-width[.='medium']|
        @css:border-top-width[.='medium']|
        @css:border-left-width[.='medium']|
        @css:border-right-width[.='medium']"
        priority="2">        
        <xsl:attribute name="{local-name()}" select="$border-width-medium"/>        
    </xsl:template>
    <xsl:template match="
        @css:border-style[not(../@css:border-width)]|
        @css:border-bottom-style[not(../@css:border-bottom-width)]|
        @css:border-top-style[not(../@css:border-top-width)]|
        @css:border-left-style[not(../@css:border-left-width)]|
        @css:border-right-style[not(../@css:border-right-width)]"
        priority="2">        
        <xsl:attribute name="{substring-before(local-name(),'-style')}-width" select="$border-width-medium"/>        
        <xsl:next-match/>
    </xsl:template>    
    
    <xsl:template match="
        @css:border-width[.='thick']|
        @css:border-bottom-width[.='thick']|
        @css:border-top-width[.='thick']|
        @css:border-left-width[.='thick']|
        @css:border-right-width[.='thick']"
        priority="2">        
        <xsl:attribute name="{local-name()}" select="$border-width-thick"/>        
    </xsl:template>
    
    <!-- 
    
    	Borders conditionality.
    	
     -->  
    <xsl:template match="@css:oxy-borders-conditionality">
        <xsl:attribute name="border-before-width.conditionality" select="../@css:oxy-borders-conditionality"/>
        <xsl:attribute name="border-after-width.conditionality" select="../@css:oxy-borders-conditionality"/>
    </xsl:template>
    
    <xsl:template match="@css:border-top-width[oxy:is-border-conditionality(..)]" priority="4">
        <xsl:variable name="nm">
            <ret>
                <xsl:next-match/>
            </ret>
        </xsl:variable>
        <xsl:attribute name="border-before-width" select="$nm/ret/@border-top-width"/>
    </xsl:template> 
    <xsl:template match="@css:border-top-style[oxy:is-border-conditionality(..)]" priority="4">
        <xsl:variable name="nm">
            <ret>
                <xsl:next-match/>
            </ret>
        </xsl:variable>
        <xsl:attribute name="border-before-style" select="$nm/ret/@border-top-style"/>
    </xsl:template> 
    <xsl:template match="@css:border-top-color[oxy:is-border-conditionality(..)]" priority="4">
        <xsl:variable name="nm">
            <ret>
                <xsl:next-match/>
            </ret>
        </xsl:variable>
        <xsl:attribute name="border-before-color" select="$nm/ret/@border-top-color"/>
    </xsl:template> 

    <xsl:template match="@css:border-bottom-width[oxy:is-border-conditionality(..)]" priority="4">
        <xsl:variable name="nm">
            <ret>
                <xsl:next-match/>
            </ret>
        </xsl:variable>
        <xsl:attribute name="border-after-width" select="$nm/ret/@border-bottom-width"/>
    </xsl:template> 
    <xsl:template match="@css:border-bottom-style[oxy:is-border-conditionality(..)]" priority="4">
        <xsl:variable name="nm">
            <ret>
                <xsl:next-match/>
            </ret>
        </xsl:variable>
        <xsl:attribute name="border-after-style" select="$nm/ret/@border-bottom-style"/>
    </xsl:template> 
    <xsl:template match="@css:border-bottom-color[oxy:is-border-conditionality(..)]" priority="4">
        <xsl:variable name="nm">
            <ret>
                <xsl:next-match/>
            </ret>
        </xsl:variable>
        <xsl:attribute name="border-after-color" select="$nm/ret/@border-bottom-color"/>
    </xsl:template> 

    <xsl:function name="oxy:is-border-conditionality" as="xs:boolean">
        <xsl:param name="elem"/>
        <xsl:sequence select="
            if ($elem/@css:oxy-borders-conditionality = 'inherit') then
            $elem/ancestor-or-self::*/@css:oxy-borders-conditionality[.='retain']
            else 
            $elem/@css:oxy-borders-conditionality = 'retain'"/>
    </xsl:function>
    
    <!-- 
    
    	Border radius.
    	
     -->
    <xsl:template match="@css:border-top-left-radius">
        <xsl:attribute name="fox:border-before-start-radius" select="."/>
    </xsl:template>
    
    <xsl:template match="@css:border-top-right-radius">
        <xsl:attribute name="fox:border-before-end-radius" select="."/>
    </xsl:template>
    
    <xsl:template match="@css:border-bottom-right-radius">
        <xsl:attribute name="fox:border-after-end-radius" select="."/>
    </xsl:template>
    
    <xsl:template match="@css:border-bottom-left-radius">
        <xsl:attribute name="fox:border-after-start-radius" select="."/>
    </xsl:template>

</xsl:stylesheet>
