<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet 
    xmlns:xsl="http://www.w3.org/1999/XSL/Transform" 
    xmlns:xs="http://www.w3.org/2001/XMLSchema" 
    xmlns:fo="http://www.w3.org/1999/XSL/Format" 
    xmlns:oxy="http://www.oxygenxml.com/css2fo" 
    xmlns:css="http://www.w3.org/1998/CSS" 
    exclude-result-prefixes="xs oxy css " version="2.0"
    xpath-default-namespace="http://www.w3.org/1998/CSS">
    
    <!-- 
        Gets the size of the label, large enaugh to accomodate rendering a number,
        in a specific format.
        
        @param number The number
        @param css-format The number format, it is one of the CSS list style values: lower-latin, upper-alpha, decimal, etc..
        @return a width.
        Context: none.
    -->
    <xsl:function name="oxy:get-number-width" as="xs:string">
        <xsl:param name="number" />
        <xsl:param name="css-format" />
        <xsl:choose>
            <xsl:when test="
                $css-format = 'upper-alpha' or 
                $css-format = 'upper-latin' or
                $css-format = 'lower-alpha' or 
                $css-format = 'lower-latin' or           
                $css-format = 'lower-greek'">
                <xsl:choose>
                    <xsl:when test="$number &lt; 27">
                        <!-- The alphabet size is 27 -->
                        <xsl:value-of select="'1.5em'"/>
                    </xsl:when>
                    <xsl:when test="$number &lt; 54">
                        <xsl:value-of select="'2.5em'"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="'3em'"/>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:when>
            <xsl:when test="$css-format='lower-roman'">
                <!-- TODO - dan - maybe with an extension to compute the number of letters:
                http://stackoverflow.com/questions/17479287/convert-int-to-roman-numeral -->
                <xsl:choose>
                    <xsl:when test="$number &lt; 7">
                        <xsl:value-of select="'1.5em'"/>
                    </xsl:when>
                    <xsl:when test="$number &lt; 17">
                        <xsl:value-of select="'2em'"/>
                    </xsl:when>
                    <xsl:when test="$number &lt; 27">
                        <xsl:value-of select="'2.5em'"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="'3.5em'"/>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:when>
            <xsl:when test="$css-format='upper-roman'">
                <!-- TODO - dan - maybe with an extension to compute the number of letters:
                http://stackoverflow.com/questions/17479287/convert-int-to-roman-numeral -->
                <xsl:choose>
                    <xsl:when test="$number&lt;7">
                        <xsl:value-of select="'2em'"/>
                    </xsl:when>
                    <xsl:when test="$number&lt;17">
                        <xsl:value-of select="'2.5em'"/>
                    </xsl:when>
                    <xsl:when test="$number&lt;27">
                        <xsl:value-of select="'3em'"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="'4em'"/>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:when>
            <xsl:otherwise>
                <xsl:choose>
                    <!-- Decimal -->
                    <xsl:when test="$number &lt; 10">
                        <xsl:value-of select="'1.5em'"/>
                    </xsl:when>
                    <xsl:when test="$number &lt; 100">
                        <xsl:value-of select="'2em'"/>
                    </xsl:when>
                    <xsl:when test="$number &lt; 1000">
                        <xsl:value-of select="'2.5em'"/>
                    </xsl:when>
                    <xsl:when test="$number &lt; 10000">
                        <xsl:value-of select="'3em'"/>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="'4em'"/>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:function>
    
    <!-- 
        Converts a CSS counter number format, to a FO format.
        counter(page, lower-alpha) -> "a"
    
        @param css-format The number format, it is one of the CSS list style values: 
            lower-latin, upper-alpha, decimal, etc..
        @return the FO number format
        
        Context: none.
    -->
    <xsl:function name="oxy:get-number-format">
        <xsl:param name="css-format"/>
        <xsl:choose>
            <xsl:when test="$css-format = 'decimal'">
                <!-- This is the default.  -->
                <xsl:value-of select="'1'"/>
            </xsl:when>
            <xsl:when test="$css-format = 'lower-roman' or $css-format = 'lower-latin'">
                <xsl:value-of select="'i'"/>
            </xsl:when>
            <xsl:when test="$css-format = 'upper-roman' or $css-format = 'upper-latin'">
                <xsl:value-of select="'I'"/>
            </xsl:when>
            <xsl:when test="$css-format = 'lower-alpha'">
                <xsl:value-of select="'a'"/>
            </xsl:when>
            <xsl:when test="$css-format = 'upper-alpha'">
                <xsl:value-of select="'A'"/>
            </xsl:when>
            <xsl:when test="$css-format = 'lower-greek'">
                <xsl:value-of select="'α'"/>
            </xsl:when>
            <xsl:when test="$css-format = 'upper-greek'">
                <xsl:value-of select="'Α'"/>
            </xsl:when>
            <xsl:when test="$css-format = 'decimal-leading-zero'">
                <xsl:value-of select="'01'"/>
            </xsl:when>
            <xsl:when test="$css-format instance of xs:string">
                <xsl:value-of select="'string'"/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:value-of select="'1'"/>
                <xsl:message>WARNING The page number format specified for the page counter is not supported: "<xsl:value-of select="$css-format"/>".</xsl:message>
            </xsl:otherwise>                
        </xsl:choose>
    </xsl:function>
    
    <!-- 
        Gets a counter reset value set on a page declaration. 
    
        It supports declaring multiple counters:
            
            counter-reset: footnote section 1 figure;
              /* Sets the footnote and figure counters to 0
                and the section counter to 1. */
    
        @param doc The document
        @param page-name The name of the page.
        @param counter-name The name of the counter. May be 'page', or 'footnote', etc.. 
        @return a number if there is a counter-reset property that refers to the 'footnote' counter, 
                or -1 otherwise.
    -->
    <xsl:function name="oxy:get-counter-reset-from-page-def" as="xs:integer">
        <xsl:param name="doc"/>
        <xsl:param name="page-name" />
        <xsl:param name="counter-name" />
        
        <!-- The last one is the most specific. -->
        <xsl:variable name="matching-pages" select="oxy:get-matching-pages($doc,$page-name, 'any', 'any', 'any')"/>
        
        <xsl:variable name="all-counter-reset" 
            select="$matching-pages/css:property[@name='counter-reset']/@value"/>
        
        <!-- Separate all with spaces. -->
        <xsl:variable name="all-counter-reset-with-spaces" select="string-join($all-counter-reset, ' ')"/>
        
        <xsl:value-of select="oxy:get-counter-reset($all-counter-reset-with-spaces, $counter-name)"/>
    </xsl:function>
    
    <!--        
        Gets a counter reset value from a string representing the value of a 'counter-reset' property. 
        It supports declaring multiple counters:
            
            counter-reset: footnote section 1 figure;
              /* Sets the footnote and figure counters to 0
                and the section counter to 1. */        
    
        @param counter-reset The value of the 'counter-reset' property. 
        @param counter-name The name of the counter. May be 'page', or 'footnote', etc.. 
        @return a number if there is a counter-reset property that refers to the 'footnote' counter, 
                or -1 otherwise.
    -->
    <xsl:function name="oxy:get-counter-reset" as="xs:integer">
        <xsl:param name="counter-reset-prop-val" />
        <xsl:param name="counter-name" />
        <xsl:variable name="counter-reset">                
            <xsl:choose>
                <xsl:when test="$counter-reset-prop-val">           
                    <!-- Refine the counter reset expression to find out the token that follows the 'footnote' name. -->        
                    
                    <!-- We reverse the toke list, because there is no last-index-of in XSLT. -->
                    <xsl:variable name="tokens" select="tokenize($counter-reset-prop-val,' ')"/>
                    <xsl:variable name="position-of-counter-name" select="index-of($tokens, $counter-name)[last()]"/>
                    
                    <xsl:choose>
                        <xsl:when test="exists($position-of-counter-name)">
                            <!-- The value of the counter reset, a token after the counter name, but in a a reversed list. -->
                            <xsl:variable name="cr" select="$tokens[$position-of-counter-name + 1]"/>
                            <xsl:value-of  
                                    select="
                                    if (not ($cr))
                                    then 
                                        0
                                    else 
                                        if (number($cr) = number($cr))
                                        then 
                                            $cr
                                        else 
                                            0"/>                            
                        </xsl:when>
                        <xsl:otherwise>
                            <!-- No searched counter name in the expression. -->
                            <xsl:value-of select="number(-1)"/>
                        </xsl:otherwise>
                    </xsl:choose>     
                </xsl:when>                    
                <xsl:otherwise>
                    <!-- No footnote counter in the expression. -->
                    <xsl:value-of select="number(-1)"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        
        <xsl:value-of select="$counter-reset"/>
    </xsl:function>
    
    <!-- 
        Computes the value of a dimension, in case the value is a percent.
        @param sd The size dimension
        @param reference The reference dimension.
        
        @return An expression.
    -->
    <xsl:function name="oxy:expand-percent">
        <xsl:param name="sd"/>
        <xsl:param name="reference"/>
        <xsl:choose>
            <xsl:when test="ends-with($sd, '%') and $reference">
                <xsl:value-of select="concat( substring-before($sd,'%') , ' * ', $reference, ' div 100')"/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:value-of select="$sd"/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:function>
    
</xsl:stylesheet>