<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    xmlns:css="http://www.w3.org/1998/CSS"
    xmlns:fo="http://www.w3.org/1999/XSL/Format"
    xmlns:oxy="http://www.oxygenxml.com/css2fo" 
    exclude-result-prefixes="xs oxy css"
    version="2.0">    
    
    <xsl:template match="@css:list-style" priority="2"/>
    <xsl:template match="@css:list-style-image" priority="2"/>
    <xsl:template match="@css:list-style-position" priority="2"/>
    <xsl:template match="@css:list-style-type" priority="2"/>
    <xsl:template match="@css:extracted-list-style-image-width" priority="2"/>
    <xsl:template match="@css:extracted-list-style-image-height" priority="2"/>
    
    <!-- Matches the list items. -->
    <xsl:template match="*[@css:display = 'list-item']" priority="3">
        <!-- The changebars that are bordering the list item content 
            are moved outside the list-item for a better rendering. The changebars 
            look better if they are outside the list-item: they are not overlapping 
            the marker, are placed at the list left margin. -->
        <xsl:apply-templates select="*[@css:display = '-oxy-changebar-start'][position() = 1]">
            <xsl:with-param name="generate" select="'yes'" tunnel="yes"/>
        </xsl:apply-templates>
        
        <xsl:variable name="list-style-position" select="oxy:get-list-style-position(.)"/>
        <xsl:variable name="fallback-fonts" select="'Times New Roman,Liberation Serif,DejaVu Serif,Symbol,Segoe UI Symbol,FreeSerif,ZapfDingbats'"/>
        <xsl:call-template name="list-item">
            <xsl:with-param name="list-item" select="."/>
            <xsl:with-param name="list-style-position" select="$list-style-position"/>
            <xsl:with-param name="marker">
                <xsl:choose>
                    <xsl:when test="$list-style-position = 'outside'">
                        <fo:block text-align="end">
                            <xsl:attribute name="font-family" select="
                                if (css:marker/@css:font-family) then
                                    concat(css:marker/@css:font-family, ',', $fallback-fonts)
                                else
                                    $fallback-fonts"/>
                            <!--
                                Copy the color, but not the background color.
                            -->
                            <xsl:apply-templates select="@css:*" mode="copy-marker-properties-from-parent"/>
                            <!--
                                We copy (and possible override) the above attributes, taken from the :marker pseudo, if
                                the pseudo marker has no content of its own (it will not generate a FO element with its attributes).
                            -->
                            <xsl:if test="not(css:marker/*)">
                                <xsl:apply-templates select="css:marker/@css:* except @css:display"/>
                            </xsl:if>
                            
                            <xsl:call-template name="list-label">
                                <xsl:with-param name="list-type" select="oxy:get-list-style-type(.)"/>
                            </xsl:call-template>
                        </fo:block>
                    </xsl:when>
                    <xsl:otherwise>
                        <fo:inline text-align="end">
                            <xsl:attribute name="font-family" select="
                                if (css:marker/@css:font-family) then
                                    concat(css:marker/@css:font-family, ',', $fallback-fonts)
                                else
                                    $fallback-fonts"/>
                            <!--
                                We copy (and possible override) the above attributes, taken from the :marker pseudo, if
                                the pseudo marker has no content of its own (it will not generate a FO element with its attributes).
                            -->
                            <xsl:if test="not(css:marker/*)">
                                <xsl:apply-templates select="css:marker/@css:* except @css:display"/>
                            </xsl:if>
                            <xsl:call-template name="list-label">
                                <xsl:with-param name="list-type" select="oxy:get-list-style-type(.)"/>
                            </xsl:call-template>
                        </fo:inline>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:with-param>
            <xsl:with-param name="content">
                <xsl:choose>
                    <xsl:when test="*[@css:display = 'list-item']">
                        <!-- 
                            There are list items directly contained in the current list item. 
                            We need to wrap them in a list. 
                        -->
                        <xsl:next-match>
                            <!--
                                No styling attributes should be set 
                                on the generated list-block since the 
                                list item was already styled. 
                            -->
                            <xsl:with-param name="generate-attributes" select="false()" tunnel="yes"/>
                        </xsl:next-match>
                    </xsl:when>
                    <xsl:otherwise>
                        <!-- 
                            Use the children.
                        -->
                        <xsl:apply-templates select="node() except css:marker"/>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:with-param>
        </xsl:call-template>
        
        <!-- The changebars that are bordering the list item content 
            are moved outside the list-item for a better rendering. -->
        <xsl:apply-templates select="*[@css:display = '-oxy-changebar-end'][position() = last()]">
            <xsl:with-param name="generate" select="'yes'" tunnel="yes"/>
        </xsl:apply-templates>
    </xsl:template>
    
    <!-- 
        The changebars that are bordering the list item content 
        are moved outside the list-item for a better rendering. 
        We let them generate change bar markup only when this template 
        is applied from the list-item template. 
        
        @param generate Set to 'yes' to allow generation.
    -->
    <xsl:template match="
        *[@css:display = 'list-item']/*[@css:display = '-oxy-changebar-start'][position() = 1] |
        *[@css:display = 'list-item']/*[@css:display = '-oxy-changebar-end'][position() = last()]">
        <xsl:param name="generate" tunnel="yes" select="'no'"/>
        <xsl:if test="$generate = 'yes'">
            <xsl:next-match/>
        </xsl:if>
    </xsl:template>
    
    <!-- 
        This template generates a list item. This is used both from the templates matching 
        the elements with display list-item, but also from the footnotes.
        
        @param list-item The element that has a list-item display type.
        @param list-style-position One of the CSS values: inside, outside.
        @param marker The generated FO content for the marker.
        @param content The generated FO content for the list item content.
    -->
    <xsl:template name="list-item">
        <xsl:param name="list-item" required="yes"/>
        <xsl:param name="list-style-position" required="yes"/>
        <xsl:param name="marker" required="yes"/>
        <xsl:param name="content" required="yes"/>
        
        <fo:list-item>
            <xsl:apply-templates select="@css:margin-top | @css:margin-bottom | @css:margin-left | @css:margin-right"/>
            <xsl:choose>
                <xsl:when test="$list-style-position = 'outside'">
                    <!-- The position of the list decorator is outside, use the fo:list-item-label -->
                    <fo:list-item-label end-indent="label-end()">
                        <xsl:copy-of select="$marker"/>
                    </fo:list-item-label>
                    
                    <fo:list-item-body start-indent="body-start()">
                        <fo:block>
                            <!-- 
                                Copy attributes.
                            -->
                            <xsl:apply-templates select="@* except (@css:margin-left, @css:margin-right, @css:margin-top, @css:margin-bottom)"/>
                            <xsl:if test="@css:padding-left">
                                <xsl:attribute name="margin-left" select="'0'"/>
                            </xsl:if>
                            
                            <xsl:choose>
                                <xsl:when test="$list-item/*[oxy:margin-fixable(.)]">
                                    <!-- This container and block defines a new reference area, so the possible 
                                        nested block containers will not bleed their content outside the list-item. -->
                                    <fo:block-container>
                                        <fo:block start-indent="0" end-indent="0">
                                            
                                            <xsl:copy-of select="$content"/>
                                        </fo:block>
                                    </fo:block-container>
                                </xsl:when>
                                <xsl:otherwise>
                                    <!-- Pure text or PI's comments.-->
                                    <xsl:copy-of select="$content"/>
                                </xsl:otherwise>
                            </xsl:choose>
                        </fo:block>
                    </fo:list-item-body>
                </xsl:when>
                <xsl:otherwise>
                    <!-- The position of the list decorator is inside, use an inline -->
                    <fo:list-item-label end-indent="0">
                        <fo:block/>
                    </fo:list-item-label>
                    <fo:list-item-body start-indent="body-start()">
                        <fo:block>
                            <xsl:apply-templates select="@* except (@css:margin-left, @css:margin-right, @css:margin-top, @css:margin-bottom)"/>
                            <xsl:if test="@css:padding-left">
                                <xsl:attribute name="margin-left" select="'0'"/>
                            </xsl:if>
                            
                            <xsl:choose>
                                <xsl:when test="$list-item/*[oxy:margin-fixable(.)]">
                                    
                                    <!-- This inline container and block defines a new reference area, so the possible 
                                        nested block containers will not bleed their content outside the list-item. -->
                                    <fo:inline-container>
                                        <fo:block start-indent="0" end-indent="0">
                                            <xsl:copy-of select="$marker"/>
                                            <xsl:copy-of select="$content"/>
                                        </fo:block>
                                    </fo:inline-container>
                                </xsl:when>
                                
                                <xsl:otherwise>
                                    <!-- Pure text or PI's comments.-->
                                    <xsl:copy-of select="$marker"/>
                                    <xsl:copy-of select="$content"/>
                                </xsl:otherwise>
                            </xsl:choose>
                        </fo:block>
                    </fo:list-item-body>
                </xsl:otherwise>
            </xsl:choose>
        </fo:list-item>
    </xsl:template>
    
    <!-- 
        The list is in a context with a direction. In this case, it should 
        be wrapped by a block container to impose the direction. 
    -->
    <xsl:template match="*[*[@css:display = 'list-item']][@css:direction]" priority="3">
        <fo:bidi-override direction="ltr" unicode-bidi="embed">
            <fo:block-container writing-mode="{oxy:get-writing-mode(@css:direction)}">
                <fo:block start-indent="0" end-indent="0">
                    <xsl:next-match/>
                </fo:block>
            </fo:block-container>
        </fo:bidi-override>
    </xsl:template>
    
    <!-- 
        The list. 
        
        @param generate-attributes If true (this is the default value), styles the list using the attributes set on the element containing the list items. 
        Can be set to false if the styling was already done on a container.
    -->
    <xsl:template match="*[*[@css:display = 'list-item']]" priority="2">
        <xsl:param name="generate-attributes" tunnel="yes" select="true()"/>
        
        <xsl:variable name="first-item" select="*[@css:display = 'list-item'][1]"/>
        <xsl:variable name="list-style-type" select="oxy:get-list-style-type($first-item)"/>
        <xsl:variable name="list-style-position" select="oxy:get-list-style-position($first-item)"/>
        <xsl:variable name="list-padding" select="$first-item/parent::*/@css:padding-left"/>
        <xsl:variable name="list-item-margin" select="$first-item/@css:margin-left"/>
        <xsl:variable name="list-item-marker" select="$first-item/css:marker"/>
        
        <xsl:variable name="is-outside-label" select="($list-style-type != 'none' or $list-item-margin or $list-item-marker/@css:width) and $list-style-position = 'outside'"/>
        
        <xsl:variable name="provisional-label-separation">
            <xsl:choose>
                <xsl:when test="$is-outside-label">
                    <xsl:text>3pt</xsl:text>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:text>0</xsl:text>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="distance-between-starts">
            <xsl:choose>
                <xsl:when test="$is-outside-label">
                    <xsl:call-template name="get-list-distance-between-starts">
                        <xsl:with-param name="list-type" select="$list-style-type"/>
                        <xsl:with-param name="list-padding" select="$list-padding"/>
                        <xsl:with-param name="list-item-marker" select="$list-item-marker"/>
                    </xsl:call-template>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:choose>
                        <xsl:when test="$list-padding">
                            <xsl:value-of select="$list-padding"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:text>0</xsl:text>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="padding-left">
            <xsl:call-template name="get-list-padding-left">
                <xsl:with-param name="list-padding" select="$list-padding"/>
                <xsl:with-param name="list-item-margin" select="$list-item-margin"/>
                <xsl:with-param name="list-item-marker" select="$list-item-marker"/>
            </xsl:call-template>
        </xsl:variable>
        
        <xsl:choose>
            <xsl:when test="*[@css:display != 'list-item']">
                <!-- 
                    It has mixed children, blocks or inlines together with list items.
                    In this case generate a block container, and a list for each sequence of items.
                -->
                <fo:block-container>
                    
                    <xsl:if test="$generate-attributes">
                        <!-- If the list is also a list item, its attributes were already used. -->
                        <xsl:apply-templates select="@* except @css:padding-left"/>
                    </xsl:if>
                    
                    <fo:block end-indent="0" start-indent="0">
                        <xsl:for-each-group select="*" group-adjacent="@css:display">
                            <xsl:choose>
                                <xsl:when test="current-grouping-key() = 'list-item'">
                                    <!-- List -->
                                    <fo:list-block relative-align="baseline">
                                        <xsl:attribute name="provisional-label-separation" select="$provisional-label-separation"/>
                                        <xsl:attribute name="provisional-distance-between-starts" select="$distance-between-starts"/>
                                        <xsl:if test="string($padding-left)">
                                            <!-- 
                                                We need to specify both the padding left and the start indent, if the
                                                start-indent is missing, the padding left will bleed to the left, enlarging
                                                the box to the left. 
                                            -->
                                            <xsl:attribute name="padding-left" select="$padding-left"/>
                                            <xsl:attribute name="start-indent" select="$padding-left"/>
                                        </xsl:if>
                                        <xsl:for-each select="current-group()">
                                            <xsl:apply-templates select="."/>
                                        </xsl:for-each>
                                    </fo:list-block>
                                </xsl:when>
                                <xsl:otherwise>
                                    <!-- Other -->
                                    <xsl:for-each select="current-group()">
                                        <xsl:apply-templates select="."/>
                                    </xsl:for-each>
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:for-each-group>
                        
                    </fo:block>
                </fo:block-container>
            </xsl:when>
            <xsl:otherwise>
                <!-- 
                    Is a simple list, contains only items. 
                -->
                <fo:list-block relative-align="baseline">
                    <xsl:attribute name="provisional-label-separation" select="$provisional-label-separation"/>
                    <xsl:attribute name="provisional-distance-between-starts" select="$distance-between-starts"/>
                    <xsl:if test="$generate-attributes">
                        <!-- If the list is also a list item, its attributes were already used. -->
                        <xsl:apply-templates select="@* except @css:padding-left"/>
                        <xsl:if test="string($padding-left)">
                            <!-- 
                                We need to specify both the padding left and the start indent, if the
                                start-indent is missing, the padding left will bleed to the left, enlarging
                                the box to the left. 
                            -->
                            <xsl:attribute name="padding-left" select="$padding-left"/>
                            <xsl:attribute name="start-indent" select="$padding-left"/>
                        </xsl:if>
                    </xsl:if>
                    <xsl:apply-templates/>
                </fo:list-block>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- 
        Gets the inside/outside positioning of the context list item. 
        Looks also at properties set on parents, to implement inheritence.
        
        @param item The list item
        @return The 'outside' string for outside placement of the marker, other value for inside placement.
    -->
    <xsl:function name="oxy:get-list-style-position" as="xs:string">
        <xsl:param name="item"/>
        
        <xsl:variable name="list-style-position" select="$item/ancestor-or-self::*[@css:list-style-position != 'inherit'][1]/@css:list-style-position"/>
        
        <xsl:value-of select="
            if (string-length($list-style-position) = 0)
            then
            'outside'
            else
            $list-style-position"/>
    </xsl:function>
    
    <!-- 
        Converts the margin/padding value '0' to '0px'. 
        
        @param value The value of margin/padding
        @return The same value or 0px if the value was 0
    -->
    <xsl:function name="oxy:convert-zero-to-unit" as="xs:string">
        <xsl:param name="value" as="xs:string"/>
        
        <xsl:value-of select="
            if ($value = '0')
            then
            '0px'
            else
            $value"/>
    </xsl:function>
    
    <!-- 
        Gets the CSS list-style-type property in the context of list item. 
        Looks also to the parents.
        
        @param item The list item to find the property for.
        @return A string 
    -->
    <xsl:function name="oxy:get-list-style-type" as="xs:string">
        <xsl:param name="item"/>
        <xsl:variable name="att" select="$item/ancestor-or-self::*[@css:list-style-type != 'inherit' or @css:list-style-image != 'inherit'][1]"/>
        
        <xsl:choose>
            <xsl:when test="$att/@css:list-style-image">
                <xsl:text>image</xsl:text>
            </xsl:when>
            <xsl:otherwise>
                <xsl:variable name="list-style-type" select="$att/@css:list-style-type"/>
                <xsl:choose>
                    <xsl:when test="string-length($list-style-type) = 0">
                        <xsl:text>disc</xsl:text>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="$list-style-type"/>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:function>
    
    <xsl:template name="get-list-distance-between-starts">
        <xsl:param name="list-type"/>
        <xsl:param name="list-padding"/>
        <xsl:param name="list-item-marker"/>
        
        <xsl:variable name="marker-width" select="oxy:get-number-width(count(*[@css:display = 'list-item']), $list-type)"/>
        
        <!-- Use the marker width from CSS, otherwise compute one. -->
        <xsl:variable name="marker-mbp">
            <xsl:for-each select="
                $list-item-marker/@css:margin-left | 
                $list-item-marker/@css:margin-right | 
                $list-item-marker/@css:padding-left | 
                $list-item-marker/@css:padding-right">
                
                <xsl:value-of select="."/>
                <xsl:if test="not(position() = last())"> + </xsl:if>
            </xsl:for-each>
        </xsl:variable>
        <xsl:variable name="list-item-marker-width" select="
                if ($list-item-marker/@css:width) then
                    $list-item-marker/@css:width
                else
                    if (string-length(normalize-space($marker-mbp)) > 0) then
                        concat($marker-width, ' + ', $marker-mbp)
                    else
                        ''"/>
        
        <xsl:choose>
            <xsl:when test="$list-item-marker-width">
                <xsl:value-of select="$list-item-marker-width"/>
            </xsl:when>
            <xsl:when test="$list-padding">
                <!-- When there is no marker width, all lengths should go to distance-between-starts -->
                <xsl:value-of select="$list-padding"/>
            </xsl:when>
            <xsl:when test="$list-type = 'image'">
                <!-- When there is no marker width, all lengths should go to distance-between-starts -->
                <xsl:text>0pt</xsl:text>
            </xsl:when>
            <xsl:when test="$list-type = 'disc' or $list-type = 'circle' or $list-type = 'square' or $list-type = 'box' or $list-type = 'check' or $list-type = 'diamond' or $list-type = 'hyphen'">
                <xsl:text>1em</xsl:text>
            </xsl:when>
            <xsl:otherwise>
                <xsl:value-of select="$marker-width"/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <xsl:template name="get-list-padding-left">
        <xsl:param name="list-padding"/>
        <xsl:param name="list-item-margin"/>
        <xsl:param name="list-item-marker"/>
        
        <xsl:if test="$list-padding and $list-item-marker/@css:width">
            <xsl:value-of select="
                concat(oxy:convert-zero-to-unit($list-padding), ' - ',
                oxy:convert-zero-to-unit($list-item-marker/@css:width),
                    if ($list-item-margin) then
                        concat(' + ', oxy:convert-zero-to-unit($list-item-margin))
                    else
                        '')"/>
        </xsl:if>            
    </xsl:template>
    
    <!--
        Generates the label of a list item.
        The context is the list item.
        
        @param list-type The type of the decorator.
        @return the string content of the decorator.
    -->
    <xsl:template name="list-label">
        <xsl:param name="list-type"/>
        <xsl:choose>
            <xsl:when test="css:marker/*">
                <!-- In case we have a marker content, use it. -->
                <xsl:apply-templates select="css:marker"/>
            </xsl:when>
            <xsl:when test="$list-type = 'image'">
                <xsl:call-template name="list-label-image"/>
            </xsl:when>
            <xsl:when test="$list-type = 'box'">
                <xsl:text>&#9633; </xsl:text>
            </xsl:when>
            <xsl:when test="$list-type = 'check'">
                <!-- This requires Segoe UI -->
                <xsl:text>&#10003; </xsl:text>
            </xsl:when>
            <xsl:when test="$list-type = 'circle'">
                <xsl:text>&#9702; </xsl:text>
            </xsl:when>
            <xsl:when test="$list-type = 'diamond'">
                <xsl:text>&#9830; </xsl:text>
            </xsl:when>
            <xsl:when test="$list-type = 'disc'">
                <xsl:text>&#8226; </xsl:text>
            </xsl:when>
            <xsl:when test="$list-type = 'hyphen'">
                <xsl:text>&#8211; </xsl:text>
            </xsl:when>
            <xsl:when test="$list-type = 'square'">
                <xsl:text>&#9642; </xsl:text>
            </xsl:when>
            <xsl:when test="$list-type = 'none'">
                <!-- No content. -->
            </xsl:when>
            <xsl:otherwise>
                <xsl:variable name="format" select="oxy:get-number-format($list-type)"/>
                <xsl:choose>
                    <xsl:when test="$format = 'string'">
                        <xsl:value-of select="$list-type"/>
                    </xsl:when>
                    <xsl:otherwise><xsl:number format="{$format}" value="count(preceding-sibling::*[@css:display = 'list-item']) + 1"/>. </xsl:otherwise>
                </xsl:choose>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- 
        Generates a fo:external graphic for the list items that have a list-style-image property.
        The context is the list item.
    -->
    <xsl:template name="list-label-image">
        <fo:external-graphic>
            <xsl:variable name="ancestor-with-list-style-image" select="ancestor-or-self::*[@css:list-style-image != 'inherit'][1]"/>
            <xsl:attribute name="src" select="$ancestor-with-list-style-image/@css:list-style-image"/>
            
            <xsl:variable name="extracted-image-width" select="$ancestor-with-list-style-image/@css:extracted-list-style-image-width"/>
            <xsl:variable name="extracted-image-height" select="$ancestor-with-list-style-image/@css:extracted-list-style-image-height"/>
            <xsl:variable name="image-resolution" select="oxy:extract-dpi-value($ancestor-with-list-style-image/ancestor-or-self::*/@css:image-resolution[1])"/>
            
            <xsl:if test="string-length($image-resolution) > 0">
                <xsl:attribute name="scaling" select="'uniform'"/>
                <xsl:attribute name="width" select="concat($extracted-image-width, 'in div ', $image-resolution)"/>
                <xsl:attribute name="content-width" select="'scale-to-fit'"/>
                <xsl:attribute name="height" select="concat($extracted-image-height, 'in div ', $image-resolution)"/>
                <xsl:attribute name="content-height" select="'scale-to-fit'"/>
            </xsl:if>
            
        </fo:external-graphic>
    </xsl:template>
    
    <xsl:template match="
        @css:color |
        @css:line-height |
        @css:font-size |
        @css:font-family |
        @css:font-weight |
        @css:font-style" mode="copy-marker-properties-from-parent">
        <xsl:attribute name="{local-name(.)}" select="."/>
    </xsl:template>
    
    <xsl:template match="@css:*" mode="copy-marker-properties-from-parent"/>
    
</xsl:stylesheet>
