<?xml version="1.0" encoding="UTF-8"?>
<!--
    
    This stylesheet deals with background images.
    
-->
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    xmlns:css="http://www.w3.org/1998/CSS"    
    xmlns:fo="http://www.w3.org/1999/XSL/Format"
    xmlns:fox="http://xmlgraphics.apache.org/fop/extensions"
    xmlns:oxy="http://www.oxygenxml.com/css2fo"
    
    exclude-result-prefixes="xs oxy fox"
    version="2.0">
    
    <!-- 
        Copies the background image CSS attribute. 
        In addition, it processes the 'image-resolution' and possible the 'background-size', this resulting 
        in setting some FOP extension attributes.
        
        @see "https://drafts.csswg.org/css-backgrounds-3/#the-background-size" 
    -->
    <xsl:template match="@css:background-image">        
        <xsl:attribute name="background-image" select="."/>
        
        <xsl:call-template name="generate-background-image-size-attributes">
            <xsl:with-param name="background-size" select="../@css:background-size"/>
            <xsl:with-param name="parent-width" select="../@css:width"/>
            <xsl:with-param name="parent-height" select="../@css:height"/>
            <xsl:with-param name="extracted-background-image-width" select="../@css:extracted-background-image-width"/>
            <xsl:with-param name="extracted-background-image-height" select="../@css:extracted-background-image-height"/>
            <xsl:with-param name="image-resolution" select="oxy:extract-dpi-value(ancestor-or-self::*/@css:image-resolution[1])"/>            
        </xsl:call-template>
    </xsl:template>
    
    <!--
        Generates the attributes needed for the background images using the FOP extensions:
        
        fox:background-image-width
        fox:background-image-height
        
        These are calculated depending on the image resolution and the specified size or the parent.
    -->
    <xsl:template name="generate-background-image-size-attributes">
        <xsl:param name="background-size" required="yes"/>
        <xsl:param name="parent-width" required="yes"/>
        <xsl:param name="parent-height" required="yes"/>
        <xsl:param name="extracted-background-image-width" required="yes"/>
        <xsl:param name="extracted-background-image-height" required="yes"/>
        <xsl:param name="image-resolution" required="yes"/>
        
        
        <xsl:choose>
            <xsl:when test="$background-size">
                <!-- A size was specified. -->
                <xsl:variable name="tokens" select="tokenize(normalize-space($background-size), ' ')"/>
                <xsl:choose>
                    <xsl:when test="count($tokens) = 1">
                        <xsl:attribute name="fox:background-image-width" select="oxy:expand-percent($tokens[1], $parent-width)"/>                        
                    </xsl:when>
                    <xsl:when test="count($tokens) = 2">
						<xsl:attribute name="fox:background-image-width" select="oxy:expand-percent($tokens[1], $parent-width)"/>
                        <xsl:attribute name="fox:background-image-height" select="oxy:expand-percent($tokens[2], $parent-height)"/>                        
                    </xsl:when>                    
                </xsl:choose>
            </xsl:when>
            <xsl:when test="$extracted-background-image-width and $extracted-background-image-height">
                <!-- No size, try determining the real size from the extracted dimensions. -->
                <xsl:if test="string-length($image-resolution) > 0">
                    <xsl:attribute name="fox:background-image-width" select="concat($extracted-background-image-width, 'in div ', $image-resolution)"/>
                    <xsl:attribute name="fox:background-image-height" select="concat($extracted-background-image-height, 'in div ', $image-resolution)"/>
                </xsl:if>
            </xsl:when>
        </xsl:choose>       
    </xsl:template>
    
    
    <!-- These should not get into the output. -->
    <xsl:template match="@css:extracted-background-image-width"/>
    <xsl:template match="@css:extracted-background-image-height"/>
    <xsl:template match="@css:background-size"/>
    
    <!-- 
       This attribute is not treated at all by the FOP. Other FO processors know what it means.
       Fortunately, with FOP we can use @background-position-vertical and @background-position-horizontal.

       @see "https://drafts.csswg.org/css-backgrounds-3/#background-position" 
    -->
    <xsl:template match="@css:background-position">
        <xsl:call-template name="generate-background-image-position-attributes">
            <xsl:with-param name="background-position" select="."/>
        </xsl:call-template>        
    </xsl:template>
    
    
    <!-- 
        Generates the @background-position-vertical and @background-position-horizontal 
        from a CSS "background-position" property value.
    
        @param background-position-value The value of the CSS "background-position" property.
    -->    
    <xsl:template name="generate-background-image-position-attributes">        
        <xsl:param name="background-position" as="xs:string" />
        <xsl:if test="$background-position">
	        <xsl:variable name="tokens" select="tokenize(normalize-space($background-position), ' ')"/>
	
	        <xsl:variable name="nt" select="count($tokens)"/>
	        <xsl:choose>
	            <xsl:when test="$nt = 1">
	                <xsl:choose>
	                    <xsl:when test="$tokens[1] = 'top'">
	                        <xsl:attribute name="background-position-horizontal" select="'center'"/>
	                        <xsl:attribute name="background-position-vertical" select="'top'"/>
	                    </xsl:when>
	                    <xsl:when test="$tokens[1] = 'bottom'">
	                        <xsl:attribute name="background-position-horizontal" select="'center'"/>
	                        <xsl:attribute name="background-position-vertical" select="'bottom'"/>
	                    </xsl:when>
	                    <xsl:when test="$tokens[1] = 'left'">
	                        <xsl:attribute name="background-position-horizontal" select="'left'"/>
	                        <xsl:attribute name="background-position-vertical" select="'center'"/>
	                    </xsl:when>
	                    <xsl:when test="$tokens[1] = 'right'">
	                        <xsl:attribute name="background-position-horizontal" select="'right'"/>
	                        <xsl:attribute name="background-position-vertical" select="'center'"/>
	                    </xsl:when>
	                    <xsl:when test="$tokens[1] = 'center'">
	                        <xsl:attribute name="background-position-horizontal" select="'center'"/>
	                        <xsl:attribute name="background-position-vertical" select="'center'"/>
	                    </xsl:when>
	
	                    <xsl:otherwise>
	                        <!--TODO- test this-->
	                        <xsl:attribute name="background-position-horizontal" select="$tokens[1]"/>
	                        <xsl:attribute name="background-position-vertical" select="'center'"/>
	                    </xsl:otherwise>
	                </xsl:choose>
	            </xsl:when>
	            <xsl:when test="$nt = 2">
	                <xsl:choose>
	                    <xsl:when
	                        test="
	                            translate($tokens[1], '1234567890', '') != $tokens[1] or
	                            translate($tokens[2], '1234567890', '') != $tokens[2]">
	                        <!-- There is percent or length in the expressions. In this case the horizontal comes first. -->
	                        <xsl:attribute name="background-position-horizontal" select="$tokens[1]"/>
	                        <xsl:attribute name="background-position-vertical" select="$tokens[2]"/>
	                    </xsl:when>
	                    <xsl:otherwise>
	                        <!-- Only keywords. They can come in any order. -->
	                        <xsl:choose>
	                            <xsl:when test="$tokens[1] = 'top'">
	                                <xsl:attribute name="background-position-vertical" select="'top'"/>
	                                <xsl:attribute name="background-position-horizontal"
	                                    select="$tokens[2]"/>
	                            </xsl:when>
	                            <xsl:when test="$tokens[1] = 'bottom'">
	                                <xsl:attribute name="background-position-vertical" select="'bottom'"/>
	                                <xsl:attribute name="background-position-horizontal"
	                                    select="$tokens[2]"/>
	                            </xsl:when>
	                            <xsl:when test="$tokens[1] = 'left'">
	                                <xsl:attribute name="background-position-vertical"
	                                    select="$tokens[2]"/>
	                                <xsl:attribute name="background-position-horizontal" select="'left'"
	                                />
	                            </xsl:when>
	                            <xsl:when test="$tokens[1] = 'right'">
	                                <xsl:attribute name="background-position-vertical"
	                                    select="$tokens[2]"/>
	                                <xsl:attribute name="background-position-horizontal"
	                                    select="'right'"/>
	                            </xsl:when>
	                            <xsl:when test="$tokens[1] = 'center'">
	                                <xsl:choose>
	                                    <xsl:when test="$tokens[2] = 'top'">
	                                        <xsl:attribute name="background-position-vertical"
	                                            select="'top'"/>
	                                        <xsl:attribute name="background-position-horizontal"
	                                            select="'center'"/>
	                                    </xsl:when>
	                                    <xsl:when test="$tokens[2] = 'bottom'">
	                                        <xsl:attribute name="background-position-vertical"
	                                            select="'bottom'"/>
	                                        <xsl:attribute name="background-position-horizontal"
	                                            select="'center'"/>
	                                    </xsl:when>
	                                    <xsl:when test="$tokens[2] = 'left'">
	                                        <xsl:attribute name="background-position-vertical"
	                                            select="'center'"/>
	                                        <xsl:attribute name="background-position-horizontal"
	                                            select="'left'"/>
	                                    </xsl:when>
	                                    <xsl:when test="$tokens[2] = 'right'">
	                                        <xsl:attribute name="background-position-vertical"
	                                            select="'center'"/>
	                                        <xsl:attribute name="background-position-horizontal"
	                                            select="'right'"/>
	                                    </xsl:when>
	                                    <xsl:when test="$tokens[2] = 'center'">
	                                        <xsl:attribute name="background-position-vertical"
	                                            select="'center'"/>
	                                        <xsl:attribute name="background-position-horizontal"
	                                            select="'center'"/>
	                                    </xsl:when>
	                                </xsl:choose>
	                            </xsl:when>
	                        </xsl:choose>
	                    </xsl:otherwise>
	                </xsl:choose>
	            </xsl:when>
	            <xsl:otherwise>
	                <xsl:message>WARNING Cannot deal with the value specified for the "background-position" property: '<xsl:value-of select="$background-position"/>'</xsl:message>
	            </xsl:otherwise>
	        </xsl:choose>
	        </xsl:if>
    </xsl:template>    
    
</xsl:stylesheet>