package ro.sync.util.editorvars.expander;

import java.util.Iterator;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ro.sync.util.editorvars.parser.model.AskEditorVariable;
import ro.sync.util.editorvars.parser.model.CompoundLexicalItem;
import ro.sync.util.editorvars.parser.model.EditorVariable;
import ro.sync.util.editorvars.parser.model.LexicalItem;
import ro.sync.util.editorvars.parser.model.LexicalItem.Type;

/**
 * Utility methods.
 */
public final class EditorVariableUtil {

  /**
   * Logger for logging.
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(EditorVariableUtil.class.getName());

  /**
   * Constructor.
   *
   * @throws UnsupportedOperationException when invoked.
   */
  private EditorVariableUtil() {
    // Private to avoid instantiations
    throw new UnsupportedOperationException("Instantiation of this utility class is not allowed!");
  }
  
  /**
   * Serializes an editor variable.
   * 
   * @param edVar Editor variable.
   * 
   * @return The string representation. For example: ${framework(name)}
   */
  public static String serialize(EditorVariable edVar) {
    StringBuilder stringBuilder = new StringBuilder("${").append(edVar.getName());
    
    if (edVar.getType() == Type.ASK_EDITOR_VARIABLE) {
      if (edVar instanceof AskEditorVariable) {
        serializeAskEditorVariableParams((AskEditorVariable) edVar, stringBuilder);
      } else {
        LOGGER.warn("Unhandled \"ask\" editor variable type: {}", edVar.getClass(), new Exception());
      }
    } else {
      List<LexicalItem> params = edVar.getParams();
      if (params != null && !params.isEmpty()) {
        stringBuilder.append('(');
        serializeParams(params, stringBuilder);
        stringBuilder.append(')');
      }
    }
    stringBuilder.append('}');
    
    return stringBuilder.toString();
  }

  /**
   * Serialize parameters.
   * 
   * @param params The list of parameters.
   * @param stringBuilder A string builder to append to.
   */
  private static void serializeParams(List<LexicalItem> params, StringBuilder stringBuilder) {
    for (int i = 0; i < params.size(); i++) {
      LexicalItem li = params.get(i);
      if (i > 0) {
        stringBuilder.append(", ");
      }

      serialize(li, stringBuilder);
    }
  }

  /**
   * Serializes the lexical unit into the string buffer.
   * 
   * @param li Lexical item to serialize.
   * @param stringBuilder Builder to receive the string version.
   */
  private static void serialize(LexicalItem li, StringBuilder stringBuilder) {
    if (li instanceof EditorVariable) {
      stringBuilder.append(serialize((EditorVariable) li));
    } else if (li instanceof CompoundLexicalItem) {
      CompoundLexicalItem cli = (CompoundLexicalItem) li;
      List<LexicalItem> lexicalItems = cli.getLexicalItems();
      for (LexicalItem lexicalItem : lexicalItems) {
        serialize(lexicalItem, stringBuilder);
    }
    } else {
      stringBuilder.append(li.getStringValue());
    }
  }

  /**
   * Serializes the parameters of an $ask editor variable.
   * 
   * @param ask Editor variable.
   * @param stringBuilder Collector for the string version.
   */
  private static void serializeAskEditorVariableParams(AskEditorVariable ask, StringBuilder stringBuilder) {
    // Special function.
    stringBuilder.append("('");
    serialize(ask.getMessage(), stringBuilder);
    stringBuilder.append('\'');
    
    if (ask.getInputType() != null) {
      stringBuilder.append(", ");
      serialize(ask.getInputType(), stringBuilder);
    }
    
    serializeAskLabelValueEnumeration(ask, stringBuilder);
    
    if (ask.getDefaultValue() != null) {
      stringBuilder.append(", '");
      serialize(ask.getDefaultValue(), stringBuilder);
      stringBuilder.append('\'');
    }
    
    if (ask.getAnswerId() != null) {
      stringBuilder.append(", ");
      serialize(ask.getAnswerId(), stringBuilder);
    }
    
    stringBuilder.append(')');
  }

  /**
   * Serialize the label - value enumeration of an $ask.
   * @param ask             The ask editor variable.
   * @param stringBuilder   I will let you guess.
   */
  private static void serializeAskLabelValueEnumeration(AskEditorVariable ask, StringBuilder stringBuilder) {
    Iterable<LexicalItem> valueEnumeration = ask.getValueEnumeration();
    Iterator<LexicalItem> iterator = valueEnumeration.iterator();
    if (!ask.hasPendingValue() && iterator.hasNext()) {
      stringBuilder.append(", (");
      while(iterator.hasNext()) {
        LexicalItem lexicalItem = iterator.next();
        stringBuilder.append('\'');
        serialize(lexicalItem, stringBuilder);
        stringBuilder.append("':'");
        if (iterator.hasNext()) {
          // They should be in pairs. Just a precaution.
          serialize(iterator.next(), stringBuilder);
        }
        stringBuilder.append('\'');
        if (iterator.hasNext()) {
          stringBuilder.append("; ");  
        }
        
      }
      stringBuilder.append(')');
    } else if (ask.hasPendingValue()) {
        stringBuilder.append(", (");
        serialize(iterator.next(), stringBuilder);
        stringBuilder.append(')');
    }
  }
}
