/*
 * Copyright (c) 2000 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 *
 * $Id$
 */
package org.w3c.flute.parser.selectors.spc;

import java.util.Map;

/**
 * An implementation of the nth-child pseudo function.
 * 
 * The :nth-child(an+b) pseudo-class notation represents an element 
 * that has <code>an+b-1</code>siblings before 
 * it in the document tree, for any positive integer or zero value 
 * of n, and has a parent element. 
 * For values of a and b greater than zero, this effectively 
 * divides the element's children into groups of a elements 
 * (the last group taking the remainder), and selecting the bth
 * element of each group. For example, this allows the selectors 
 * to address every other row in a table, and could be used to 
 * alternate the color of paragraph text in a cycle 
 * of four. The a and b values must be integers (positive, 
 * negative, or zero). The index of the first child of an element 
 * is 1.
 * 
 * In addition to this, :nth-child() can take 'odd' and 'even' 
 * as arguments instead. 'odd' has the same signification as 2n+1, 
 * and 'even' has the same signification as 2n.
 * The argument to :nth-child() must match the grammar below, where 
 * INTEGER matches the token [0-9]+ and the rest of the tokenization 
 * is given by the Lexical scanner in section 10.2:
  * <pre>
  *  nth
  *    : S* [ ['-'|'+']? INTEGER? {N} [ S* ['-'|'+'] S* INTEGER ]? |
  *           ['-'|'+']? INTEGER | {O}{D}{D} | {E}{V}{E}{N} ] S*
  *    ;
 *  </pre>
 *  
 * @author dan
 * @see "http://www.w3.org/TR/css3-selectors/#nth-child-pseudo"
 */
public class NthChildSPCCondition extends SPCCondition {

  /**
   * The cycle of the condition. May be -1 or a a positive value. -1 means the 
   * condition is matching the first children with their index (1 based) less or 
   * equal to delta.
   */
  private int cycle;

  /**
   * The offset in the cycle. May be negative.
   */
  private int delta;

  /**
   * Constructor.
   * 
   * @param cycle The cycle after which the delta is triggered. For instance if cycle 
   * is 10 and delta is -1, the condition will match the child element with number 9, 19, 29, 39, etc..
   * It is 1 based. 
   * @param delta The offset in the cycle. It is one based, may be negative. 
   */
  public NthChildSPCCondition(int cycle, int delta) {
    this.cycle = cycle;
    this.delta = delta;
  }

  /**
   * @see org.w3c.flute.parser.selectors.spc.SPCCondition#getType()
   */
  @Override
  public Type getType() {
    return Type.NTH_CHILD;
  }

  /**
   * @return Returns the cycle.
   */
  public int getCycle() {
    return cycle;
  }

  /**
   * @return Returns the delta.
   */
  public int getDelta() {
    return delta;
  }

  /**
   * @see org.w3c.flute.css.sac.Condition#getSerialization(java.util.Map)
   */
  @Override
  public String getSerialization(Map<String, String> proxyNamespaceMapping) {
    return getSerialization("nth-child");
  }

  /**
   * Gets the condition serialization
   * 
   * @param name Can be one of nth-of-child, last-nth-of-child, etc..
   * @return The serialization of the condition.
   */
  protected String getSerialization(String name) {
    StringBuilder sb = new StringBuilder();
    sb.append(':');
    sb.append(name);
    sb.append('(');
    if (cycle != 0){
      sb.append(cycle);
      if (delta != 0) {
        sb.append('+');
        sb.append(delta);
      }
    } else {
      if (delta != 0) {
        sb.append(delta);
      }
    }
    sb.append(')');
    return sb.toString();
  }
}
