/*
 * Copyright (c) 2018 Syncro Soft SRL - All Rights Reserved.
 *
 * This file contains proprietary and confidential source code.
 * Unauthorized copying of this file, via any medium, is strictly prohibited.
 */
package org.w3c.flute.parser.selectors;

import java.util.Map;

import org.w3c.flute.css.sac.Condition;
import org.w3c.flute.css.sac.HasCondition;
import org.w3c.flute.css.sac.Selector;

/**
 * The condition matches the current element when the element has at 
 * least one descendant matching the relative selector.
 * 
 * https://drafts.csswg.org/selectors-4/#relational
 * 
 * @author dan
 */
public class HasConditionImpl implements HasCondition {
  
  /**
   * The relative selector.
   */
  private Selector selector;
  
  /**
   * The relative depth from the subject to the right end side of the selector. This 
   * is used only to implement old style subject selectors.
   * 
   * This is used when the given selector is a 
   * compatibility old style subject selector. For this selector
   * <pre>
   * A &gt; B &gt; C! &gt; D &gt;E
   * </pre>
   * the relative depth is 2, so the useful part of the selector will be <code>D &gt; E</code>. 
   * When using this constructor for the new :has() pseudo-class syntax:
   * <pre>
   * A &gt; B &gt; C:has(&gt; D &gt;E)
   * </pre>, the selector argument is <code>D &gt; E</code>, while the relative depth should be -1. 
   */
  private int relativeDepth;
  
  /**
   * The relative depth from the subject to the right end side of the selector. This 
   * is used only to implement old style subject selectors.
   * 
   * This is used when the given selector is a 
   * compatibility old style subject selector. For this selector
   * <pre>
   * A &gt; B &gt; C! &gt; D &gt;E
   * </pre>
   * the relative depth is 2, so the useful part of the selector will be <code>D &gt; E</code>. 
   * When using this constructor for the new :has() pseudo-class syntax:
   * <pre>
   * A &gt; B &gt; C:has(&gt; D &gt;E)
   * </pre>, the selector argument is <code>D &gt; E</code>, while the relative depth should be -1. 
   * 
   * @return Returns the relative depth.
   */
  public int getRelativeDepth() {
    return relativeDepth;
  }

  /**
   * Builds a :has condition.
   * 
   * @param selector The selector that is evaluated by the condition. If the selector
   * is not matched, the condition is accomplished.
   */
  public HasConditionImpl(Selector selector) {
    this(selector, -1);
  }

  /**
   * Builds a :has condition.
   * 
   * @param selector The selector that is evaluated by the condition. If the selector
   * is not matched, the condition is accomplished.
   * @param relativeDepth This is used when the given selector is a 
   * compatibility old style subject selector. For this selector
   * <pre>
   * A &gt; B &gt; C! &gt; D &gt;E
   * </pre>
   * the relative depth is 2, so the useful part of the selector will be <code>D &gt; E</code>. 
   * When using this constructor for the new :has() pseudo-class syntax:
   * <pre>
   * A &gt; B &gt; C:has(&gt; D &gt;E)
   * </pre>, the selector argument is <code>D &gt; E</code>, while the relative depth should be -1. 
   */
  public HasConditionImpl(Selector selector, int relativeDepth) {
    this.selector = selector;
    this.relativeDepth = relativeDepth;    
  }

  /**
   * @see org.w3c.css.sac.Condition#getConditionType()
   */
  @Override
  public short getConditionType() {
    return Condition.SAC_HAS_CONDITION;
  }


  /**
   * Gets the relative selector.
   * 
   * @return the selector, never <code>null</code>.
   */
  @Override
  public Selector getSelector() {
    return selector;    
  }

  /**
   * @see org.w3c.flute.css.sac.Condition#getSerialization(Map)
   */
  @Override
  public String getSerialization(Map<String, String> proxyNamespaceMapping) {   
    return ":has(" + selector.getSerialization(proxyNamespaceMapping) + ")";
  }
}
