/*
 * Copyright (c) 2018 Syncro Soft SRL - All Rights Reserved.
 *
 * This file contains proprietary and confidential source code.
 * Unauthorized copying of this file, via any medium, is strictly prohibited.
 */
package org.w3c.flute.parser;

import java.util.LinkedHashSet;
import java.util.Set;

/**
 * A media query part.
 */
public class MediaQuery implements Cloneable {

  /**
   * Line where the media query was defined.
   */
  private final int definedInLine;
  /**
   * Column where it was defined.
   */
  private final int definedInColumn;
  
  /**
   * List of possible prefixes, NOT and ONLY.
   */
  public enum MediaQueryPrefix {
    NOT,
    ONLY;
    @Override
    public String toString() {
      return this.name().toLowerCase();
    }
  }
  /**
   * The prefix
   */
  private final MediaQueryPrefix prefix;
  /**
   * Media type.
   */
  private String mediaType;
  /**
   * List of feature, expression pairs
   */
  private final Set<MediaFeatureExpression> mediaFeatureExpresions;
  
  /**
   * Constructor.
   * 
   * @param mediaType The media type.
   */
  public MediaQuery(String mediaType) {
    this(mediaType, -1, -1);
  }

  /**
   * Constructor.
   * 
   * @param mediaTypeName Media type.
   * @param line          Line 
   * @param col           Column
   */
  public MediaQuery(String mediaTypeName, int line, int col) {
    this(null, mediaTypeName, line, col);
  }

  /**
   * Constructor.
   * @param prefix Prefix
   * @param mediaTypeName Media type name.
   * @param line          Line 
   * @param coloumn       Column.
   */
  public MediaQuery(MediaQueryPrefix prefix, String mediaTypeName, int line, int coloumn) {
    definedInLine = line;
    definedInColumn = coloumn;
    mediaType = mediaTypeName;
    mediaFeatureExpresions = new LinkedHashSet<MediaFeatureExpression>();
    this.prefix = prefix;
  }

  /**
   * Get media type.
   * 
   * @return The media type.
   */
  public String getMediaType() {
    return mediaType;
  }
  
  /**
   * Add media feature expression.
   * 
   * @param expression The added expression.
   */
  public void addMediaFeatureExpression(MediaFeatureExpression expression) {
    mediaFeatureExpresions.add(expression);
  }
  
  /**
   * @see java.lang.Object#toString()
   */
  @Override
  public String toString() {
    StringBuilder toReturn = new StringBuilder();
    if (this.prefix != null)
      toReturn.append(prefix + " ");
    toReturn.append(mediaType);
    for (MediaFeatureExpression mfe : mediaFeatureExpresions) {
      if (toReturn.length() > 0)
        toReturn.append(" and ");
      toReturn.append(mfe);
    }
    return toReturn.toString();
  }

  /**
   * @see java.lang.Object#clone()
   */
  @Override
  public MediaQuery clone() {
    MediaQuery toReturn =  new MediaQuery(prefix, mediaType, getLine(), getColumn());
    for (MediaFeatureExpression mfe : mediaFeatureExpresions)
      toReturn.mediaFeatureExpresions.add(mfe.clone());
    return toReturn;
  }

  /**
   * @return The line
   */
  public int getLine() {
    return definedInLine;
  }

  /**
   * @return The column.
   */
  public int getColumn() {
    return definedInColumn;
  }

  /**
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + definedInColumn;
    result = prime * result + definedInLine;
    result = prime
        * result
        + ((mediaFeatureExpresions == null) ? 0
            : mediaFeatureExpresions.hashCode());
    result = prime * result
        + ((mediaType == null) ? 0 : mediaType.hashCode());
    result = prime * result + ((prefix == null) ? 0 : prefix.hashCode());
    return result;
  }

  /**
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    MediaQuery other = (MediaQuery) obj;
    if (definedInColumn != other.definedInColumn)
      return false;
    if (definedInLine != other.definedInLine)
      return false;
    if (mediaType == null) {
      if (other.mediaType != null)
        return false;
    } else if (!mediaType.equals(other.mediaType))
      return false;
    if (prefix != other.prefix)
      return false;
    if (mediaFeatureExpresions == null) {
      if (other.mediaFeatureExpresions != null)
        return false;
    } else if (!mediaFeatureExpresions.equals(other.mediaFeatureExpresions))
      return false;
    return true;
  }
  
  /**
   * Get the media features expressions
   * 
   * @return Returns the media Feature Expresions.
   */
  public Set<MediaFeatureExpression> getMediaFeatureExpresions() {
    return mediaFeatureExpresions;
  }
}
