<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    xmlns:map="http://www.w3.org/2005/xpath-functions/map"
    xmlns:f="http://www.oxygenxml.com/xsl/functions" exclude-result-prefixes="xs f map"
    version="2.0">
    
    <!-- Generate DITA for Security Requirement Objects. -->
    <xsl:template name="securityRequirementObject">
        <xsl:param name="securityReqNodes"/>
        <section outputclass="openapi-security-requirements">
            <title>Security Requirements</title>
            <xsl:if test="f:arrayHasElements($securityReqNodes)">
                <ul outputclass="openapi-security-requirements-list">
                	<xsl:for-each-group select="$securityReqNodes[not(local-name() = 'array')], $securityReqNodes[local-name() = 'array']/*" group-by="local-name()">
                	    <li outputclass="openapi-security-requirement">
                	        <xref outputclass="openapi-security-requirement-name">
                                <xsl:attribute name="href" select="concat('#security-schemes/', current-grouping-key())"/>
                                <xsl:value-of select="current-grouping-key()"/>
                            </xref>
                            <xsl:variable name="currentGroup" select="current-group()"/>
                            <xsl:variable name="firstValue">
                                <xsl:choose>
                                    <xsl:when test="$currentGroup[1]/array">
                                        <xsl:value-of select="$currentGroup[1]/array"/>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <xsl:value-of select="$currentGroup[1]"/>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </xsl:variable>
                            <xsl:if test="count($currentGroup) > 1 or string-length($firstValue) > 0">
                                <ul outputclass="openapi-security-requirement-scopes">
                                    <xsl:for-each select="current-group()">
                                        <xsl:variable name="scopeVal">
                                            <xsl:choose>
                                                <xsl:when test="./array">
                                                    <xsl:value-of select="./*"/>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <xsl:value-of select="."/>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </xsl:variable>
                                        <li outputclass="openapi-security-requirement-scope">
                                            <xref>
                                                <xsl:attribute name="href" select="concat('#security-schemes/', $scopeVal)"/>
                                                <xsl:value-of select="$scopeVal"/>
                                            </xref>
                                        </li>
                                    </xsl:for-each>
                                </ul>
                            </xsl:if>
                        </li>
                    </xsl:for-each-group>
                </ul>
            </xsl:if>
        </section>
    </xsl:template>
    
    <!-- Generate DITA content for Security Scheme Objects map from Components. -->
    <xsl:template name="securitySchemes">
        <xsl:param name="securitySchemeMapItems"/>
        <xsl:if test="$securitySchemeMapItems and count($securitySchemeMapItems) > 0">
            <topic id="security-schemes" outputclass="openapi-securitySchemes">
                <title>Security schemes</title>
                <body>
                    <xsl:for-each select="$securitySchemeMapItems">
                        <xsl:call-template name="securitySchemeObject">
                            <xsl:with-param name="securitySchemeObjectNode" select="."/>
                        </xsl:call-template>
                    </xsl:for-each>
                </body>
            </topic>
        </xsl:if>
    </xsl:template>
    
    <!-- Generate DITA content for Security Scheme Object -->
    <xsl:template name="securitySchemeObject">
        <xsl:param name="securitySchemeObjectNode"/>
        <xsl:if test="$securitySchemeObjectNode">
            <xsl:variable name="schemeName" select="local-name($securitySchemeObjectNode)"/>
            
            <xsl:variable name="type" select="$securitySchemeObjectNode/type"/>
            <xsl:variable name="description" select="$securitySchemeObjectNode/description"/>
            <xsl:variable name="name" select="$securitySchemeObjectNode/name"/>
            <xsl:variable name="in" select="$securitySchemeObjectNode/in"/>
            <xsl:variable name="scheme" select="$securitySchemeObjectNode/scheme"/>
            <xsl:variable name="bearerFormat" select="$securitySchemeObjectNode/bearerFormat"/>
            <xsl:variable name="flows" select="$securitySchemeObjectNode/flows"/>
            <xsl:variable name="openIdConnectUrl" select="$securitySchemeObjectNode/openIdConnectUrl"/>
            
            <section>
                <xsl:attribute name="id"><xsl:value-of select="$schemeName"/></xsl:attribute>
                <xsl:attribute name="outputclass" select="concat('openapi-securityScheme ', $toUnescapeIdOutputClass)"/>
                <title>
                    <xsl:attribute name="outputclass" select="concat('openapi-securityScheme-name ', $toUnescapeOutputClass)"/>
                    <xsl:value-of select="$schemeName"/>
                </title>
                
                <xsl:choose>
                    <xsl:when test="$securitySchemeObjectNode/_X24_ref">
                        <!-- It's a reference to a security scheme -->
                        <xsl:call-template name="reference">
                            <xsl:with-param name="referenceNode" select="$securitySchemeObjectNode/_X24_ref"></xsl:with-param>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:otherwise>
                        <dl outputclass="openapi-securityScheme-properties">
                            <xsl:if test="$name">
                                <dlentry>
                                    <dt>Name</dt>
                                    <dd outputclass="openapi-securityScheme-name">
                                        <xsl:value-of select="$name"/>
                                    </dd>
                                </dlentry>
                            </xsl:if>
                            <xsl:if test="$description">
                                <dlentry>
                                    <dt>Description</dt>
                                    <dd>
                                        <xsl:attribute name="outputclass" select="concat('openapi-securityScheme-description ', $markdownContentOutputClass)"/>
                                        <xsl:value-of select="$description"/>
                                    </dd>
                                </dlentry>
                            </xsl:if>
                            <xsl:if test="$type">
                                <dlentry>
                                    <dt>Type</dt>
                                    <dd outputclass="openapi-securityScheme-type">
                                        <xsl:value-of select="$type"/>
                                    </dd>
                                </dlentry>
                            </xsl:if>
                            <xsl:if test="$in">
                                <dlentry>
                                    <dt>Location</dt>
                                    <dd outputclass="openapi-securityScheme-in">
                                        <xsl:value-of select="$in"/>
                                    </dd>
                                </dlentry>
                            </xsl:if>
                            <xsl:if test="$scheme">
                                <dlentry>
                                    <dt>HTTP Authorization scheme</dt>
                                    <dd outputclass="openapi-securityScheme-scheme">
                                        <xsl:value-of select="$scheme"/>
                                    </dd>
                                </dlentry>
                            </xsl:if>
                            <xsl:if test="$bearerFormat">
                                <dlentry>
                                    <dt>Bearer token format</dt>
                                    <dd outputclass="openapi-securityScheme-bearerFormat"><xsl:value-of select="$bearerFormat"/></dd>
                                </dlentry>
                            </xsl:if>
                            <xsl:if test="$openIdConnectUrl">
                                <dlentry>
                                    <dt>OpenId Connect URL</dt>
                                    <dd outputclass="openapi-securityScheme-openIdConnectUrl"><xsl:value-of select="$openIdConnectUrl"/></dd>
                                </dlentry>
                            </xsl:if>
                            <xsl:if test="$flows">
                                <xsl:call-template name="oAuthFlows">
                                    <xsl:with-param name="flowsObject" select="$flows"/>
                                </xsl:call-template>
                            </xsl:if>
                        </dl>
                        
                        <xsl:call-template name="extensionsHandlerAsList">
                            <xsl:with-param name="element" select="$securitySchemeObjectNode"/>
                        </xsl:call-template>
                    </xsl:otherwise>
                </xsl:choose>
            </section>
        </xsl:if>
    </xsl:template>
    
    <!-- Generate DITA content for OAuth Flows Object -->
    <xsl:template name="oAuthFlows">
        <xsl:param name="flowsObject"/>
        
        <xsl:if test="$flowsObject/implicit">
            <dlentry outputclass="openapi-oAuthFlow-implicit">
                <dt>OAuth Implicit flow</dt>
                <dd>
                    <xsl:call-template name="oAuthFlow">
                        <xsl:with-param name="oAuthFlowObject" select="$flowsObject/implicit"/>
                    </xsl:call-template>
                </dd>
            </dlentry>
        </xsl:if>
        <xsl:if test="$flowsObject/password">
            <dlentry outputclass="openapi-oAuthFlow-password">
                <dt>OAuth Resource Owner Password flow</dt>
                <dd>
                    <xsl:call-template name="oAuthFlow">
                        <xsl:with-param name="oAuthFlowObject" select="$flowsObject/password"/>
                    </xsl:call-template>
                </dd>
            </dlentry>
        </xsl:if>
        <xsl:if test="$flowsObject/clientCredentials">
            <dlentry outputclass="openapi-oAuthFlow-clientCredentials">
                <dt>OAuth Client Credentials flow</dt>
                <dd>
                    <xsl:call-template name="oAuthFlow">
                        <xsl:with-param name="oAuthFlowObject" select="$flowsObject/clientCredentials"/>
                    </xsl:call-template>
                </dd>
            </dlentry>
        </xsl:if>
        <xsl:if test="$flowsObject/authorizationCode">
            <dlentry outputclass="openapi-oAuthFlow-authorizationCode">
                <dt>OAuth Authorization Code flow</dt>
                <dd>
                    <xsl:call-template name="oAuthFlow">
                        <xsl:with-param name="oAuthFlowObject" select="$flowsObject/authorizationCode"/>
                    </xsl:call-template>
                </dd>
            </dlentry>
        </xsl:if>
    </xsl:template>
    
    <!-- Generate DITA content for OAuth Flow Object -->
    <xsl:template name="oAuthFlow">
        <xsl:param name="oAuthFlowObject"/>
        <xsl:variable name="authorizationUrl" select="$oAuthFlowObject/authorizationUrl"/>
        <xsl:variable name="tokenUrl" select="$oAuthFlowObject/tokenUrl"/>
        <xsl:variable name="refreshUrl" select="$oAuthFlowObject/refreshUrl"/>
        <xsl:variable name="scopes" select="$oAuthFlowObject/scopes"/>
        
        <dl outputclass="openapi-oAuthFlow-properties">
            <xsl:if test="$authorizationUrl">
                <dlentry>
                    <dt>Authorization URL</dt>
                    <dd outputclass="openapi-oAuthFlow-authorizationUrl">
                        <xsl:value-of select="$authorizationUrl"/>
                    </dd>
                </dlentry>
            </xsl:if>
            <xsl:if test="$tokenUrl">
                <dlentry>
                    <dt>Token URL</dt>
                    <dd outputclass="openapi-oAuthFlow-tokenUrl">
                        <xsl:value-of select="$tokenUrl"/>
                    </dd>
                </dlentry>
            </xsl:if>
            <xsl:if test="$refreshUrl">
                <dlentry>
                    <dt>URL for refresh tokens</dt>
                    <dd outputclass="openapi-oAuthFlow-refreshUrl">
                        <xsl:value-of select="$refreshUrl"/>
                    </dd>
                </dlentry>
            </xsl:if>
            <xsl:if test="$scopes">
                <dlentry>
                    <dt>Scopes</dt>
                    <dd>
                        <dl outputclass="openapi-oAuthFlow-scopes">
                            <xsl:for-each select="$scopes/*">
                                <dlentry>
                                    <xsl:attribute name="id" select="local-name()"/>
                                    <xsl:attribute name="outputclass" select="concat('openapi-oAuthFlow-scope ', $toUnescapeIdOutputClass)"/>
                                    <dt>
                                        <xsl:attribute name="outputclass" select="concat('openapi-oAuthFlow-scope-name ',$toUnescapeOutputClass)"/>
                                        <xsl:value-of select="local-name()"/>
                                    </dt>
                                    <dd outputclass="openapi-oAuthFlow-scope-description">
                                        <xsl:value-of select="."/>
                                    </dd>
                                </dlentry>
                            </xsl:for-each>
                        </dl>
                    </dd>
                </dlentry>
            </xsl:if>
        </dl>
        <xsl:call-template name="extensionsHandlerAsList">
            <xsl:with-param name="element" select="$oAuthFlowObject"/>
        </xsl:call-template>
    </xsl:template>
</xsl:stylesheet>