<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    xmlns:map="http://www.w3.org/2005/xpath-functions/map"
    xmlns:f="http://www.oxygenxml.com/xsl/functions" exclude-result-prefixes="xs f map"
    version="2.0">
    
    <!-- The format value for the xref value to OpenAPI documents. -->
    <xsl:param name="openApiXrefFormat" select="''" as="xs:string"/>
    
    <!-- Generate DITA content for Schema objects. -->
    <xsl:template name="schemas">
        <xsl:param name="schemasNodes"/>
        <xsl:if test="$schemasNodes and count($schemasNodes) > 0">
            <topic id="schemas" outputclass="openapi-components-schemas">
                <title>Schemas</title>
                <body>
                    <xsl:for-each select="$schemasNodes">
                        <xsl:if test=".">
                            <section outputclass="openapi-schema">
                                <xsl:attribute name="id"><xsl:value-of select="local-name()"/></xsl:attribute>
                                <xsl:attribute name="outputclass" select="concat('openapi-schema ', $toUnescapeIdOutputClass)" />
                                <title>
                                    <xsl:attribute name="outputclass" select="concat('openapi-schema-key ', $toUnescapeOutputClass)"/>
                                    <xsl:value-of select="local-name()"/>
                                </title>
                                <xsl:call-template name="schemaObject">
                                    <xsl:with-param name="schemaNode" select="."></xsl:with-param>
                                </xsl:call-template>
                            </section>
                        </xsl:if>
                    </xsl:for-each>
                </body>
            </topic>
        </xsl:if>
    </xsl:template>
    
    <!-- Generate DITA content for Schema object. -->
    <xsl:template name="schemaObject">
        <xsl:param name="schemaNode"/>
        <xsl:if test="$schemaNode/properties">
            <dl outputclass="openapi-schema-properties">
                <dlentry>
                    <dt>Properties</dt>
                    <dd>
                        <xsl:call-template name="schemaProperties">
                            <xsl:with-param name="properties" select="$schemaNode/properties/*"></xsl:with-param>
                        </xsl:call-template>
                    </dd>
                </dlentry>
            </dl>
        </xsl:if>
        <xsl:if test="$schemaNode/*[not(local-name() = 'properties')]">
            <xsl:call-template name="schemaProperties">
                <xsl:with-param name="properties" select="$schemaNode/*[not(local-name() = 'properties')]"></xsl:with-param>
            </xsl:call-template>
        </xsl:if>
    </xsl:template>
    
    <!-- Handle properties from Schema object. -->
    <xsl:template name="schemaProperties">
        <xsl:param name="properties"/>
        <xsl:if test="$properties">
            <xsl:choose>
                <xsl:when test="count($properties) > 0">
                    <dl>
                        <xsl:attribute name="outputclass" select="concat('openapi-schema-values ', $toUnescapeIdOutputClass)"/>
                        <xsl:for-each-group select="$properties[not(array)], $properties[array]/*" 
                            group-by="local-name()" 
                            composite="no">
                            <dlentry>
                                <dt>
                                    <xsl:attribute name="outputclass" select="$toUnescapeOutputClass"/>
                                    <xsl:value-of select="current-grouping-key()"/>
                                </dt>
                                <dd>
                                    <xsl:if test="current-grouping-key() = 'description'">
                                        <xsl:attribute name="outputclass" select="$markdownContentOutputClass"/>
                                    </xsl:if>
                                    <xsl:variable name="propertiesWithSameName" select="current-group()"/>
                                    <xsl:if test="$propertiesWithSameName">
                                        <xsl:choose>
                                            <xsl:when test="count($propertiesWithSameName) > 1">
                                                <ul>
                                                    <xsl:for-each select="$propertiesWithSameName">
                                                        <li>
                                                            <xsl:call-template name="schemaPropertyValue">
                                                                <xsl:with-param name="propertyValue" select="."></xsl:with-param>
                                                            </xsl:call-template>
                                                        </li>
                                                    </xsl:for-each>
                                                </ul>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:call-template name="schemaPropertyValue">
                                                    <xsl:with-param name="propertyValue" select="$propertiesWithSameName"></xsl:with-param>
                                                </xsl:call-template>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:if>
                                </dd>
                            </dlentry>
                        </xsl:for-each-group>
                    </dl>
                </xsl:when>
                <xsl:otherwise>
                    <p><xsl:value-of select="$properties"/></p>
                </xsl:otherwise>
            </xsl:choose> 
        </xsl:if>
    </xsl:template>
    
    <!-- Handle property value from Schema object. -->
    <xsl:template name="schemaPropertyValue">
        <xsl:param name="propertyValue"/>
        <xsl:choose>
            <xsl:when test="local-name($propertyValue) = '_X24_ref'">
                <xsl:call-template name="reference">
                    <xsl:with-param name="referenceNode" select="."></xsl:with-param>
                </xsl:call-template>
            </xsl:when>
            <xsl:when test="count($propertyValue/*) > 0">
                <xsl:choose>
                    <xsl:when test="$propertyValue/type">
                        <div>
                            <ph outputclass="openapi-schema-type">
                                <xsl:value-of select="$propertyValue/type"/>
                            </ph>
                            <xsl:if test="$propertyValue/format">
                                <i outputclass="openapi-schema-format">
                                    <xsl:value-of select="concat('(', $propertyValue/format, ')')"/>
                                </i>
                            </xsl:if>
                        </div>
                        <xsl:call-template name="schemaProperties">
                            <xsl:with-param name="properties" select="$propertyValue/*[not(local-name() = 'type' or local-name() = 'format')]"/>
                        </xsl:call-template>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:call-template name="schemaProperties">
                            <xsl:with-param name="properties" select="$propertyValue/*"/>
                        </xsl:call-template>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:when>
            <xsl:otherwise>
                <xsl:value-of select="."/>
            </xsl:otherwise>
        </xsl:choose>
        
    </xsl:template>
    
    <!-- Generate DITA content from Reference Object -->
    <xsl:template name="reference">
        <xsl:param name="referenceNode"/>
        
        <xsl:if test="local-name($referenceNode) = '_X24_ref'">
            <xsl:variable name="refValue" select="$referenceNode/text()"/>   
            <!-- Open API 3.1 fields -->
            <xsl:variable name="summary" select="$referenceNode/following-sibling::summary"/>
            <xsl:variable name="description" select="$referenceNode/following-sibling::description"/>
            
            <xref outputclass="openapi-reference">
                <xsl:choose>
                    <xsl:when test="starts-with($refValue, '#')">
                        <xsl:attribute name="href">
                            <xsl:value-of select="f:processOpenApiAnchor($refValue)"/>
                        </xsl:attribute>
                        <xsl:if test="$openApiXrefFormat">
                            <xsl:attribute name="format" select="$openApiXrefFormat"/>
                        </xsl:if>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:choose>
                            <xsl:when test="contains($refValue, '#')">
                                <xsl:attribute name="outputclass" select="concat('openapi-reference ', $externalOpenApiOutputClass)"/>
                                <xsl:variable name="filePath" select="substring-before($refValue, '#')"/>
                                <xsl:variable name="anchor" select="f:processOpenApiAnchor(substring-after($refValue, '#'))"/>
                                <xsl:attribute name="href" select="concat($filePath, $anchor)"/>
                                
                                <xsl:if test="$openApiXrefFormat">
                                    <xsl:attribute name="format" select="$openApiXrefFormat"/>
                                </xsl:if>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:attribute name="outputclass" select="'openapi-reference'"/>
                                <xsl:attribute name="href" select="$refValue"/>
                            </xsl:otherwise>
                        </xsl:choose>
                    </xsl:otherwise>
                </xsl:choose>
                <xsl:if test="$summary">
                    <xsl:value-of select="$summary"/>
                </xsl:if>
            </xref>
            <xsl:if test="$description">
                <div>
                    <xsl:attribute name="outputclass" select="concat('openapi-reference-description ', $markdownContentOutputClass)"/>
                    <xsl:value-of select="$description"/>
                </div>
            </xsl:if>
        </xsl:if>
    </xsl:template>
    
    <!-- Process the anchor -->
    <xsl:function name="f:processOpenApiAnchor" as="xs:string">
        <xsl:param name="anchor"/>
        <xsl:variable name="valAfterFirtStep">
            <xsl:choose>
                <xsl:when test="starts-with($anchor, '#/')">
                    <xsl:value-of select="substring($anchor, 2)"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="substring($anchor, 1)"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:choose>
            <xsl:when test="contains($valAfterFirtStep, 'components/callbacks/')">
                <xsl:value-of select="concat('#', substring-after($valAfterFirtStep, 'components/callbacks/'))"/>
            </xsl:when>
            <xsl:when test="contains($valAfterFirtStep, 'components/pathItems/')">
                <xsl:value-of select="concat('#', substring-after($valAfterFirtStep, 'components/pathItems/'))"/>
            </xsl:when>
            <xsl:when test="contains($valAfterFirtStep, 'components/')">
                <xsl:value-of select="concat('#', substring-after($valAfterFirtStep, 'components/'))"/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:value-of select="concat('#', $valAfterFirtStep)"/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:function>
</xsl:stylesheet>