<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    xmlns:map="http://www.w3.org/2005/xpath-functions/map"
    xmlns:f="http://www.oxygenxml.com/xsl/functions" exclude-result-prefixes="xs f map"
    version="2.0">
    
    <!-- Generate DITA for OpenAPI Parameter objects array. -->
    <xsl:template name="params">
        <xsl:param name="operationParams"/>
        <xsl:param name="pathParams"/>
        <xsl:if test="$operationParams or $pathParams">
            <section outputclass="openapi-parameters">
                <title>Parameters</title>
                <xsl:if test="f:arrayHasElements($pathParams) or f:arrayHasElements($operationParams)">
                    <dl outputclass="openapi-parameters-list">
                        <!-- We merge (and override) the params defined on this operation with the params defined from parent path  -->
                        <xsl:for-each-group
                            select="$operationParams[not(array)], $operationParams[array]/array, $pathParams[not(array)], $pathParams[array]/array"
                            group-by="concat(./name, '-', ./in)">
                            <xsl:choose>
                                <xsl:when test="current-grouping-key() = '-'">
                                    <!-- It's not a params group. It's a references to params from components group-->
                                    <xsl:for-each select="current-group()/*">
                                        <dlentry outputclass="openapi-parameter">
                                            <dt outputclass="openapi-parameter-reference">
                                                <xsl:call-template name="reference">
                                                    <xsl:with-param name="referenceNode" select="."/>
                                                </xsl:call-template>
                                            </dt>
                                            <dd></dd>
                                        </dlentry>
                                    </xsl:for-each>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="paramObject">
                                        <xsl:with-param name="paramObjectNode" select="current-group()[1]"/>
                                        <xsl:with-param name="paramId" select="current-grouping-key()"/>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:for-each-group>
                    </dl>
                </xsl:if>
            </section>
        </xsl:if>
    </xsl:template>
    
    <!-- Generate DITA for an OpenAPI Parameter object. -->
    <xsl:template name="paramObject">
        <xsl:param name="paramObjectNode"/>
        <xsl:param name="paramId"/>
        <dlentry outputclass="openapi-parameter">
            <xsl:attribute name="id"><xsl:value-of select="f:correctId($paramId)"/></xsl:attribute>
            <xsl:attribute name="outputclass" select="concat('openapi-parameter ', $toUnescapeIdOutputClass)"/>
            <dt outputclass="openapi-parameter-name-in">
                <xsl:choose>
                    <xsl:when test="$paramObjectNode/name">
                        <ph outputclass="openapi-parameter-name">
                            <xsl:value-of select="$paramObjectNode/name"/>
                        </ph>
                        <xsl:if test="$paramObjectNode/in">
                            <ph outputclass="openapi-parameter-in">
                                <xsl:value-of select="concat('(', $paramObjectNode/in, ')')"/>
                            </ph>
                        </xsl:if>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="$paramId"/>
                    </xsl:otherwise>
                </xsl:choose>
            </dt>
            <dd>
                <xsl:call-template name="paramDesc">
                    <xsl:with-param name="paramObject" select="$paramObjectNode"/>
                </xsl:call-template>
            </dd>
        </dlentry>  
    </xsl:template>
    
    <!-- Generate DITA for an OpenAPI Parameter object's fields. -->
    <xsl:template name="paramDesc">
        <xsl:param name="paramObject"/>
        <xsl:choose>
            <xsl:when test="$paramObject/_X24_ref">
                <!-- It's a reference to a parameter/header -->
                <xsl:call-template name="reference">
                    <xsl:with-param name="referenceNode" select="$paramObject/_X24_ref"></xsl:with-param>
                </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
                <xsl:variable name="description" select="$paramObject/description"/>
                
                <xsl:variable name="required" select="$paramObject/required"/>
                <xsl:variable name="deprecated" select="$paramObject/deprecated"/>
                <xsl:variable name="allowEmptyValue" select="$paramObject/allowEmptyValue"/>
                
                <xsl:variable name="style" select="$paramObject/style"/>
                <xsl:variable name="explode" select="$paramObject/explode"/>
                <xsl:variable name="allowReserved" select="$paramObject/allowReserved"/>
                <xsl:variable name="schema" select="$paramObject/schema"/>
                <xsl:variable name="example" select="$paramObject/example"/>
                <xsl:variable name="examples" select="$paramObject/examples"/>
                
                <xsl:variable name="content" select="$paramObject/content"/>
                
                <xsl:if test="$description">
                    <div>
                        <xsl:attribute name="outputclass" select="$markdownContentOutputClass"/>
                        <xsl:value-of select="$description"/>
                    </div>
                </xsl:if>
                
                <xsl:if test="$schema">
                    <xsl:choose>
                        <xsl:when test="$schema/_X24_ref">
                            <b>Schema: </b>
                            <xsl:call-template name="reference">
                                <xsl:with-param name="referenceNode" select="$schema/_X24_ref"></xsl:with-param>
                            </xsl:call-template>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:call-template name="schemaObject">
                                <xsl:with-param name="schemaNode" select="$schema"></xsl:with-param>
                            </xsl:call-template>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:if>
                
                <xsl:if test="$required or $deprecated or $allowEmptyValue or  $style or $explode or $allowReserved or $example or $examples or $content">
                    <dl>
                        <xsl:attribute name="outputclass" select="'openapi-parameter-properties'"/>
                        <xsl:if test="$required">
                            <dlentry>
                                <xsl:attribute name="outputclass" select="'openapi-parameter-required'"/>
                                <dt>required</dt>
                                <dd><xsl:value-of select="$required"/></dd>
                            </dlentry>
                        </xsl:if>
                        <xsl:if test="$deprecated">
                            <dlentry>
                                <xsl:attribute name="outputclass" select="'openapi-parameter-deprecated'"/>
                                <dt>deprecated</dt>
                                <dd><xsl:value-of select="$deprecated"/></dd>
                            </dlentry>
                        </xsl:if>
                        <xsl:if test="$allowEmptyValue">
                            <dlentry>
                                <xsl:attribute name="outputclass" select="'openapi-parameter-allowEmptyValue'"/>
                                <dt>allow empty value</dt>
                                <dd><xsl:value-of select="$allowEmptyValue"/></dd>
                            </dlentry>
                        </xsl:if>
                        
                        <xsl:if test="$style">
                            <dlentry>
                                <xsl:attribute name="outputclass" select="'openapi-parameter-style'"/>
                                <dt>style</dt>
                                <dd>
                                    <xsl:value-of select="$style"/>
                                </dd>
                            </dlentry>
                        </xsl:if>
                        <xsl:if test="$explode">
                            <dlentry>
                                <xsl:attribute name="outputclass" select="'openapi-parameter-explode'"/>
                                <dt>explode</dt>
                                <dd>
                                    <xsl:value-of select="$explode"/>
                                </dd>
                            </dlentry>
                        </xsl:if>
                        <xsl:if test="$allowReserved">
                            <dlentry>
                                <xsl:attribute name="outputclass" select="'openapi-parameter-allowReserved'"/>
                                <dt>allow reserved</dt>
                                <dd><xsl:value-of select="$allowReserved"/></dd>
                            </dlentry>
                        </xsl:if>
                        <xsl:if test="$example">
                            <dlentry>
                                <xsl:attribute name="outputclass" select="'openapi-parameter-example'"/>
                                <dt>example</dt>
                                <dd>
                                    <xsl:call-template name="handleExample">
                                        <xsl:with-param name="example" select="$example"/>
                                    </xsl:call-template>
                                </dd>
                            </dlentry>
                        </xsl:if>
                        <xsl:if test="$examples">
                            <dlentry>
                                <xsl:attribute name="outputclass" select="'openapi-parameter-examples'"/>
                                <dt>examples</dt>
                                <dd>
                                    <xsl:call-template name="exemplesMap">
                                        <xsl:with-param name="exempleItems" select="$examples/*"></xsl:with-param>
                                    </xsl:call-template>
                                </dd>
                            </dlentry>
                        </xsl:if>
                        <xsl:if test="$content">
                            <xsl:call-template name="contentDlEntry">
                                <xsl:with-param name="mediaTypesNode" select="$content"/>
                                <xsl:with-param name="outputclassVal" select="'openapi-parameter-content'"/>
                            </xsl:call-template>
                        </xsl:if>
                    </dl> 
                </xsl:if>
                
                <xsl:call-template name="extensionsHandlerAsList">
                    <xsl:with-param name="element" select="$paramObject"/>
                </xsl:call-template>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- Generate DITA for OpenAPI Parameter Example -->
    <xsl:template name="handleExample">
        <xsl:param name="example"/>
        <codeblock>
            <xsl:choose>
                <xsl:when test="count($example/*) > 0">
                    <xsl:attribute name="outputclass" select="$toOriginalFormatOutputClass"/>
                    <xsl:copy-of select="$example/*"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="$example"/>
                </xsl:otherwise>
            </xsl:choose>
        </codeblock>
    </xsl:template>
    
    <!-- Generate DITA section for OpenAPI Request Body -->
    <xsl:template name="requestBodySection">
        <xsl:param name="requestBody" required="no"/>
        <xsl:if test="$requestBody">
            <section outputclass="openapi-requestBody">
                <title>Request body</title>
                <xsl:call-template name="requestBodyObject">
                    <xsl:with-param name="requestBody" select="$requestBody"></xsl:with-param>
                </xsl:call-template>
            </section>
        </xsl:if>
    </xsl:template>
    
    <!-- Generate DITA for OpenAPI Request Body Object-->
    <xsl:template name="requestBodyObject">
        <xsl:param name="requestBody"/>
        <xsl:choose>
            <xsl:when test="$requestBody/_X24_ref">
                <!-- It's a reference to a requestBody -->
                <xsl:call-template name="reference">
                    <xsl:with-param name="referenceNode" select="$requestBody/_X24_ref"></xsl:with-param>
                </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
                <xsl:if test="$requestBody/description">
                    <div outputclass="openapi-requestBody-description">
                        <xsl:attribute name="outputclass" select="$markdownContentOutputClass"/>
                        <xsl:value-of select="$requestBody/description"/>
                    </div>
                </xsl:if>
                <xsl:if test="$requestBody/required">
                    <p  outputclass="openapi-requestBody-required">
                        <b>Required:</b>
                        <xsl:text> </xsl:text>
                        <ph><xsl:value-of select="$requestBody/required"/></ph>
                    </p>
                </xsl:if>
                <xsl:if test="$requestBody/content">
                    <dl>
                        <xsl:call-template name="contentDlEntry">
                            <xsl:with-param name="mediaTypesNode" select="$requestBody/content"/>
                            <xsl:with-param name="outputclassVal" select="'openapi-requestBody-content'"/>
                        </xsl:call-template>
                    </dl>
                </xsl:if>
                
                <xsl:call-template name="extensionsHandlerAsList">
                    <xsl:with-param name="element" select="$requestBody"/>
                </xsl:call-template>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- Generate DITA for Responses Object. -->
    <xsl:template name="responses">
        <xsl:param name="responses"/>
        <xsl:if test="$responses and count($responses/*) > 0">
            <section outputclass="openapi-responses">
                <title>Responses</title>
                <dl outputclass="openapi-responses-list">
                    <xsl:for-each select="$responses/*">
                        <dlentry outputclass="openapi-response">
                            <dt>
                                <xsl:attribute name="outputclass" select="concat('openapi-response-key ' ,$toUnescapeOutputClass)"/>
                                <xsl:value-of select="local-name()"/>
                            </dt>
                            <dd>
                                <xsl:call-template name="responseObject">
                                    <xsl:with-param name="responseObjectNode" select="."></xsl:with-param>
                                </xsl:call-template>
                            </dd>
                        </dlentry>
                    </xsl:for-each>
                </dl>
                <xsl:call-template name="extensionsHandlerAsList">
                    <xsl:with-param name="element" select="$responses"/>
                </xsl:call-template>
            </section>
        </xsl:if>
    </xsl:template>
    
    <!-- Generate DITA for Response Object. -->
    <xsl:template name="responseObject">
        <xsl:param name="responseObjectNode"/>
        <xsl:choose>
            <xsl:when test="$responseObjectNode/_X24_ref">
                <xsl:call-template name="reference">
                    <xsl:with-param name="referenceNode" select="$responseObjectNode/_X24_ref"></xsl:with-param>
                </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
                <div>
                    <xsl:attribute name="outputclass" select="concat('openapi-response-description ', $markdownContentOutputClass)"/>
                    <xsl:value-of select="./description"/>
                </div>
                <xsl:if test="count(./*) > 1">
                    <dl outputclass="openapi-response-properties">
                        <xsl:if test="./content">
                            <xsl:call-template name="contentDlEntry">
                                <xsl:with-param name="mediaTypesNode" select="./content"/>
                                <xsl:with-param name="outputclassVal" select="'openapi-response-content'"/>
                            </xsl:call-template>
                        </xsl:if>
                        <xsl:if test="./headers">
                            <dlentry outputclass="openapi-response-headers">
                                <dt>Headers</dt>
                                <dd>
                                    <xsl:call-template name="headersMap">
                                        <xsl:with-param name="headerNodeItems" select="./headers/*"/>
                                        <xsl:with-param name="isFromComponents" select="false()"/>
                                    </xsl:call-template>
                                </dd>
                            </dlentry>
                        </xsl:if>
                        <xsl:if test="./links">
                            <dlentry outputclass="openapi-response-links">
                                <dt>Links</dt>
                                <dd>
                                    <xsl:call-template name="linksMap">
                                        <xsl:with-param name="linkItems" select="./links/*"/>
                                    </xsl:call-template>
                                </dd>
                            </dlentry>
                        </xsl:if>
                    </dl>
                </xsl:if>
                <xsl:call-template name="extensionsHandlerAsList">
                    <xsl:with-param name="element" select="$responseObjectNode"/>
                </xsl:call-template>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- Generate DITA for OpenAPI links. -->
    <xsl:template name="linksMap">
        <xsl:param name="linkItems"/>
        <xsl:if test="$linkItems and count($linkItems) > 0">
            <dl outputclass="openapi-links">
                <xsl:for-each select="$linkItems">
                    <dlentry outputclass="openapi-link">
                        <dt>
                            <xsl:attribute name="outputclass" select="concat('openapi-link-key ' ,$toUnescapeOutputClass)"/>
                            <xsl:value-of select="local-name()"/>
                        </dt>
                        <dd>
                            <xsl:call-template name="link">
                                <xsl:with-param name="linkObject" select="."></xsl:with-param>
                            </xsl:call-template>
                        </dd>
                    </dlentry>
                </xsl:for-each>
            </dl>
        </xsl:if>
    </xsl:template>
    
    <!-- Generate DITA for OpenAPI Link Objects. -->
    <xsl:template name="link">
        <xsl:param name="linkObject"/>
        <xsl:choose>
            <xsl:when test="$linkObject/_X24_ref">
                <!-- It's a reference to a link -->
                <xsl:call-template name="reference">
                    <xsl:with-param name="referenceNode" select="$linkObject/_X24_ref"></xsl:with-param>
                </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
                <xsl:if test="$linkObject/description">
                    <div>
                        <xsl:attribute name="outputclass" select="concat('openapi-link-description ' ,$markdownContentOutputClass)"/>
                        <xsl:value-of select="$linkObject/description"/>
                    </div>
                </xsl:if>
                <xsl:if test="$linkObject/operationId or $linkObject/operationRef 
                    or $linkObject/parameters or $linkObject/requestBody or $linkObject/server">
                    <dl outputclass="openapi-link-fields-list">
                        <xsl:if test="$linkObject/operationId">
                            <dlentry>
                                <dt>Operation</dt>
                                <dd>
                                    <xref outputclass="openapi-link-operationId">
                                        <xsl:attribute name="href" select="concat('#', f:correctId($linkObject/operationId))"></xsl:attribute>
                                        <xsl:value-of select="$linkObject/operationId"/>
                                    </xref>
                                </dd>
                            </dlentry>
                        </xsl:if>
                        
                        <xsl:if test="$linkObject/operationRef">
                            <dlentry>
                                <dt>Operation</dt>
                                <dd><xref outputclass="openapi-link-operationRef">
                                    <xsl:attribute name="href" select="f:correctId($linkObject/operationRef)"></xsl:attribute>
                                    <xsl:value-of select="$linkObject/operationRef"/>
                                </xref></dd>
                            </dlentry>
                        </xsl:if>
                        
                        <xsl:if test="$linkObject/parameters">
                            <dlentry>
                                <dt>Parameters</dt>
                                <dd outputclass="openapi-link-parameters">
                                    <xsl:call-template name="linkParametersMap">
                                        <xsl:with-param name="linkParamItems" select="$linkObject/parameters/*"></xsl:with-param>
                                    </xsl:call-template>
                                </dd>
                            </dlentry>
                        </xsl:if>
                        
                        <xsl:if test="$linkObject/requestBody">
                            <dlentry>
                                <dt>Request body</dt>
                                <dd>
                                    <ph outputclass="openapi-link-requestBody">
                                        <xsl:value-of select="$linkObject/requestBody"/>
                                    </ph>
                                </dd>
                            </dlentry>
                        </xsl:if>
                        
                        <xsl:if test="$linkObject/server">
                            <dlentry>
                                <dt>Server</dt>
                                <dd>
                                    <dl outputclass="openapi-link-server">
                                        <xsl:call-template name="server">
                                            <xsl:with-param name="serverObject" select="$linkObject/server"></xsl:with-param>
                                        </xsl:call-template>
                                    </dl>
                                </dd>
                            </dlentry>
                        </xsl:if>
                    </dl>
                </xsl:if>
                <xsl:call-template name="extensionsHandlerAsList">
                    <xsl:with-param name="element" select="$linkObject"/>
                </xsl:call-template>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- Generate DITA for OpenAPI parameters from Link Object. -->
    <xsl:template name="linkParametersMap">
        <xsl:param name="linkParamItems"/>
        <xsl:if test="count($linkParamItems) > 0">
            <dl outputclass="openapi-link-parameters-list">
                <xsl:for-each select="$linkParamItems">
                    <dlentry outputclass="openapi-link-parameter">
                        <dt>
                            <xsl:attribute name="outputclass" select="concat('openapi-link-parameter-key ', $toUnescapeOutputClass)"/>
                            <xsl:value-of select="local-name()"/>
                        </dt>
                        <dd outputclass="openapi-link-parameter-value">
                            <xsl:value-of select="."/>
                        </dd>
                    </dlentry>
                </xsl:for-each>
            </dl>
        </xsl:if>
    </xsl:template>
    
    <!-- Generate DITA dlEntry for OpenAPI media types. -->
    <xsl:template name="contentDlEntry">
        <xsl:param name="mediaTypesNode"/>
        <xsl:param name="outputclassVal"/>
        <dlentry>
            <xsl:attribute name="outputclass" select="$outputclassVal"/>
            <dt>Media types</dt> 
            <dd outputclass="openapi-mediaTypes-data">
                <xsl:call-template name="content">
                    <xsl:with-param name="mediaTypesNode" select="$mediaTypesNode"/>
                </xsl:call-template>
            </dd>
        </dlentry>
    </xsl:template>
    
    <!-- Generate DITA for OpenAPI media types. -->
    <xsl:template name="content">
        <xsl:param name="mediaTypesNode"/>
        <xsl:if test="count($mediaTypesNode/*) > 0">
            <dl outputclass="openapi-mediaTypes">
                <xsl:for-each select="$mediaTypesNode/*">
                    <dlentry outputclass="openapi-mediaType">
                        <dt>
                            <xsl:attribute name="outputclass" select="concat('openapi-mediaType-key ', $toUnescapeOutputClass)"/>
                            <xsl:value-of select="local-name()"/>
                        </dt>
                        <dd>
                        	<xsl:call-template name="mediaType">
                        	    <xsl:with-param name="mediaTypeObject" select="."></xsl:with-param>
                        	</xsl:call-template>
                        </dd>
                    </dlentry>
                </xsl:for-each>
            </dl>
        </xsl:if>
    </xsl:template>
    
    <!-- Generate DITA for OpenAPI MediaType Object. -->
    <xsl:template name="mediaType">
        <xsl:param name="mediaTypeObject"/>
        <xsl:if test="$mediaTypeObject/schema or $mediaTypeObject/example or $mediaTypeObject/examples or $mediaTypeObject/encoding">
            <dl outputclass="openapi-mediaType-fields-list">
                <xsl:if test="$mediaTypeObject/schema">
                    <dlentry outputclass="openapi-mediaType-schema">
                        <dt>Schema</dt>
                        <dd>
                            <xsl:choose>
                                <xsl:when test="$mediaTypeObject/schema/_X24_ref">
                                    <xsl:call-template name="reference">
                                        <xsl:with-param name="referenceNode" select="$mediaTypeObject/schema/_X24_ref"></xsl:with-param>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="schemaObject">
                                        <xsl:with-param name="schemaNode" select="$mediaTypeObject/schema"></xsl:with-param>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                        </dd>
                    </dlentry>
                </xsl:if>
                <xsl:if test="$mediaTypeObject/example">
                    <dlentry outputclass="openapi-mediaType-example">
                        <dt>Example</dt>
                        <dd>
                            <xsl:call-template name="handleExample">
                                <xsl:with-param name="example" select="$mediaTypeObject/example"/>
                            </xsl:call-template>
                        </dd>
                    </dlentry>
                </xsl:if>
                <xsl:if test="$mediaTypeObject/examples">
                    <dlentry outputclass="openapi-mediaType-examples">
                        <dt>Examples</dt>
                        <dd>
                            <xsl:call-template name="exemplesMap">
                                <xsl:with-param name="exempleItems" select="$mediaTypeObject/examples/*"></xsl:with-param>
                            </xsl:call-template>
                        </dd>
                    </dlentry>
                </xsl:if>
                
                <xsl:if test="$mediaTypeObject/encoding">
                    <dlentry outputclass="openapi-mediaType-encodings">
                        <dt>Encodings</dt>
                        <dd>
                            <xsl:call-template name="encodingsMap">
                                <xsl:with-param name="encodingItems" select="$mediaTypeObject/encoding/*"></xsl:with-param>
                            </xsl:call-template>
                        </dd>
                    </dlentry>
                </xsl:if>
            </dl>
            <xsl:call-template name="extensionsHandlerAsList">
                <xsl:with-param name="element" select="$mediaTypeObject"/>
            </xsl:call-template>
        </xsl:if>
    </xsl:template>
    
    <!-- Generate DITA for OpenAPI Encoding objects map. -->
    <xsl:template name="encodingsMap">
        <xsl:param name="encodingItems"/>
        <xsl:if test="$encodingItems and count($encodingItems) > 0">
            <dl outputclass="openapi-encodings">
                <xsl:for-each select="$encodingItems">
                    <dlentry outputclass="openapi-encoding">
                        <dt>
                            <xsl:attribute name="outputclass" select="concat('openapi-encoding-key ', $toUnescapeOutputClass)"/>
                            <xsl:value-of select="local-name()"/>
                        </dt>
                        <dd>
                            <xsl:call-template name="encodingObject">
                                <xsl:with-param name="encodingObject" select="."/>
                            </xsl:call-template>
                        </dd>
                    </dlentry>
                </xsl:for-each>
            </dl>
        </xsl:if>
    </xsl:template>
    
    <!-- Generate DITA for OpenAPI Encoding object -->
    <xsl:template name="encodingObject">
        <xsl:param name="encodingObject"/>
        <xsl:if test="$encodingObject/contentType">
            <p><b>Content-Type:</b>
                <ph outputclass="openapi-encoding-contentType"><xsl:value-of select="$encodingObject/contentType"/></ph></p>
        </xsl:if>
        
        <xsl:if test="$encodingObject/headers">
            <p outputclass="openapi-encoding-headers"><b>Headers:</b></p>
            <xsl:call-template name="headersMap">
                <xsl:with-param name="headerNodeItems" select="$encodingObject/headers/*"></xsl:with-param>
                <xsl:with-param name="isFromComponents" select="false()"/>
            </xsl:call-template>
        </xsl:if>
        
        <xsl:if test="$encodingObject/style">
            <p><b>Style:</b>
                <ph outputclass="openapi-encoding-style"><xsl:value-of select="$encodingObject/style"/></ph></p>
        </xsl:if>
        <xsl:if test="$encodingObject/explode">
            <p><b>Explode:</b>
                <ph outputclass="openapi-encoding-explode"><xsl:value-of select="$encodingObject/explode"/></ph></p>
        </xsl:if>
        <xsl:if test="$encodingObject/allowReserved">
            <p><b>Allow reserved characters:</b>
                <ph outputclass="openapi-encoding-allowReserved"><xsl:value-of select="$encodingObject/allowReserved"/></ph></p>
        </xsl:if>
        
        <xsl:call-template name="extensionsHandlerAsList">
            <xsl:with-param name="element" select="$encodingObject"/>
        </xsl:call-template>
    </xsl:template>
    
    <!-- Generate DITA for OpenAPI Example objects map. -->
    <xsl:template name="exemplesMap">
        <xsl:param name="exempleItems"/>
        <xsl:if test="$exempleItems and count($exempleItems) > 0">
        <dl outputclass="openapi-examples">
            <xsl:for-each select="$exempleItems">
                <dlentry outputclass="openapi-example">
                    <dt>
                        <xsl:attribute name="outputclass" select="concat('openapi-example-key ', $toUnescapeOutputClass)"/>
                        <xsl:value-of select="local-name()"/>
                    </dt>
                    <dd>
                        <xsl:call-template name="exempleObject">
                            <xsl:with-param name="exempleObject" select="."/>
                        </xsl:call-template>
                    </dd>
                </dlentry>
            </xsl:for-each>
        </dl>
        </xsl:if>
    </xsl:template>
    
    <!-- Generate DITA for OpenAPI Example object. -->
    <xsl:template name="exempleObject">
        <xsl:param name="exempleObject"/>
        <xsl:choose>
            <xsl:when test="$exempleObject/_X24_ref">
                <!-- It's a reference to an example -->
                <xsl:call-template name="reference">
                    <xsl:with-param name="referenceNode" select="$exempleObject/_X24_ref"></xsl:with-param>
                </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
                <xsl:if test="$exempleObject/summary">
                    <p outputclass="openapi-exemple-summary"><xsl:value-of select="$exempleObject/summary"/></p>
                </xsl:if>
                <xsl:if test="$exempleObject/description">
                    <div>
                        <xsl:attribute name="outputclass" select="concat('openapi-exemple-description ', $markdownContentOutputClass)"/>
                        <xsl:value-of select="$exempleObject/description"/>
                    </div>
                </xsl:if>
                <xsl:if test="$exempleObject/externalValue">
                    <p>
                        <xref format="html" scope="external" outputclass="openapi-exemple-externalValue">
                            <xsl:attribute name="href"><xsl:value-of select="$exempleObject/externalValue"/></xsl:attribute>
                        </xref>
                    </p>
                </xsl:if>
                <xsl:if test="$exempleObject/value">
                    <codeblock>
                        <xsl:choose>
                            <xsl:when test="count($exempleObject/value/*) > 0">
                                <xsl:attribute name="outputclass" select="concat('openapi-exemple-value language-json ', $toOriginalFormatOutputClass)"/>
                                <xsl:copy-of select="$exempleObject/value/*"></xsl:copy-of>
                            </xsl:when>
                            <xsl:otherwise>
                                <xsl:attribute name="outputclass" select="'openapi-exemple-value language-json'"/>
                                <xsl:value-of select="$exempleObject/value"/>
                            </xsl:otherwise>
                        </xsl:choose>
                    </codeblock>
                </xsl:if>
                <xsl:call-template name="extensionsHandlerAsList">
                    <xsl:with-param name="element" select="$exempleObject"/>
                </xsl:call-template>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- Generate DITA for OpenAPI Header objects map. -->
    <xsl:template name="headersMap">
        <xsl:param name="headerNodeItems"/>
        <xsl:param name="isFromComponents"/>
        <xsl:if test="count($headerNodeItems) > 0">
            <table frame="all" rowsep="1" colsep="1" outputclass="openapi-headers">
                <tgroup cols="3">
                    <colspec colname="c1" colnum="1" colwidth="1*"/>
                    <colspec colname="c2" colnum="2" colwidth="2*"/>
                    <colspec colname="c3" colnum="3" colwidth="1*"/>
                    <thead>
                        <row>
                            <entry>Name</entry>
                            <entry>Description</entry>
                            <entry>Schema</entry>
                        </row>
                    </thead>
                    
                    <tbody>
                        <xsl:for-each select="$headerNodeItems">
                            <xsl:call-template name="headerDesc">
                                <xsl:with-param name="headerObject" select="."/>
                                <xsl:with-param name="isFromComponents" select="$isFromComponents"/>
                            </xsl:call-template>
                        </xsl:for-each>
                    </tbody>
                </tgroup>
            </table>
        </xsl:if>
    </xsl:template>
    
    <!-- Generate DITA table row for an OpenAPI Header object's fields. -->
    <xsl:template name="headerDesc">
        <xsl:param name="headerObject"/>
        <xsl:param name="isFromComponents" as="xs:boolean"/>
        <row>
            <xsl:if test="$isFromComponents">
                <xsl:attribute name="id"><xsl:value-of select="f:correctId(local-name())"/></xsl:attribute>
                <xsl:attribute name="outputclass" select="concat('openapi-header ', $toUnescapeIdOutputClass)"/>
            </xsl:if>
            <entry>
                <xsl:attribute name="outputclass" select="concat('openapi-header-key ', $toUnescapeOutputClass)"/>
                <xsl:value-of select="local-name()"/>
            </entry>
            <xsl:choose>
                <xsl:when test="$headerObject/_X24_ref">
                    <!-- It's a reference to a parameter/header -->
                    <entry namest="c2" nameend="c3">
                        <xsl:call-template name="reference">
                            <xsl:with-param name="referenceNode" select="$headerObject/_X24_ref"></xsl:with-param>
                        </xsl:call-template>
                    </entry>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:variable name="description" select="$headerObject/description"/>
                    
                    <xsl:variable name="required" select="$headerObject/required"/>
                    <xsl:variable name="deprecated" select="$headerObject/deprecated"/>
                    <xsl:variable name="allowEmptyValue" select="$headerObject/allowEmptyValue"/>
                    
                    <xsl:variable name="style" select="$headerObject/style"/>
                    <xsl:variable name="explode" select="$headerObject/explode"/>
                    <xsl:variable name="allowReserved" select="$headerObject/allowReserved"/>
                    <xsl:variable name="schema" select="$headerObject/schema"/>
                    <xsl:variable name="example" select="$headerObject/example"/>
                    <xsl:variable name="examples" select="$headerObject/examples"/>
                    
                    <xsl:variable name="content" select="$headerObject/content"/>
                    
                    <entry>
                        <xsl:if test="$description">
                            <div>
                                <xsl:attribute name="outputclass" select="$markdownContentOutputClass"/>
                                <xsl:value-of select="$description"/>
                            </div>
                        </xsl:if>
                        
                        <xsl:if test="$required or $deprecated or $allowEmptyValue or  $style or $explode or $allowReserved or $example or $examples or $content">
                            <dl>
                                <xsl:attribute name="outputclass" select="'openapi-header-properties'"/>
                                <xsl:if test="$required">
                                    <dlentry>
                                        <xsl:attribute name="outputclass" select="'openapi-header-required'"/>
                                        <dt>required</dt>
                                        <dd><xsl:value-of select="$required"/></dd>
                                    </dlentry>
                                </xsl:if>
                                <xsl:if test="$deprecated">
                                    <dlentry>
                                        <xsl:attribute name="outputclass" select="'openapi-header-deprecated'"/>
                                        <dt>deprecated</dt>
                                        <dd><xsl:value-of select="$deprecated"/></dd>
                                    </dlentry>
                                </xsl:if>
                                <xsl:if test="$allowEmptyValue">
                                    <dlentry>
                                        <xsl:attribute name="outputclass" select="'openapi-header-allowEmptyValue'"/>
                                        <dt>allow empty value</dt>
                                        <dd><xsl:value-of select="$allowEmptyValue"/></dd>
                                    </dlentry>
                                </xsl:if>
                                
                                <xsl:if test="$style">
                                    <dlentry>
                                        <xsl:attribute name="outputclass" select="'openapi-header-style'"/>
                                        <dt>style</dt>
                                        <dd>
                                            <xsl:value-of select="$style"/>
                                        </dd>
                                    </dlentry>
                                </xsl:if>
                                <xsl:if test="$explode">
                                    <dlentry>
                                        <xsl:attribute name="outputclass" select="'openapi-header-explode'"/>
                                        <dt>explode</dt>
                                        <dd>
                                            <xsl:value-of select="$explode"/>
                                        </dd>
                                    </dlentry>
                                </xsl:if>
                                <xsl:if test="$allowReserved">
                                    <dlentry>
                                        <xsl:attribute name="outputclass" select="'openapi-header-allowReserved'"/>
                                        <dt>allow reserved</dt>
                                        <dd><xsl:value-of select="$allowReserved"/></dd>
                                    </dlentry>
                                </xsl:if>
                                <xsl:if test="$example">
                                    <dlentry>
                                        <xsl:attribute name="outputclass" select="'openapi-header-example'"/>
                                        <dt>example</dt>
                                        <dd> 
                                            <xsl:call-template name="handleExample">
                                                <xsl:with-param name="example" select="$example"/>
                                            </xsl:call-template>
                                        </dd>
                                    </dlentry>
                                </xsl:if>
                                <xsl:if test="$examples">
                                    <dlentry>
                                        <xsl:attribute name="outputclass" select="'openapi-header-examples'"/>
                                        <dt>examples</dt>
                                        <dd> <xsl:call-template name="exemplesMap">
                                            <xsl:with-param name="exempleItems" select="$examples/*"></xsl:with-param>
                                        </xsl:call-template></dd>
                                    </dlentry>
                                </xsl:if>
                                <xsl:if test="$content">
                                    <xsl:call-template name="contentDlEntry">
                                        <xsl:with-param name="mediaTypesNode" select="$content"/>
                                        <xsl:with-param name="outputclassVal" select="'openapi-header-content'"/>
                                    </xsl:call-template>
                                </xsl:if>
                            </dl> 
                        </xsl:if>
                    </entry>
                    <entry>
                        <xsl:attribute name="outputclass" select="'openapi-header-schema'"/>
                        <xsl:if test="$schema">
                            <xsl:choose>
                                <xsl:when test="$schema/_X24_ref">
                                    <xsl:call-template name="reference">
                                        <xsl:with-param name="referenceNode" select="$schema/_X24_ref"></xsl:with-param>
                                    </xsl:call-template>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:call-template name="schemaObject">
                                        <xsl:with-param name="schemaNode" select="$schema"></xsl:with-param>
                                    </xsl:call-template>
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:if>
                    </entry>
                </xsl:otherwise>
            </xsl:choose>
        </row>
    </xsl:template>
    
    <!-- Check if the given array has elements with content -->
    <xsl:function name="f:arrayHasElements" as="xs:boolean">
        <xsl:param name="arrayElements"/>
        <xsl:sequence select="
            (count($arrayElements[not(array)][count(./*) > 0 or string-length(.) > 0])
            + count($arrayElements[array][count(./array/*) > 0  or string-length(./array) > 0])) > 0"/>
    </xsl:function>
    
    <xsl:function name="f:getOperationId">
        <xsl:param name="operationElement"/>
        <xsl:param name="path"/>
        <xsl:param name="prefix"/>
        <xsl:variable name="temp1">
            <xsl:choose>
                <xsl:when test="$operationElement/operationId">
                    <xsl:value-of select="$operationElement/operationId"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="concat(local-name($operationElement), '_', $path)"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="temp2">
            <xsl:choose>
                <xsl:when test="$prefix and $prefix != ''">
                    <xsl:value-of select="concat($prefix, '_', $temp1)"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="$temp1"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:value-of select="f:correctId($temp2)"/>
    </xsl:function>
    
    <xsl:function name="f:correctId" as="xs:string">
        <xsl:param name="text" as="xs:string"/>
        <!-- Replace json escaped chars -->        
        <xsl:variable name="tempId" select="replace(xs:string($text), '_X([0-9A-Fa-f]+)_', '_')"/>
        
        <xsl:variable name="tempId2" select="replace($tempId, '%20', '_')"/>
        <xsl:variable name="tempId3" select="replace($tempId2, '[^\w_-]|[+]', '_')"/>
        <xsl:variable name="tempId4" select="replace($tempId3,'[_]+', '_')"/>
        <xsl:value-of select="replace($tempId4, '^[0-9.-/_]+', '')"/>
    </xsl:function>
</xsl:stylesheet>